<?php

namespace StatenWeb\BlockGenerator\Commands;

use WP_CLI;
use WP_CLI_Command;

class Block_Generator_Command extends WP_CLI_Command {

	/**
	 * Generates a new block.
	 *
	 * ## OPTIONS
	 *
	 * <block-name>
	 * : The name of the block (e.g., "My Custom Block").
	 *
	 * ## EXAMPLES
	 *
	 *     wp blockgen generate "My Custom Block"
	 *
	 * @when after_wp_load
	 */
	public function generate( $args ) {
		list( $block_name ) = $args;

		if ( empty( $block_name ) ) {
			WP_CLI::error( "Block name is required." );
		}

		$block_slug  = $this->derive_block_slug( $block_name );
		$block_class = $this->derive_block_class( $block_name );
		$block_name = $this->derive_block_name( $block_name );

		// Get theme directory and ensure the "theme" subdirectory is handled correctly
		$theme_dir = get_template_directory();

		if ( str_ends_with( $theme_dir, '/theme' ) ) {
			$theme_path = $theme_dir;
		} else {
			$theme_path = rtrim( $theme_dir, '/' ) . '/theme';
		}

		// Corrected paths
		$block_template_file = "{$theme_path}/block/{$block_slug}.php";
		$block_class_file = "{$theme_path}/Victoria/Blocks/{$block_class}.php";

		// Ensure directories exist
		$this->ensure_directory_exists( dirname( $block_template_file ) );
		$this->ensure_directory_exists( dirname( $block_class_file ) );

		// Generate files from stubs
		$this->generate_file_from_stub(
			'block-template.stub',
			$block_template_file,
			[
				'BLOCK_NAME' => $block_name,
				'BLOCK_SLUG' => $block_slug,
			]
		);

		$this->generate_file_from_stub(
			'block-definition.stub',
			$block_class_file,
			[
				'BLOCK_CLASS' => $block_class,
				'BLOCK_SLUG'  => $block_slug,
				'BLOCK_NAME'  => $block_name,
			]
		);

		// Update Blocks_Provider.php
		$this->update_block_registry( $theme_path, $block_class );

		WP_CLI::success( "Block '{$block_name}' created successfully!" );
	}

	private function derive_block_slug( $name ) {
		return strtolower( str_replace( [' ', '_'], '-', trim($name) ) );
	}

	private function derive_block_class( $name ) {
		return implode( '_', array_map( 'ucfirst', explode( ' ', str_replace( ['-', '_'], ' ', trim($name) ) ) ) );
	}

	private function derive_block_name( $name ) {
		// Replace hyphens and underscores with spaces
		$name = str_replace( ['-', '_'], ' ', $name );

		// Convert to title case (capitalize first letter of each word)
		$name = ucwords( strtolower( $name ) );

		return $name;
	}

	private function ensure_directory_exists( $path ) {
		if ( ! is_dir( $path ) ) {
			if ( ! mkdir( $path, 0755, true ) ) {
				WP_CLI::error( "Failed to create directory: {$path}" );
			}
		}
	}

	private function generate_file_from_stub( $stub_name, $target_path, $replacements ) {
		$stub_path = dirname( __DIR__, 2 ) . "/stubs/{$stub_name}";

		if ( ! file_exists( $stub_path ) ) {
			WP_CLI::error( "Stub file not found: {$stub_path}" );
		}

		$stub_content = file_get_contents( $stub_path );

		foreach ( $replacements as $key => $value ) {
			if ( $key === 'BLOCK_NAME' ) {
				// Convert block name to Title Case
				$value = ucwords( strtolower( $value ) );
			}

			if ( $key === 'BLOCK_SLUG' ) {
				// Ensure slug is lowercase and hyphen-separated
				$value = strtolower( str_replace( ' ', '-', $value ) );
			}

			$stub_content = str_replace( "{{{$key}}}", $value, $stub_content );
		}

		if ( file_put_contents( $target_path, $stub_content ) === false ) {
			WP_CLI::error( "Failed to write file: {$target_path}" );
		}

		WP_CLI::log( "File created: {$target_path}" );
	}

	private function update_block_registry( $theme_path, $block_class ) {
		// Correct path to Blocks_Provider.php
		$blocks_provider_path = "{$theme_path}/Victoria/Providers/Blocks_Provider.php";

		if ( ! file_exists( $blocks_provider_path ) ) {
			WP_CLI::error( "Blocks_Provider.php not found at: {$blocks_provider_path}" );
		}

		$content = file_get_contents( $blocks_provider_path );

		// Ensure namespace exists
		if ( ! str_contains( $content, 'namespace Victoria\Providers;' ) ) {
			WP_CLI::error( "Invalid Blocks_Provider.php structure." );
		}

		// Check if the `use` statement for the new block already exists
		$new_use_statement = "use Victoria\\Blocks\\{$block_class};";
		if ( ! str_contains( $content, $new_use_statement ) ) {
			// Find the last `use Victoria\Blocks\` statement position
			preg_match_all( '/^use Victoria\\\\Blocks\\\\.*;$/m', $content, $matches, PREG_OFFSET_CAPTURE );

			if ( ! empty( $matches[0] ) ) {
				$last_use_position = end( $matches[0] )[1] + strlen( end( $matches[0] )[0] );

				// Insert the new `use` statement after the last existing one
				$content = substr_replace( $content, "\n{$new_use_statement}", $last_use_position, 0 );
			} else {
				// If no `use Victoria\Blocks` statements exist, insert it after the namespace
				$content = preg_replace( '/namespace Victoria\\\\Providers;/', "namespace Victoria\\Providers;\n\n{$new_use_statement}", $content, 1 );
			}
		}

		// Process the protected $items array
		preg_match( '/protected\s+array\s+\$items\s+=\s+\[(.*?)\];/s', $content, $matches );
		$existing_items = array_filter( array_map( 'trim', explode( ',', trim( $matches[1] ?? '') ) ) );

		$block_entry = "{$block_class}::class";

		if ( ! in_array( $block_entry, $existing_items ) ) {
			$existing_items[] = $block_entry;
		}

		// Reformat the items array with proper indentation and original order
		$formatted_items = "\t\t" . implode( ",\n\t\t", $existing_items ) . ",";
		$content = preg_replace( '/protected\s+array\s+\$items\s+=\s+\[.*?\];/s',
			"protected array \$items = [\n{$formatted_items}\n\t];",
			$content
		);

		// Remove any duplicate use statements
		$content = preg_replace( '/^(use Victoria\\\\Blocks\\\\Hero;)(?=.*\1)/m', '', $content );

		// Remove any excessive blank lines to maintain clean formatting
		$content = preg_replace( "/\n{3,}/", "\n\n", $content );

		// Write the updated content back to the file
		if ( file_put_contents( $blocks_provider_path, $content ) === false ) {
			WP_CLI::error( "Failed to update Blocks_Provider.php" );
		}

		WP_CLI::log( "Blocks_Provider.php updated with: {$block_class}" );
	}
}
