<?php

namespace StatenWeb\VictoriaPackage\Commands;

use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use WP_CLI;
use WP_CLI_Command;

class Publish_Victoria_Command extends WP_CLI_Command {

	/**
	 * Moves the Victoria package from `vendor` to the correct theme directory.
	 *
	 * ## OPTIONS
	 *
	 * [--overwrite-all]
	 * : Overwrite all existing files without prompting.
	 *
	 * @when after_wp_load
	 */
	public function publish( $args, $assoc_args ) {
		$themes_dir = WP_CONTENT_DIR . '/themes';
		$theme_dirs = array_filter( glob( $themes_dir . '/*' ), 'is_dir' );

		if ( empty( $theme_dirs ) ) {
			WP_CLI::error( "No theme directories found in {$themes_dir}." );
			return;
		}

		$theme_dir = $theme_dirs[0]; // Take the first theme directory found
		$destination = $theme_dir . '/theme/Victoria';

		$source = dirname( WP_CONTENT_DIR, 2 ) . '/vendor/statenweb/victoria/files/Victoria';

		if ( ! is_dir( $source ) ) {
			WP_CLI::error( "Victoria package files not found in: $source" );
			return;
		}

		// Ensure 'theme/Victoria' directory exists before copying
		if ( ! is_dir( $destination ) ) {
			mkdir( $destination, 0755, true );
		}

		$overwrite_all = isset( $assoc_args['overwrite-all'] );

		WP_CLI::log( "Copying Victoria package files from: {$source} to {$destination}" );

		// Get the length of the base `files/Victoria/` directory to strip it from the path
		$baseLength = strlen( $source ) + 1;

		// Recursively copy `files/Victoria/` directory
		$iterator = new RecursiveIteratorIterator(
			new RecursiveDirectoryIterator( $source, RecursiveDirectoryIterator::SKIP_DOTS ),
			RecursiveIteratorIterator::SELF_FIRST
		);

		foreach ( $iterator as $item ) {
			$relative_path = substr( $item->getPathname(), $baseLength ); // Strip "files/Victoria/" from path
			$destPath = $destination . DIRECTORY_SEPARATOR . $relative_path;

			if ( $item->isDir() ) {
				if ( ! is_dir( $destPath ) ) {
					mkdir( $destPath, 0755, true );
				}
			} else {
				if ( file_exists( $destPath ) ) {
					if ( $overwrite_all ) {
						WP_CLI::log( "Overwriting: {$destPath}" );

						copy( $item->getPathname(), $destPath );
					} else {
						WP_CLI::log( "File exists: {$destPath} - Overwrite? (y/n)" );

						// Ask for user input
						$response = trim( fgets( STDIN ) );

						if ( strtolower( $response ) === 'y' ) {
							copy( $item->getPathname(), $destPath );

							WP_CLI::log( "Overwritten: {$destPath}" );
						} else {
							WP_CLI::log( "Skipped: {$destPath}" );
						}
					}

					WP_CLI::log( "" );
				} else {
					copy( $item->getPathname(), $destPath );
				}
			}
		}

		WP_CLI::success( "Victoria package files copied successfully." );
	}
}
