<?php

namespace Victoria\Hooks;

use Victoria\Abstracts\Hook;
use function Env\env;

class Swup extends Hook {
	public function attach_hooks(): void {
		if ( ! self::swup_enabled() ) return;

		add_action( 'wp_enqueue_scripts' , function () {
			// Core Swup
			wp_enqueue_script(
				'swup',
				'https://unpkg.com/swup@4/dist/Swup.umd.js',
				[],
				null,
				true
			);

			// Head plugin (keeps <head> in sync: title, canonical, meta, JSON-LD)
			wp_enqueue_script(
				'swup-head',
				'https://unpkg.com/@swup/head-plugin@2',
				['swup'],
				null,
				true
			);

			// Scripts plugin (re-runs marked <script> tags after navigation)
			wp_enqueue_script(
				'swup-scripts',
				'https://unpkg.com/@swup/scripts-plugin@2',
				['swup'],
				null,
				true
			);

			// Init AFTER swup-scripts (so both plugins are available)
			wp_add_inline_script('swup-scripts', <<<JS
				(function () {
					const swup = new Swup({
						containers: ['#swup'],
						plugins: [
							new SwupHeadPlugin({
								awaitAssets: true,
								// Keep permanent head tags persistent (tweak as needed)
								persistTags: 'link[rel="preconnect"],link[rel="dns-prefetch"],link[rel="preload"]'
							}),
							// Opt-in: only scripts marked with data-swup-reload-script will re-run
							new SwupScriptsPlugin({ optin: true, head: true, body: true })
						]
					});
					window.swup = swup;

					// ===== Duplicate-safe runtime injector =====
					(function () {
						const RUNTIME_ATTR = 'data-swup-runtime';
						const RELOAD_ATTR  = 'data-swup-reload-script';

						// Remove any previously injected runtime scripts before a new visit.
						swup.hooks.on('visit:start', () => {
							document.querySelectorAll('script[' + RUNTIME_ATTR + ']').forEach(function (n) { n.remove(); });
						});

						// After Swup parses the next page, clone & run any opt-in scripts from the fetched HTML,
						// even if they sit outside #swup. Skip if an equivalent already exists in the current DOM.
						swup.hooks.on('content:replace', function ({ to }) {
							const doc = to.document;
							if (!doc) return;

							var nodes = doc.querySelectorAll('script[' + RELOAD_ATTR + ']');

							nodes.forEach(function (srcNode) {
								var isExternal = !!srcNode.src;

								// Skip if an equivalent script is already present in the current DOM.
								if (isExternal) {
									var target = normalizeUrl(srcNode.src);
									var exists = Array.prototype.some.call(document.scripts, function (s) {
										return s.src && normalizeUrl(s.src) === target;
									});
									if (exists) return;
								} else {
									var code = (srcNode.textContent || '').trim();
									if (code) {
										var h = simpleHash(code);
										if (document.querySelector('script[' + RUNTIME_ATTR + '][data-hash=\"' + h + '\"]')) return;
									}
								}

								// Create a fresh <script> and copy relevant attributes.
								var el = document.createElement('script');
								['type','crossorigin','referrerpolicy','nomodule','defer','async','integrity','nonce','id'].forEach(function (a) {
									var v = srcNode.getAttribute(a);
									if (v !== null) el.setAttribute(a, v);
								});

								if (isExternal) {
									el.src = srcNode.src;
								} else {
									el.textContent = srcNode.textContent || '';
									el.dataset.hash = simpleHash(el.textContent);
								}

								el.setAttribute(RUNTIME_ATTR, '1'); // marker so we can remove on next visit
								document.body.appendChild(el);
							});

							function normalizeUrl(u) {
								try { return new URL(u, location.href).href; } catch (e) { return u; }
							}
							// Tiny stable hash for inline code (de-dupe only, not cryptographic)
							function simpleHash(str) {
								var h = 0, i = 0, len = str.length;
								while (i < len) { h = ((h << 5) - h + str.charCodeAt(i++)) | 0; }
								return String(h);
							}
						});
					})();

					// Capture-phase guard so other scripts can't force hard navigations
					document.addEventListener('click', function (e) {
						const a = e.target.closest('a[href]');
						if (!a) return;

						const url = new URL(a.href, location.href);
						if (url.origin !== location.origin) return;                      // external
						if (a.target === '_blank' || a.hasAttribute('download')) return;
						if (a.hasAttribute('data-no-swup')) return;

						const samePathQuery = (url.pathname + url.search) === (location.pathname + location.search);
						if (samePathQuery && url.hash) return;                           // same-page anchor
						if (/\\/wp-admin\\b|\\/wp-login\\.php\\b|\\/cart\\b|\\/checkout\\b|\\/my-account\\b/.test(url.pathname)) return;

						e.preventDefault();
						swup.navigate(url.pathname + url.search + url.hash);
					}, true);
				})();
			JS);
		}, PHP_INT_MAX);
	}

	public static function swup_enabled(): bool {
		$value = function_exists( '\\Env\\env' )
			? env( 'ENABLE_SWUP_PAGE_ANIMATION' )
			: getenv( 'ENABLE_SWUP_PAGE_ANIMATION' );

		$bool = filter_var( $value, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE );

		return (bool) $bool;
	}
}
