<?php

namespace Victoria\Abstracts;

use Victoria\Hooks\Swup;
use Victoria\Interfaces\Initiable;

abstract class Block implements Initiable {
	const BLOCK_SLUG = '';
	const BLOCK_NAME = '';

	protected static array $swup_reload_handles = [];
	protected static bool $swup_filter_added = false;

	abstract public function get_block_definition(): array;

	public function init(): void {
		$this->register_block();
	}

	protected function register_block(): self {
		if (
			function_exists( 'acf_register_block_type' )
			&& $block_definition = $this->get_block_definition()
		) {
			// If Swup is enabled and block defines enqueue_assets, wrap it so we can auto-mark enqueued scripts.
			if (
				Swup::swup_enabled()
				&& isset( $block_definition['enqueue_assets'] )
				&& is_callable( $block_definition['enqueue_assets'] )
			) {
				$block_definition['enqueue_assets'] = $this->wrap_enqueue_assets( $block_definition['enqueue_assets'] );
			}

			add_action( 'acf/init', fn () => acf_register_block_type( $block_definition ), PHP_INT_MAX );
		}

		return $this;
	}

	/**
	 * Decorates the original enqueue callback:
	 * - snapshots enqueued scripts before/after it runs,
	 * - collects new handles,
	 * - keeps only top-level handles (excludes deps of those handles),
	 * - adds them to the auto-reload list for Swup Scripts Plugin (opt-in).
	 */
	protected function wrap_enqueue_assets( $original ): callable {
		return static function () use ( $original ) {
			$before = self::get_enqueued_script_handles();

			// Run the original block enqueue callback.
			$original();

			$after = self::get_enqueued_script_handles();

			$new_handles = array_values( array_diff( $after, $before ) );
			if ( $new_handles ) {
				$new_handles = self::filter_top_level_handles( $new_handles );
			}

			if ( $new_handles ) {
				self::add_swup_reload_handles( $new_handles );
			}
		};
	}

	/** @return string[] Currently queued script handles (front-end queue snapshot). */
	protected static function get_enqueued_script_handles(): array {
		global $wp_scripts;
		return ( is_object( $wp_scripts ) && isset( $wp_scripts->queue ) ) ? (array) $wp_scripts->queue : [];
	}

	/**
	 * Keep only handles that are not a dependency of another handle in the same set.
	 *
	 * @param string[] $handles
	 * @return string[]
	 */
	protected static function filter_top_level_handles( $handles ): array {
		global $wp_scripts;

		if ( ! is_object( $wp_scripts ) || ! isset( $wp_scripts->registered ) ) {
			return $handles;
		}

		$in_set     = array_flip( $handles );
		$deps_in_set = [];

		foreach ( $handles as $h ) {
			if ( isset( $wp_scripts->registered[ $h ] ) ) {
				$deps = (array) $wp_scripts->registered[ $h ]->deps;
				foreach ( $deps as $dep ) {
					if ( isset( $in_set[ $dep ] ) ) {
						$deps_in_set[ $dep ] = true;
					}
				}
			}
		}

		// Top-level = handles not present as a dependency of any other new handle.
		return array_values( array_filter( $handles, fn( $h ) => ! isset( $deps_in_set[ $h ] ) ) );
	}

	/** @param string[] $handles */
	protected static function add_swup_reload_handles( $handles ): void {
		// Merge unique
		self::$swup_reload_handles = array_values( array_unique( array_merge( self::$swup_reload_handles, $handles ) ) );

		// Add the tag filter once (front-end).
		if ( ! self::$swup_filter_added ) {
			add_filter( 'script_loader_tag', [ self::class, 'filter_script_loader_tag' ], PHP_INT_MAX, 3 );
			self::$swup_filter_added = true;
		}
	}

	/**
	 * Adds data-swup-reload-script to scripts enqueued from block enqueue_assets callbacks.
	 * Swup Scripts Plugin (optin: true) will re-run those on each navigation.
	 */
	public static function filter_script_loader_tag( $tag, $handle, $src ): string {
		if ( is_admin() ) {
			return $tag; // skip editor/admin
		}

		if ( in_array( $handle, self::$swup_reload_handles, true ) ) {
			return str_replace( '<script ', '<script data-swup-reload-script ', $tag );
		}

		return $tag;
	}

	public function get_acf_unique_name(): string {
		return sprintf( 'sw-%s', static::BLOCK_SLUG );
	}
}
