<?php

namespace StatenWeb\Commands;

use StatenWeb\Generators\Api_Block_Generator;
use StatenWeb\Generators\Block_Generator;
use StatenWeb\Generators\Cpt_Generator;
use StatenWeb\Generators\Enqueue_Generator;
use StatenWeb\Generators\Hook_Generator;
use StatenWeb\Generators\Provider_Generator;
use StatenWeb\Generators\Shortcode_Generator;
use StatenWeb\Generators\Sidebar_Generator;
use StatenWeb\Traits\Api_Helper;
use WP_CLI;
use WP_CLI_Command;

/**
 * WP-CLI commands for generating assets.
 *
 * ## AVAILABLE ASSETS
 *
 *     block, cpt, enqueue, hook, provider, shortcode, sidebar
 *
 * ## EXAMPLES
 *
 *     # Generate from local stubs
 *     wp sw-generator generate block "My Custom Block"
 *     wp sw-generator generate cpt "Books"
 *     wp sw-generator generate enqueue "Front Scripts"
 *     wp sw-generator generate hook "Init Cleanup"
 *     wp sw-generator generate provider "My Provider"
 *     wp sw-generator generate shortcode "My Shortcode"
 *     wp sw-generator generate sidebar "Blog Sidebar"
 *
 *     # Generate a concrete API block (uses .env SW_GENERATOR_API_URL/TOKEN)
 *     wp sw-generator generate block "Hero Renamed"  --type=hero
 *     wp sw-generator generate block "Cards Custom" --type=cards --force
 *
 *     # Pick an API block interactively (keeps the provided name)
 *     wp sw-generator generate block "My Custom Block" --pick
 *
 *     # List available API blocks
 *     wp sw-generator blocks-list
 *     wp sw-generator blocks-list --interactive
 */
class Sw_Generator extends WP_CLI_Command {
	use Api_Helper;

	private const ASSETS = [
		'block',
		'cpt',
		'enqueue',
		'hook',
		'provider',
		'shortcode',
		'sidebar',
	];

	/**
	 * Generates a new asset.
	 *
	 * ## OPTIONS
	 *
	 * <asset>
	 * : The asset type (e.g., block, cpt, enqueue, hook, provider, shortcode, sidebar).
	 *
	 * <asset-name>
	 * : The human name of the asset (e.g., "My Custom Block").
	 *   When using `--pick`, this name is **preserved** and used for file/class naming.
	 *
	 * [--type=<slug>]
	 * : When <asset> is "block", fetch a concrete API block by slug (e.g., hero, cards).
	 *   If omitted (and --pick not used), a stub block is generated.
	 *
	 * [--pick]
	 * : Interactively pick a block slug from the API list. **Does not change <asset-name>**.
	 *
	 * [--force]
	 * : Overwrite existing files (applies when using API-backed blocks).
	 *
	 * ## EXAMPLES
	 *   wp sw-generator generate block "My Custom Block"
	 *   wp sw-generator generate block "My Custom Hero" --type=hero
	 *   wp sw-generator generate block "My Custom Cards" --type=cards --force
	 *   wp sw-generator generate block "My Custom Block" --pick
	 *   wp sw-generator generate cpt "Books"
	 *
	 * @when after_wp_load
	 */
	public function generate( $args ) {
		[ $asset, $asset_name ] = array_pad( $args, 2, null );

		if ( empty( $asset ) ) {
			WP_CLI::error( 'Asset is required.' );
		}

		if ( empty( $asset_name ) ) {
			WP_CLI::error( 'Asset name is required.' );
		}

		if ( ! in_array( $asset, self::ASSETS, true ) ) {
			WP_CLI::error( 'Asset is not available.' );
		}

		// If generating a BLOCK and a --type or --pick is provided, route to API-backed generator.
		if ( 'block' === $asset && ( isset( WP_CLI::get_runner()->assoc_args['type'] ) || isset( WP_CLI::get_runner()->assoc_args['pick'] ) ) ) {
			$type  = WP_CLI::get_runner()->assoc_args['type'] ?? null;
			$force = isset( WP_CLI::get_runner()->assoc_args['force'] );

			// If --pick, fetch list and prompt.
			if ( isset( WP_CLI::get_runner()->assoc_args['pick'] ) ) {
				$list = $this->api_get_block_list();

				if ( empty( $list ) ) {
					WP_CLI::error( 'API returned no blocks.' );
				}

				$choices = array_values( array_map( fn( $b ) => $b['slug'], $list ) );
				WP_CLI::line( 'Select a block type:' );

				foreach ( $choices as $i => $slug ) {
					WP_CLI::line( sprintf( '  [%d] %s', $i + 1, $slug ) );
				}

				$idx = (int) \cli\prompt( 'Enter number', 1 );

				if ( $idx < 1 || $idx > count( $choices ) ) {
					WP_CLI::error( 'Invalid selection.' );
				}

				$type = $choices[ $idx - 1 ];
			}

			if ( empty( $type ) ) {
				WP_CLI::error( 'Missing --type=<slug> (or use --pick).' );
			}

			// Fetch payload from API (env-based).
			$payload = $this->api_get_block_payload( $type );

			$generator = new Api_Block_Generator();

			$ok = $generator( $asset_name, [
				'type'    => $type,
				'slug'    => $this->to_kebab( $asset_name ),
				'force'   => $force,
				'payload' => $payload,
			] );

			$ok
				? WP_CLI::success( "API block '{$asset_name}' ({$type}) created successfully!" )
				: WP_CLI::error( "API block '{$asset_name}' ({$type}) was not created." );

			return;
		}

		// Otherwise: generate from stubs
		$generator = match ( $asset ) {
			'block'     => new Block_Generator(),
			'cpt'       => new Cpt_Generator(),
			'enqueue'   => new Enqueue_Generator(),
			'hook'      => new Hook_Generator(),
			'provider'  => new Provider_Generator(),
			'shortcode' => new Shortcode_Generator(),
			'sidebar'   => new Sidebar_Generator(),
			default     => null,
		};

		if ( is_null( $generator ) ) {
			WP_CLI::error( 'Asset is not supported.' );
		}

		$asset_generated = $generator( $asset_name );

		$asset_generated
			? WP_CLI::success( "Asset '{$asset_name}' created successfully!" )
			: WP_CLI::error( "Asset '{$asset_name}' was not created." );
	}

	/**
	 * List available API blocks. Optionally pick one interactively to scaffold.
	 *
	 * ## OPTIONS
	 * [--interactive]
	 * : Show a picker and generate the selected block.
	 *
	 * ## EXAMPLES
	 *   wp sw-generator blocks-list
	 *   wp sw-generator blocks-list --interactive
	 *
	 * @when after_wp_load
	 */
	public function blocks_list( $args, $assoc_args ) {
		$list = $this->api_get_block_list();

		if ( empty( $list ) ) {
			WP_CLI::warning( 'No blocks returned from API.' );
			return;
		}

		\WP_CLI\Utils\format_items(
			'table',
			array_map(
				function ( $b ) {
					return [
						'slug' => $b['slug'] ?? '',
						'name' => $b['name'] ?? '',
					];
				},
				$list
			),
			[ 'slug', 'name' ]
		);

		if ( isset( $assoc_args['interactive'] ) ) {
			$choices = array_values( array_map( fn( $b ) => $b['slug'], $list ) );

			WP_CLI::line( '' );
			WP_CLI::line( 'Select a block to generate:' );

			foreach ( $choices as $i => $slug ) {
				WP_CLI::line( sprintf( '  [%d] %s', $i + 1, $slug ) );
			}

			$idx = (int) \cli\prompt( 'Enter number', 1 );

			if ( $idx < 1 || $idx > count( $choices ) ) {
				WP_CLI::error( 'Invalid selection.' );
			}

			$slug = $choices[ $idx - 1 ];

			// Use the API-provided name for the selected slug.
			$selected = null;
			foreach ( $list as $b ) {
				if ( isset( $b['slug'] ) && $b['slug'] === $slug ) {
					$selected = $b;
					break;
				}
			}
			$name = $selected['name'] ?? $this->humanize_slug( $slug );

			WP_CLI::line( '' );
			WP_CLI::log( "Generating '{$name}' from API (slug: {$slug})..." );

			// Delegate to `generate block ... --type=<slug>` with the API name.
			WP_CLI::runcommand(
				sprintf(
					'sw-generator generate block %s --type=%s',
					escapeshellarg( $name ),
					escapeshellarg( $slug )
				),
				[ 'exit_error' => true ]
			);
		}
	}
}
