<?php

namespace StatenWeb\Generators;

use StatenWeb\Abstracts\Generator;
use StatenWeb\Traits\Api_Helper;
use WP_CLI;

class Api_Block_Generator extends Generator {
	use Api_Helper;

	/**
	 * Usage:
	 *   $gen( 'Cards', [
	 *     'type'    => 'hero',   // API slug to pull from server
	 *     'payload' => [ ... ],  // optional pre-fetched payload (skips fetch)
	 *     'force'   => true,     // optional overwrite for block files
	 *   ] );
	 *
	 * Behavior:
	 * - Derives NEW slug/class/name from $asset_name (the quoted name in CLI).
	 * - Renames destination file paths to match NEW slug/class.
	 * - Rewrites text file contents:
	 *   - class Foo → class NEWCLASS
	 *   - const BLOCK_SLUG → 'new-slug'
	 *   - const BLOCK_NAME → 'New Name'
	 *   - 'render_template' => 'block/new-slug.php'
	 * - Non-text files are copied (with path rename for class-based names).
	 *
	 * Special shared files (downloaded every API generation if present in payload):
	 * - Victoria/Abstracts/Fields_Builder.php
	 *     • overwrite if different; skip if identical
	 * - Victoria/Fields/FieldsBuilder.php
	 *     • create if missing; skip if exists (never overwrite)
	 *
	 * @param string $asset_name Human name from CLI quotes, e.g. "Cards".
	 * @param array  $options
	 * @return bool
	 */
	public function __invoke( $asset_name, $options = [] ) {
		$this->theme_path = $this->get_theme_path();
		$force            = ! empty( $options['force'] );
		$payload          = isset( $options['payload'] ) ? $options['payload'] : null;

		if ( ! $payload ) {
			$type = $options['type'] ?? null;
			if ( empty( $type ) ) {
				WP_CLI::error( 'Api_Block_Generator: missing "type" (API block slug).' );
			}
			$payload = $this->api_get_block_payload( $type ); // centralized HTTP from trait
		}

		if ( empty( $payload['files'] ) || ! is_array( $payload['files'] ) ) {
			WP_CLI::error( 'API payload missing or invalid "files" array.' );
		}

		// NEW values derived from the quoted name (authoritative).
		$new_slug  = $this->derive_asset_slug( $asset_name );
		$new_class = $this->derive_asset_class( $asset_name );
		$new_name  = $this->derive_asset_name( $asset_name );

		// Expose to parent for provider registration.
		$this->asset_slug  = $new_slug;
		$this->asset_class = $new_class;
		$this->asset_name  = $new_name;

		// ORIGINAL values as provided by payload (or inferred).
		[ $orig_slug, $orig_class, $orig_name ] = $this->infer_originals( $payload, $new_slug, $new_class, $new_name );

		$all_ok = true;

		foreach ( $payload['files'] as $file ) {
			$rel_path = isset( $file['path'] ) ? (string) $file['path'] : '';
			if ( '' === $rel_path ) {
				WP_CLI::warning( 'Skipping file with empty path: ' . json_encode( $file ) );
				$all_ok = false;
				continue;
			}

			// Determine contents (supports plain and base64)
			$contents = null;
			if ( isset( $file['contents_b64'] ) ) {
				$decoded = base64_decode( (string) $file['contents_b64'], true );
				if ( false === $decoded ) {
					WP_CLI::warning( "Invalid base64 for {$rel_path}; skipping." );
					$all_ok = false;
					continue;
				}
				$contents = $decoded;
			} elseif ( isset( $file['contents'] ) ) {
				$contents = (string) $file['contents'];
			} else {
				WP_CLI::warning( "Missing contents for {$rel_path}; skipping." );
				$all_ok = false;
				continue;
			}

			$is_shared_abstract = $this->is_shared_fields_abstract( $rel_path );
			$is_shared_concrete = $this->is_shared_fields_concrete( $rel_path );

			// For shared files: keep original path/content (no renames/replacements)
			if ( $is_shared_abstract || $is_shared_concrete ) {
				$dest = rtrim( $this->theme_path, '/' ) . '/' . ltrim( $rel_path, '/' );
				$this->ensure_directory_exists( dirname( $dest ) );
				$write_ok = $this->write_shared_file( $dest, $contents, $is_shared_abstract );
				if ( $write_ok === true ) {
					WP_CLI::log( 'Synced: ' . $this->relative_to_theme( $dest ) );
				} elseif ( $write_ok === 'skip-exists' ) {
					WP_CLI::log( 'Skipped (exists): ' . $this->relative_to_theme( $dest ) );
				} elseif ( $write_ok === 'skip-uptodate' ) {
					WP_CLI::log( 'Skipped (up-to-date): ' . $this->relative_to_theme( $dest ) );
				} else {
					WP_CLI::warning( 'Failed to write: ' . $this->relative_to_theme( $dest ) );
					$all_ok = false;
				}
				continue;
			}

			// Normal block files: allow rename + rewrite
			$rel_path_new = $this->rename_path( $rel_path, $orig_slug, $new_slug, $orig_class, $new_class );
			$dest         = rtrim( $this->theme_path, '/' ) . '/' . ltrim( $rel_path_new, '/' );
			$dest_dir     = dirname( $dest );

			// Rewrites only for text files
			if ( $this->is_text_ext( pathinfo( $rel_path, PATHINFO_EXTENSION ) ) ) {
				$contents = $this->rewrite_contents( $contents, $orig_slug, $new_slug, $orig_class, $new_class, $orig_name, $new_name );
			}

			$this->ensure_directory_exists( $dest_dir );

			// Guard: prevent writing outside theme root.
			$theme_root_real = realpath( $this->theme_path );
			$theme_base      = $theme_root_real ? rtrim( $theme_root_real, '/' ) . '/' : rtrim( $this->theme_path, '/' ) . '/';
			$dest_dir_real   = realpath( $dest_dir );
			$dest_real       = ( $dest_dir_real ? $dest_dir_real : $dest_dir ) . '/' . basename( $dest );

			if ( 0 !== strpos( $dest_real, $theme_base ) ) {
				WP_CLI::warning( "Blocked writing outside theme: {$rel_path_new}" );
				$all_ok = false;
				continue;
			}

			if ( file_exists( $dest_real ) && ! $force ) {
				WP_CLI::warning( 'File exists (skip): ' . $this->relative_to_theme( $dest_real ) . ' (use --force to overwrite)' );
				continue;
			}

			if ( false === file_put_contents( $dest_real, $contents ) ) {
				WP_CLI::warning( 'Failed to write: ' . $this->relative_to_theme( $dest_real ) );
				$all_ok = false;
			} else {
				WP_CLI::log( 'File created: ' . $this->relative_to_theme( $dest_real ) );
			}
		}

		// Register in Provider using NEW class.
		$provider_ok = $this->update_provider_registry();

		return $all_ok && $provider_ok;
	}

	// ---------------------------------------------------------------------
	// Shared Files Handling

	private function is_shared_fields_abstract( $rel_path ) {
		return (bool) preg_match( '#(?:^|/)Victoria/Abstracts/Fields_Builder\.php$#', $rel_path );
	}

	private function is_shared_fields_concrete( $rel_path ) {
		return (bool) preg_match( '#(?:^|/)Victoria/Fields/FieldsBuilder\.php$#', $rel_path );
	}

	/**
	 * Write shared fields files with specific overwrite policy.
	 *
	 * @param string $dest Absolute path.
	 * @param string $contents
	 * @param bool   $is_abstract If true: overwrite if different; else: skip if exists.
	 * @return bool|string true on wrote, 'skip-exists', 'skip-uptodate' or false on failure.
	 */
	private function write_shared_file( $dest, $contents, $is_abstract ) {
		if ( file_exists( $dest ) ) {
			$current = file_get_contents( $dest );
			if ( $is_abstract ) {
				// Overwrite if different, skip if identical.
				if ( $current === $contents ) {
					return 'skip-uptodate';
				}
				return ( false !== file_put_contents( $dest, $contents ) );
			}
			// Concrete FieldsBuilder: never overwrite.
			return 'skip-exists';
		}
		return ( false !== file_put_contents( $dest, $contents ) );
	}

	// ---------------------------------------------------------------------
	// Inference & Rewriting helpers

	/**
	 * Infer original slug/class/name from payload or file paths/contents.
	 *
	 * @return array [ $orig_slug, $orig_class, $orig_name ]
	 */
	private function infer_originals( $payload, $fallback_slug, $fallback_class, $fallback_name ) {
		$orig_slug  = isset( $payload['slug'] )  && $payload['slug']  !== '' ? (string) $payload['slug']  : null;
		$orig_class = isset( $payload['class'] ) && $payload['class'] !== '' ? (string) $payload['class'] : null;
		$orig_name  = isset( $payload['name'] )  && $payload['name']  !== '' ? (string) $payload['name']  : null;

		// Try to infer from paths if missing.
		foreach ( $payload['files'] as $f ) {
			$p = isset( $f['path'] ) ? (string) $f['path'] : '';
			if ( ! $p ) continue;

			// block/{slug}.php
			if ( preg_match( '#(?:^|/)block/([a-z0-9\-]+)\.php$#i', $p, $m ) ) {
				if ( ! $orig_slug ) {
					$orig_slug = strtolower( $m[1] );
				}
			}

			// Victoria/Blocks/{Class}.php
			if ( preg_match( '#(?:^|/)Victoria/Blocks/([A-Za-z0-9_]+)\.php$#', $p, $m ) ) {
				if ( ! $orig_class ) {
					$orig_class = $m[1];
				}
			}
		}

		// As a last resort, infer name from slug.
		if ( ! $orig_name && $orig_slug ) {
			$orig_name = $this->humanize_slug( $orig_slug );
		}

		// Fallbacks to NEW values if still unknown.
		return [
			$orig_slug  ?: $fallback_slug,
			$orig_class ?: $fallback_class,
			$orig_name  ?: $fallback_name,
		];
	}

	/**
	 * Rename destination relative path by swapping original slug/class to new ones.
	 */
	private function rename_path( $rel_path, $orig_slug, $new_slug, $orig_class, $new_class ) {
		$new = $rel_path;

		// block/{slug}.php → block/{new_slug}.php
		$new = preg_replace(
			'#(^|/)block/' . preg_quote( $orig_slug, '#' ) . '\.php$#i',
			'$1block/' . $new_slug . '.php',
			$new
		);

		// Victoria/Blocks/{Class}.ext → Victoria/Blocks/{NewClass}.ext
		$new = preg_replace(
			'#(^|/)Victoria/Blocks/' . preg_quote( $orig_class, '#' ) . '(\.[A-Za-z0-9]+)$#',
			'$1Victoria/Blocks/' . $new_class . '$2',
			$new
		);

		return $new;
	}

	/**
	 * Rewrite text contents with new class/slug/name and template path.
	 */
	private function rewrite_contents( $contents, $orig_slug, $new_slug, $orig_class, $new_class, $orig_name, $new_name ) {
		$out = $contents;

		// class Foo → class NewClass
		$out = preg_replace(
			'/(\bclass\s+)' . preg_quote( $orig_class, '/' ) . '(\b)/',
			'$1' . $new_class . '$2',
			$out
		);

		// const BLOCK_SLUG = '...';
		$out = preg_replace(
			'/const\s+BLOCK_SLUG\s*=\s*([\'"])[^\'"]*\1/i',
			"const BLOCK_SLUG = '$new_slug'",
			$out
		);

		// const BLOCK_NAME = '...';
		$out = preg_replace(
			'/const\s+BLOCK_NAME\s*=\s*([\'"])[^\'"]*\1/i',
			"const BLOCK_NAME = '$new_name'",
			$out
		);

		// 'render_template' => 'block/{slug}.php'
		$out = preg_replace(
			'/([\'"]render_template[\'"]\s*=>\s*[\'"])block\/' . preg_quote( $orig_slug, '/' ) . '\.php([\'"])/i',
			'$1block/' . $new_slug . '.php$2',
			$out
		);

		return $out;
	}

	private function is_text_ext( $ext ) {
		$ext = strtolower( (string) $ext );
		$text = [ 'php', 'json', 'js', 'css', 'scss', 'sass', 'md', 'txt', 'html', 'twig' ];
		return in_array( $ext, $text, true );
	}

	private function relative_to_theme( $abs ) {
		$base = rtrim( $this->theme_path, '/' ) . '/';
		return 0 === strpos( $abs, $base ) ? substr( $abs, strlen( $base ) ) : $abs;
	}

	// ---- Abstract contract bindings ------------------------------------

	protected function get_provider_class_name() {
		return 'Blocks_Provider';
	}

	protected function get_assets_directory_name() {
		return 'Blocks';
	}

	protected function get_asset_files() {
		return [];
	}
}
