<?php

namespace StatenWeb\Commands;

use StatenWeb\Generators\Block_Generator;
use StatenWeb\Generators\Cpt_Generator;
use StatenWeb\Generators\Enqueue_Generator;
use StatenWeb\Generators\Hook_Generator;
use StatenWeb\Generators\Provider_Generator;
use StatenWeb\Generators\Shortcode_Generator;
use StatenWeb\Generators\Sidebar_Generator;
use WP_CLI;
use WP_CLI_Command;

/**
 * WP-CLI commands for generating assets.
 *
 * ## AVAILABLE ASSETS
 *
 *     block, cpt, enqueue, hook, provider, shortcode, sidebar
 *
 * ## EXAMPLES
 *
 *     wp sw-generator generate block "My Custom Block"
 *     wp sw-generator generate cpt "My Custom Cpt"
 *     wp sw-generator generate enqueue "My Custom Enqueue"
 *     wp sw-generator generate hook "My Custom Hook"
 *     wp sw-generator generate provider "My Custom Provider"
 *     wp sw-generator generate shortcode "My Custom Shortcode"
 *     wp sw-generator generate sidebar "My Custom Sidebar"
 */
class Sw_Generator extends WP_CLI_Command {
	private const ASSETS = [
		'block',
		'cpt',
		'enqueue',
		'hook',
		'provider',
		'shortcode',
		'sidebar'
	];

	/**
	 * Generates a new asset.
	 *
	 * ## OPTIONS
	 *
	 * <asset>
	 * : The asset type (e.g., block).
	 *
	 * Available assets: block, cpt, enqueue, hook, provider, shortcode, sidebar.
	 *
	 * <asset-name>
	 * : The name of the asset (e.g., "My Custom Block").
	 *
	 * ## EXAMPLES
	 *
	 *     wp sw-generator generate block "My Custom Block"
	 *     wp sw-generator generate cpt "My Custom Cpt"
	 *     wp sw-generator generate enqueue "My Custom Enqueue"
	 *     wp sw-generator generate hook "My Custom Hook"
	 *     wp sw-generator generate provider "My Custom Provider"
	 *     wp sw-generator generate shortcode "My Custom Shortcode"
	 *     wp sw-generator generate sidebar "My Custom Sidebar"
	 *
	 * @when after_wp_load
	 */
	public function generate( $args ) {
		[ $asset, $asset_name ] = $args;

		if ( empty( $asset ) ) {
			WP_CLI::error( "Asset is required." );
		}

		if ( empty( $asset_name ) ) {
			WP_CLI::error( "Asset name is required." );
		}

		if ( ! in_array( $asset, self::ASSETS ) ) {
			WP_CLI::error( "Asset is not available." );
		}

		$generator = match ( $asset ) {
			'block' => new Block_Generator(),
			'cpt' => new Cpt_Generator(),
			'enqueue' => new Enqueue_Generator(),
			'hook' => new Hook_Generator(),
			'provider' => new Provider_Generator(),
			'shortcode' => new Shortcode_Generator(),
			'sidebar' => new Sidebar_Generator(),
			default => null
		};

		if ( is_null( $generator ) ) {
			WP_CLI::error( "Asset is not supported." );
		}

		$asset_generated = $generator( $asset_name );

		if ( $asset_generated ) {
			WP_CLI::success( "Asset '{$asset_name}' created successfully!" );
		} else {
			WP_CLI::error( "Asset '{$asset_name}' was not created." );
		}
	}
}
