<?php

namespace StatenWeb\Abstracts;

use WP_CLI;

abstract class Generator {
	protected $asset_slug;
	protected $asset_class;
	protected $asset_name;
	protected $theme_path;

	abstract protected function get_provider_class_name();

	abstract protected function get_assets_directory_name();

	abstract protected function get_asset_files();

	public function __invoke( $asset_name ) {
		$this->asset_slug = $this->derive_asset_slug( $asset_name );
		$this->asset_class = $this->derive_asset_class( $asset_name );
		$this->asset_name = $this->derive_asset_name( $asset_name );

		$this->theme_path = $this->get_theme_path();

		$asset_files = $this->get_asset_files();

		$files_created = true;

		if ( ! empty( $asset_files ) ) {
			foreach ( $asset_files as $asset_file ) {
				$this->ensure_directory_exists( dirname( $asset_file['file_path'] ) );

				$file_created = $this->generate_file_from_stub(
					$asset_file['stub_file'],
					$asset_file['file_path'],
					$asset_file['replace']
				);

				$files_created = $files_created && $file_created;
			}
		}

		$provider_registry_updated = $this->update_provider_registry();

		return $files_created && $provider_registry_updated;
	}

	protected function derive_asset_slug( $name ) {
		return strtolower( str_replace( [' ', '_'], '-', trim($name) ) );
	}

	protected function derive_asset_class( $name ) {
		return implode( '_', array_map( 'ucfirst', explode( ' ', str_replace( ['-', '_'], ' ', trim($name) ) ) ) );
	}

	protected function derive_asset_name( $name ) {
		// Replace hyphens and underscores with spaces
		$name = str_replace( ['-', '_'], ' ', $name );

		// Convert to title case (capitalize first letter of each word)
		$name = ucwords( strtolower( $name ) );

		return $name;
	}

	protected function get_theme_path() {
		$theme_dir = get_template_directory();

		if ( str_ends_with( $theme_dir, '/theme' ) ) {
			$theme_path = $theme_dir;
		} else {
			$theme_path = rtrim( $theme_dir, '/' ) . '/theme';
		}

		return $theme_path;
	}

	protected function ensure_directory_exists( $path ) {
		if ( ! is_dir( $path ) ) {
			if ( ! mkdir( $path, 0755, true ) ) {
				WP_CLI::error( "Failed to create directory: {$path}" );
			}
		}
	}

	protected function generate_file_from_stub( $stub_name, $target_path, $replacements ) {
		$stub_path = dirname( __DIR__, 2 ) . "/stubs/{$stub_name}";

		if ( ! file_exists( $stub_path ) ) {
			WP_CLI::error( "Stub file not found: {$stub_path}" );
		}

		$stub_content = file_get_contents( $stub_path );

		foreach ( $replacements as $key => $value ) {
			$stub_content = str_replace( "{{{$key}}}", $value, $stub_content );
		}

		if ( file_put_contents( $target_path, $stub_content ) === false ) {
			WP_CLI::error( "Failed to write file: {$target_path}" );

			return false;
		}

		WP_CLI::log( "File created: {$target_path}" );

		return true;
	}

	protected function update_provider_registry() {
		$theme_path = $this->get_theme_path();
		$provider_class = $this->get_provider_class_name();
		$assets_directory_name = $this->get_assets_directory_name();

		if ( is_null( $provider_class ) ) return true;

		$provider_path = "{$theme_path}/Victoria/Providers/{$provider_class}.php";

		if ( ! file_exists( $provider_path ) ) {
			WP_CLI::error( "{$provider_class}.php not found at: {$provider_path}" );
		}

		$content = file_get_contents( $provider_path );

		// Ensure namespace exists
		if ( ! str_contains( $content, 'namespace Victoria\Providers;' ) ) {
			WP_CLI::error( "Invalid {$provider_class}.php structure." );
		}

		// Check if the `use` statement for the new asset already exists
		$new_use_statement = "use Victoria\\{$assets_directory_name}\\{$this->asset_class};";

		if ( ! str_contains( $content, $new_use_statement ) ) {
			// Find the last `use Victoria\<assets>\` statement position
			preg_match_all( "/^use Victoria\\\\{$assets_directory_name}\\\\.*;$/m", $content, $matches, PREG_OFFSET_CAPTURE );

			if ( ! empty( $matches[0] ) ) {
				$last_use_position = end( $matches[0] )[1] + strlen( end( $matches[0] )[0] );

				// Insert the new `use` statement after the last existing one
				$content = substr_replace( $content, "\n{$new_use_statement}", $last_use_position, 0 );
			} else {
				// If no `use Victoria\Blocks` statements exist, insert it after the namespace
				$content = preg_replace( '/namespace Victoria\\\\Providers;/', "namespace Victoria\\Providers;\n\n{$new_use_statement}", $content, 1 );
			}
		}

		// Process the protected $items array
		preg_match( '/protected\s+array\s+\$items\s+=\s+\[(.*?)\];/s', $content, $matches );
		$existing_items = array_filter( array_map( 'trim', explode( ',', trim( $matches[1] ?? '') ) ) );

		$asset_entry = "{$this->asset_class}::class";

		if ( ! in_array( $asset_entry, $existing_items ) ) {
			$existing_items[] = $asset_entry;
		}

		// Reformat the items array with proper indentation and original order
		$formatted_items = "\t\t" . implode( ",\n\t\t", $existing_items ) . ",";
		$content = preg_replace( '/protected\s+array\s+\$items\s+=\s+\[.*?\];/s',
			"protected array \$items = [\n{$formatted_items}\n\t];",
			$content
		);

		// Remove any excessive blank lines to maintain clean formatting
		$content = preg_replace( "/\n{3,}/", "\n\n", $content );

		// Write the updated content back to the file
		if ( file_put_contents( $provider_path, $content ) === false ) {
			WP_CLI::error( "Failed to update {$provider_class}.php" );

			return false;
		}

		WP_CLI::log( "{$provider_class}.php updated with: {$this->asset_class}" );

		return true;
	}
}