<?php

namespace StatenWeb\Traits;

use WP_CLI;

trait Api_Helper {
	/**
	 * Read SW_GENERATOR_API_URL (required) and SW_GENERATOR_API_TOKEN (optional)
	 * strictly from environment (.env). No CLI flags/constants/options.
	 *
	 * @return array [ $api_url, $token ]
	 */
	private function resolve_api_creds() {
		$api_url = getenv( 'SW_GENERATOR_API_URL' );
		if ( $api_url === false || $api_url === '' ) {
			$api_url = isset( $_ENV['SW_GENERATOR_API_URL'] ) ? $_ENV['SW_GENERATOR_API_URL'] : null;
		}

		$token = getenv( 'SW_GENERATOR_API_TOKEN' );
		if ( $token === false ) {
			$token = isset( $_ENV['SW_GENERATOR_API_TOKEN'] ) ? $_ENV['SW_GENERATOR_API_TOKEN'] : null;
		}

		if ( empty( $api_url ) ) {
			WP_CLI::error( 'Missing SW_GENERATOR_API_URL in environment (.env).' );
		}

		return [ $api_url, $token ];
	}

	/**
	 * Build REST API base from a plain site URL.
	 * Ensures /wp-json/sw/v1 prefix is present.
	 *
	 * @param string $api_url
	 * @return string
	 */
	private function api_base( $api_url ) {
		return rtrim( $api_url, '/' ) . '/wp-json/sw/v1';
	}

	/**
	 * Decide SSL verification based on environment.
	 * production => true, otherwise false.
	 *
	 * Recognized env keys: WP_ENV, APP_ENV, ENVIRONMENT
	 *
	 * @return bool
	 */
	private function is_production_env() {
		$keys = [ 'WP_ENV', 'APP_ENV', 'ENVIRONMENT' ];
		foreach ( $keys as $k ) {
			$val = getenv( $k );
			if ( $val === false ) {
				$val = isset( $_ENV[ $k ] ) ? $_ENV[ $k ] : null;
			}
			if ( $val && strtolower( $val ) === 'production' ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Common HTTP args for wp_remote_* calls.
	 *
	 * @param string|null $token
	 * @return array
	 */
	private function http_args( $token ) {
		$args = [
			'headers'   => [ 'Accept' => 'application/json' ],
			'timeout'   => 60,
			'sslverify' => $this->is_production_env(),
		];

		if ( $token ) {
			$args['headers']['Authorization'] = 'Bearer ' . $token;
		}

		return $args;
	}

	/**
	 * Fetch list of blocks from API (env-based creds).
	 *
	 * @return array
	 */
	private function api_get_block_list() {
		[ $api_url, $token ] = $this->resolve_api_creds();

		$url  = $this->api_base( $api_url ) . '/blocks';
		$args = $this->http_args( $token );

		$res = wp_remote_get( $url, $args );

		if ( is_wp_error( $res ) ) {
			WP_CLI::error( 'API error: ' . $res->get_error_message() );
		}

		$code = wp_remote_retrieve_response_code( $res );

		if ( $code !== 200 ) {
			WP_CLI::error( 'API returned HTTP ' . $code . ' for ' . $url );
		}

		$body = json_decode( wp_remote_retrieve_body( $res ), true );

		return is_array( $body ) ? $body : [];
	}

	/**
	 * Fetch a specific block payload from API (env-based creds).
	 *
	 * @param string $type
	 * @return array
	 */
	private function api_get_block_payload( $type ) {
		[ $api_url, $token ] = $this->resolve_api_creds();

		$url  = $this->api_base( $api_url ) . '/blocks/' . rawurlencode( $type );
		$args = $this->http_args( $token );

		$res = wp_remote_get( $url, $args );

		if ( is_wp_error( $res ) ) {
			WP_CLI::error( 'API error: ' . $res->get_error_message() );
		}

		$code = wp_remote_retrieve_response_code( $res );

		if ( $code !== 200 ) {
			WP_CLI::error( 'API returned HTTP ' . $code . ' for ' . $url );
		}

		$body = json_decode( wp_remote_retrieve_body( $res ), true );

		if ( empty( $body['files'] ) || ! is_array( $body['files'] ) ) {
			WP_CLI::error( 'API payload missing "files".' );
		}

		return $body;
	}

	/**
	 * Convert string to kebab-case slug.
	 *
	 * @param string $value
	 * @return string
	 */
	private function to_kebab( $value ) {
		$value = preg_replace( '/[^\p{L}\p{N}]+/u', '-', $value ? $value : '' );
		return strtolower( trim( (string) $value, '-' ) ) ?: 'block';
	}

	/**
	 * Humanize slug (kebab-case to Title Case).
	 *
	 * @param string $slug
	 * @return string
	 */
	private function humanize_slug( $slug ) {
		return ucwords( str_replace( '-', ' ', $slug ) );
	}
}
