<?php

namespace StatenWeb\Abstracts;

use WP_CLI;

abstract class Generator {
	protected $asset_slug;
	protected $asset_class;
	protected $asset_name;
	protected $theme_path;

	abstract protected function get_provider_class_name();

	abstract protected function get_assets_directory_name();

	abstract protected function get_asset_files();

	public function __invoke( $asset_name ) {
		$this->asset_slug = $this->derive_asset_slug( $asset_name );
		$this->asset_class = $this->derive_asset_class( $asset_name );
		$this->asset_name = $this->derive_asset_name( $asset_name );

		$this->theme_path = $this->get_theme_path();

		$asset_files = $this->get_asset_files();

		$files_created = true;

		if ( ! empty( $asset_files ) ) {
			foreach ( $asset_files as $asset_file ) {
				$this->ensure_directory_exists( dirname( $asset_file['file_path'] ) );

				$file_created = $this->generate_file_from_stub(
					$asset_file['stub_file'],
					$asset_file['file_path'],
					$asset_file['replace']
				);

				$files_created = $files_created && $file_created;
			}
		}

		$provider_registry_updated = $this->update_provider_registry();

		return $files_created && $provider_registry_updated;
	}

	protected function derive_asset_slug( $name ) {
		return strtolower( str_replace( [' ', '_'], '-', trim($name) ) );
	}

	protected function derive_asset_class( $name ) {
		return implode( '_', array_map( 'ucfirst', explode( ' ', str_replace( ['-', '_'], ' ', trim($name) ) ) ) );
	}

	protected function derive_asset_name( $name ) {
		// Replace hyphens and underscores with spaces
		$name = str_replace( ['-', '_'], ' ', $name );

		// Convert to title case (capitalize first letter of each word)
		$name = ucwords( strtolower( $name ) );

		return $name;
	}

	protected function get_theme_path() {
		$theme_dir = get_template_directory();

		if ( str_ends_with( $theme_dir, '/theme' ) ) {
			$theme_path = $theme_dir;
		} else {
			$theme_path = rtrim( $theme_dir, '/' ) . '/theme';
		}

		return $theme_path;
	}

	protected function ensure_directory_exists( $path ) {
		if ( ! is_dir( $path ) ) {
			if ( ! mkdir( $path, 0755, true ) ) {
				WP_CLI::error( "Failed to create directory: {$path}" );
			}
		}
	}

	protected function generate_file_from_stub( $stub_name, $target_path, $replacements ) {
		$stub_path = dirname( __DIR__, 2 ) . "/stubs/{$stub_name}";

		if ( ! file_exists( $stub_path ) ) {
			WP_CLI::error( "Stub file not found: {$stub_path}" );
		}

		$stub_content = file_get_contents( $stub_path );

		foreach ( $replacements as $key => $value ) {
			$stub_content = str_replace( "{{{$key}}}", $value, $stub_content );
		}

		if ( file_put_contents( $target_path, $stub_content ) === false ) {
			WP_CLI::error( "Failed to write file: {$target_path}" );

			return false;
		}

		WP_CLI::log( "File created: {$target_path}" );

		return true;
	}

	protected function update_provider_registry() {
		$theme_path = $this->get_theme_path();
		$provider_class = $this->get_provider_class_name();
		$assets_directory_name = $this->get_assets_directory_name();

		if ( is_null( $provider_class ) ) return true;

		$provider_path = "{$theme_path}/Victoria/Providers/{$provider_class}.php";

		if ( ! file_exists( $provider_path ) ) {
			WP_CLI::error( "{$provider_class}.php not found at: {$provider_path}" );
		}

		$content = file_get_contents( $provider_path );

		// Ensure namespace exists
		if ( ! str_contains( $content, 'namespace Victoria\Providers;' ) ) {
			WP_CLI::error( "Invalid {$provider_class}.php structure." );
		}

		// Check if the `use` statement for the new asset already exists
		$new_use_statement = "use Victoria\\{$assets_directory_name}\\{$this->asset_class};";

		if ( ! str_contains( $content, $new_use_statement ) ) {
			// Find the last `use Victoria\<assets>\` statement position
			preg_match_all( "/^use Victoria\\\\{$assets_directory_name}\\\\.*;$/m", $content, $matches, PREG_OFFSET_CAPTURE );

			if ( ! empty( $matches[0] ) ) {
				$last_use_position = end( $matches[0] )[1] + strlen( end( $matches[0] )[0] );

				// Insert the new `use` statement after the last existing one
				$content = substr_replace( $content, "\n{$new_use_statement}", $last_use_position, 0 );
			} else {
				// If no `use Victoria\Blocks` statements exist, insert it after the namespace
				$content = preg_replace( '/namespace Victoria\\\\Providers;/', "namespace Victoria\\Providers;\n\n{$new_use_statement}", $content, 1 );
			}
		}

		// Process the protected $items array
		preg_match( '/protected\s+array\s+\$items\s+=\s+\[(.*?)\];/s', $content, $matches );
		$existing_items = array_filter( array_map( 'trim', explode( ',', trim( $matches[1] ?? '') ) ) );

		$asset_entry = "{$this->asset_class}::class";

		if ( ! in_array( $asset_entry, $existing_items ) ) {
			$existing_items[] = $asset_entry;
		}

		// Reformat the items array with proper indentation and original order
		$formatted_items = "\t\t" . implode( ",\n\t\t", $existing_items ) . ",";
		$content = preg_replace( '/protected\s+array\s+\$items\s+=\s+\[.*?\];/s',
			"protected array \$items = [\n{$formatted_items}\n\t];",
			$content
		);

		// Remove any excessive blank lines to maintain clean formatting
		$content = preg_replace( "/\n{3,}/", "\n\n", $content );

		// Write the updated content back to the file
		if ( file_put_contents( $provider_path, $content ) === false ) {
			WP_CLI::error( "Failed to update {$provider_class}.php" );

			return false;
		}

		WP_CLI::log( "{$provider_class}.php updated with: {$this->asset_class}" );

		return true;
	}

	/**
	 * Ensures required Utils methods exist in Victoria\Utilities\Utils.php
	 * Called before block generation to ensure required utility methods are available.
	 */
	protected function ensure_utils_gff_method_exists() {
		$theme_path = $this->get_theme_path();
		$utils_path = "{$theme_path}/Victoria/Utilities/Utils.php";

		// If Utils.php doesn't exist, skip (it's optional)
		if ( ! file_exists( $utils_path ) ) {
			return;
		}

		$content = file_get_contents( $utils_path );

		// Define methods that need to exist
		$methods = [
			'get_gff_classes_from_block' => "\n\tpublic static function get_gff_classes_from_block( \$block ) : array {\n\t\tstatic \$waypoint_config = null;\n\n\t\t// Load and cache waypoint.json\n\t\tif ( \$waypoint_config === null ) {\n\t\t\t\$waypoint_json_path = get_stylesheet_directory() . '/theme/waypoint.json';\n\t\t\tif ( file_exists( \$waypoint_json_path ) ) {\n\t\t\t\t\$waypoint_config = json_decode( file_get_contents( \$waypoint_json_path ), true );\n\t\t\t}\n\t\t\tif ( ! \$waypoint_config ) {\n\t\t\t\t\$waypoint_config = [];\n\t\t\t}\n\t\t}\n\n\t\t\$classes = [];\n\t\t// GFF attributes are at top level of \$block, not in \$block['attrs']\n\t\t\$attrs = \$block;\n\t\t\$block_options = \$waypoint_config['blockOptions'] ?? [];\n\n\t\tforeach ( \$block_options as \$key => \$config ) {\n\t\t\t\$attr_name = 'gff_' . \$key;\n\t\t\tif ( ! empty( \$attrs[ \$attr_name ] ) ) {\n\t\t\t\t\$value = \$attrs[ \$attr_name ];\n\n\t\t\t\t// Find the matching value in config to get its selector class\n\t\t\t\tforeach ( \$config['values'] ?? [] as \$option ) {\n\t\t\t\t\tif ( ( \$option['value'] ?? '' ) === \$value ) {\n\t\t\t\t\t\t\$selector = \$option['selector'] ?? '';\n\t\t\t\t\t\t// Extract class from selector (e.g., \".max-width-wider\" -> \"max-width-wider\")\n\t\t\t\t\t\tif ( ! empty( \$selector ) && preg_match( '/\\.([a-zA-Z0-9_-]+)/', \$selector, \$matches ) ) {\n\t\t\t\t\t\t\t\$classes[] = \$matches[1];\n\t\t\t\t\t\t}\n\t\t\t\t\t\tbreak;\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t}\n\t\t}\n\n\t\t// Add className if present\n\t\tif ( ! empty( \$block['className'] ) ) {\n\t\t\t\$classes[] = \$block['className'];\n\t\t}\n\t\t// Add backgroundColor class if present\n\t\tif ( ! empty( \$block['backgroundColor'] ) ) {\n\t\t\t\$classes[] = 'bg-' . esc_attr( \$block['backgroundColor'] );\n\t\t}\n\t\t// Add textColor class if present\n\t\tif ( ! empty( \$block['textColor'] ) ) {\n\t\t\t\$classes[] = 'text-' . esc_attr( \$block['textColor'] );\n\t\t}\n\n\t\treturn \$classes;\n\t}\n",
			'get_id_from_block' => "\n\tpublic static function get_id_from_block( \$block ) {\n\t\t\$id = ! empty( \$block['anchor'] )\n\t\t\t? esc_attr( \$block['anchor'] )\n\t\t\t: esc_attr( \$block['id'] );\n\t\treturn \$id;\n\t}\n",
		];

		// Check which methods need to be added/updated
		$methods_to_add = [];

		// Always overwrite get_gff_classes_from_block if it exists
		$gff_method_exists = preg_match( '/public\s+static\s+function\s+get_gff_classes_from_block\s*\(/', $content, $matches, PREG_OFFSET_CAPTURE );
		if ( $gff_method_exists ) {
			// Find the method start position
			$method_start = $matches[0][1];

			// Find the opening brace of the method
			$open_brace_pos = strpos( $content, '{', $method_start );
			if ( $open_brace_pos !== false ) {
				$brace_count = 1;
				$pos = $open_brace_pos + 1;
				$method_end = false;

				// Count braces to find the matching closing brace
				while ( $pos < strlen( $content ) && $brace_count > 0 ) {
					if ( $content[$pos] === '{' ) {
						$brace_count++;
					} elseif ( $content[$pos] === '}' ) {
						$brace_count--;
						if ( $brace_count === 0 ) {
							$method_end = $pos + 1;
							break;
						}
					}
					$pos++;
				}

				if ( $method_end !== false ) {
					// Find the start of the line containing the method (to remove leading newline/whitespace)
					$line_start = $method_start;
					while ( $line_start > 0 && $content[$line_start - 1] !== "\n" ) {
						$line_start--;
					}

					// Remove the method (including any leading whitespace/newline)
					$content = substr_replace( $content, '', $line_start, $method_end - $line_start );
				}
			}
		}
		$methods_to_add['get_gff_classes_from_block'] = $methods['get_gff_classes_from_block'];

		// Only add get_id_from_block if it doesn't exist
		if ( ! preg_match( '/public\s+static\s+function\s+get_id_from_block\s*\(/', $content ) ) {
			$methods_to_add['get_id_from_block'] = $methods['get_id_from_block'];
		}

		// If no methods need to be added, we're done
		if ( empty( $methods_to_add ) ) {
			return;
		}

		// Find the Utils class and insert methods before its closing brace
		// Look for class Utils { ... } pattern
		if ( ! preg_match( '/class\s+Utils\s*(?:extends\s+\w+)?\s*\{/', $content, $matches, PREG_OFFSET_CAPTURE ) ) {
			// No Utils class found - this shouldn't happen if the file exists, but handle gracefully
			WP_CLI::warning( "Utils.php exists but Utils class not found. Skipping method addition." );
			return;
		}

		$class_start = $matches[0][1];

		// Find the matching closing brace for the class
		$brace_count = 0;
		$insert_pos = false;

		// Find the opening brace of the class
		$open_brace_pos = strpos( $content, '{', $class_start );
		if ( $open_brace_pos !== false ) {
			$brace_count = 1;
			$pos = $open_brace_pos + 1;

			// Count braces to find the matching closing brace
			while ( $pos < strlen( $content ) && $brace_count > 0 ) {
				if ( $content[$pos] === '{' ) {
					$brace_count++;
				} elseif ( $content[$pos] === '}' ) {
					$brace_count--;
					if ( $brace_count === 0 ) {
						$insert_pos = $pos;
						break;
					}
				}
				$pos++;
			}
		}

		if ( $insert_pos === false ) {
			// Fallback: append before the last closing brace
			$last_brace_pos = strrpos( $content, "\n}" );
			if ( $last_brace_pos !== false ) {
				$insert_pos = $last_brace_pos;
			} else {
				WP_CLI::warning( "Could not find insertion point for Utils methods." );
				return;
			}
		}

		// Combine all methods to add into one code block
		$methods_code = implode( '', $methods_to_add );

		// Insert the methods before the closing brace
		// Make sure we're on a new line
		$before_brace = substr( $content, max( 0, $insert_pos - 10 ), 10 );
		if ( strpos( $before_brace, "\n" ) === false ) {
			$methods_code = "\n" . $methods_code;
		}
		$content = substr_replace( $content, $methods_code, $insert_pos, 0 );

		// Write the updated content back
		if ( file_put_contents( $utils_path, $content ) === false ) {
			WP_CLI::warning( "Failed to add/update Utils methods in Utils.php" );
			return;
		}

		$method_names = implode( ', ', array_keys( $methods_to_add ) );
		$action = $gff_method_exists ? 'Updated' : 'Added';
		WP_CLI::log( "{$action} Utils methods in Utils.php: {$method_names}" );
	}
}