<?php

namespace StatenWeb\Commands;

use ReflectionException;
use WP_CLI;
use ReflectionClass;
use WP_CLI_Command;
use ZipArchive;

class Sw_Exporter extends WP_CLI_Command {

	/**
	 * Exports a given class, trait, or interface and its dependencies to a .zip file
	 *
	 * ## OPTIONS
	 *
	 * <entity>
	 * : The class, trait, or interface name to export
	 *
	 * ## EXAMPLES
	 *
	 * wp sw-exporter export Hero
	 *
	 * @when after_wp_load
	 */
	public function export( $args ) {
		$entity_name = self::resolve_entity_name( $args[0] );

		if (
			! $entity_name
			|| ( ! class_exists( $entity_name ) && ! interface_exists( $entity_name ) && ! trait_exists( $entity_name ) )
		) {
			WP_CLI::error( "Entity {$args[0]} does not exist in any Victoria namespace." );
		}

		$files = self::get_entity_dependencies( $entity_name );

		if ( empty( $files ) ) {
			WP_CLI::error( "Could not find files for {$entity_name}." );
		}

		$zip_file = self::create_zip( $entity_name, $files );

		WP_CLI::success( "Exported {$entity_name} to " . str_replace( '\\', '/', $zip_file ) );
	}

	/**
	 * Finds all necessary files for an entity (class, trait, or interface)
	 */
	private static function get_entity_dependencies( $entity_name ) {
		$files = [];

		try {
			$reflector = new ReflectionClass( $entity_name );
		} catch ( ReflectionException $e ) {
			return [];
		}

		if ( ! self::is_victoria_namespace( $reflector->getName() ) ) {
			WP_CLI::error( "Class {$entity_name} is outside the Victoria namespace." );
		}

		$files[ $reflector->getFileName() ] = true;

		// Add parent class
		$parent = $reflector->getParentClass();

		while ( $parent && self::is_victoria_namespace( $parent->getName() ) ) {
			$files[ $parent->getFileName() ] = true;

			$parent = $parent->getParentClass();
		}

		// Add interfaces
		foreach ( $reflector->getInterfaces() as $interface ) {
			if ( self::is_victoria_namespace( $interface->getName() ) ) {
				$files[ $interface->getFileName() ] = true;
			}
		}

		// Add traits
		foreach ( $reflector->getTraits() as $trait ) {
			if ( self::is_victoria_namespace( $trait->getName() ) ) {
				$files[ $trait->getFileName() ] = true;
			}
		}

		// Add other used classes (parsed from the file)
		$used_classes = self::find_used_classes( $reflector->getFileName() );

		foreach ( $used_classes as $file ) {
			$class_name = self::get_class_name_from_file( $file );

			if ( $class_name && self::is_victoria_namespace( $class_name ) ) {
				$files[ $file ] = true;
			}
		}

		return array_keys( $files );
	}

	/**
	 * Parses a PHP file to find other used classes
	 */
	private static function find_used_classes( $file ) {
		$used_classes = [];

		if ( ! file_exists( $file ) ) return $used_classes;

		$tokens = token_get_all( file_get_contents( $file ) );

		for ( $i = 0; $i < count( $tokens ); $i++ ) {
			if (
				T_USE === $tokens[ $i ][0]
				&& T_STRING === $tokens[ $i + 2 ][0]
			) {
				$used_classes[] = trim( $tokens[ $i + 2 ][1] );
			}
		}

		return $used_classes;
	}

	/**
	 * Extracts the namespace from tokenized PHP code
	 */
	private static function get_namespace_from_tokens( $tokens, &$index ) {
		$namespace = '';
		$index += 2;

		while (
			isset( $tokens[ $index ] )
			&& ( T_STRING === $tokens[ $index ][0] || T_NS_SEPARATOR === $tokens[ $index ][0] )
		) {
			$namespace .= $tokens[ $index ][1];

			$index++;
		}

		return $namespace;
	}

	/**
	 * Creates a zip archive from the collected files
	 */
	private static function create_zip( $entity_name, $files ) {
		$export_directory = self::get_theme_directory() . '/Victoria/exports';

		if ( ! file_exists( $export_directory ) ) {
			mkdir( $export_directory, 0777, true );
		}

		$zip_file = "{$export_directory}/{$entity_name}.zip";

		$zip = new ZipArchive();

		if ( true === $zip->open( $zip_file, ZipArchive::CREATE | ZipArchive::OVERWRITE ) ) {
			foreach ( $files as $file ) {
				if ( $file && file_exists( $file ) ) {
					$zip->addFile( $file, basename( $file ) );
				}
			}

			$zip->close();
		} else {
			WP_CLI::error( "Could not create zip file." );
		}

		return $zip_file;
	}

	/**
	 * Checks if an entity belongs to the Victoria namespace
	 */
	private static function is_victoria_namespace( $entity ) {
		return 0 === strpos( $entity, 'Victoria\\' );
	}

	/**
	 * Extracts the class name from a file
	 */
	private static function get_class_name_from_file( $file ) {
		if ( ! file_exists( $file ) ) return null;

		$tokens = token_get_all( file_get_contents( $file ) );

		$namespace = '';
		$class_name = '';

		for ( $i = 0; $i < count( $tokens ); $i++ ) {
			if ( T_NAMESPACE === $tokens[ $i ][0] ) {
				$namespace = self::get_namespace_from_tokens( $tokens, $i );
			}

			if ( T_CLASS === $tokens[ $i ][0] && T_STRING === $tokens[ $i + 2 ][0]) {
				$class_name = $tokens[ $i + 2 ][1];

				break;
			}
		}

		return $namespace ? $namespace . '\\' . $class_name : $class_name;
	}

	/**
	 * Resolves the full name of a class, trait, or interface within the Victoria namespace dynamically
	 */
	private static function resolve_entity_name( $short_entity_name ) {
		$base_namespace = 'Victoria\\';

		$base_directory = self::get_theme_directory() . '/Victoria/';

		$iterator = new \RecursiveIteratorIterator(
			new \RecursiveDirectoryIterator( $base_directory, \RecursiveDirectoryIterator::SKIP_DOTS )
		);

		foreach ( $iterator as $file ) {
			if (
				'php' === $file->getExtension()
				&& $file->getBasename( '.php' ) === $short_entity_name
			) {
				$relative_path = str_replace( $base_directory, '', $file->getPath() );

				$namespace = str_replace( ['/', '\\'], '\\', trim( $relative_path, '/' ) );

				return $base_namespace . $namespace . '\\' . $short_entity_name;
			}
		}

		return null;
	}

	/**
	 * Finds the first theme directory inside wp-content/themes (Bedrock structure boilerplate)
	 */
	private static function get_theme_directory() {
		$themes_path = realpath( WP_CONTENT_DIR . '/themes' );

		$directories = array_filter( glob( $themes_path . '/*' ), 'is_dir' );

		if ( ! empty( $directories ) ) {
			return reset( $directories ) . '/theme';
		}

		WP_CLI::error( "Could not determine theme directory." );
	}
}
