<?php

namespace SW;

use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

class Blocks_API {

	/**
	 * SW_GENERATOR_API_TOKEN is set in .env, require a matching Bearer token.
	 */
	public function permission() {
		$env_token = $this->env( 'SW_GENERATOR_API_TOKEN' );
		if ( empty( $env_token ) ) {
			return new WP_Error( 'sw_forbidden', 'Unauthorized.', [ 'status' => 401 ] );
		}

		$auth = isset( $_SERVER['HTTP_AUTHORIZATION'] ) ? $_SERVER['HTTP_AUTHORIZATION'] : '';
		if ( empty( $auth ) && function_exists( 'apache_request_headers' ) ) {
			$headers = apache_request_headers();
			if ( isset( $headers['Authorization'] ) ) {
				$auth = $headers['Authorization'];
			}
		}

		if ( preg_match( '/Bearer\s+(.+)/i', $auth, $m ) ) {
			$provided = trim( $m[1] );
			if ( hash_equals( $env_token, $provided ) ) {
				return true;
			}
		}

		return new WP_Error( 'sw_forbidden', 'Unauthorized.', [ 'status' => 401 ] );
	}

	/**
	 * GET /sw/v1/blocks
	 * Scan active theme for Victoria block definition files and list blocks.
	 */
	public function list_blocks( WP_REST_Request $req ) {
		$paths = $this->theme_paths();
		if ( ! is_dir( $paths['defs'] ) ) {
			return new WP_Error( 'sw_defs_missing', 'Blocks directory not found: ' . $paths['defs'], [ 'status' => 500 ] );
		}

		$defs = glob( $paths['defs'] . '/*.php' ) ?: [];

		$out = [];

		foreach ( $defs as $def_file ) {
			$basename = basename( $def_file, '.php' ); // Class name by convention.
			$raw      = file_get_contents( $def_file );

			$consts = $this->parse_block_consts( $raw );

			$slug = $consts['slug'] ? $consts['slug'] : strtolower( $basename );
			$name = $consts['name'] ? $consts['name'] : $basename;

			$out[] = [
				'slug'  => $slug,
				'name'  => $name,
				'class' => $basename,
			];
		}

		return new WP_REST_Response( $out, 200 );
	}

	/**
	 * GET /sw/v1/blocks/{slug}
	 * Return concrete files for a given slug:
	 *  - theme/block/{slug}.php
	 *  - theme/Victoria/Blocks/{Class}.php
	 *  - additional assets referenced in get_block_definition()['enqueue_assets'] closure
	 *  - shared: Victoria/Abstracts/Fields_Builder.php
	 *  - shared: Victoria/Fields/FieldsBuilder.php
	 */
	public function get_block( WP_REST_Request $req ) {
		$slug  = (string) $req->get_param( 'slug' );
		$paths = $this->theme_paths();

		if ( ! is_dir( $paths['defs'] ) ) {
			return new WP_Error( 'sw_defs_missing', 'Blocks directory not found: ' . $paths['defs'], [ 'status' => 500 ] );
		}

		$defs = glob( $paths['defs'] . '/*.php' ) ?: [];

		$match_file  = null;
		$match_class = null;
		$match_name  = null;
		$match_raw   = null;

		foreach ( $defs as $def_file ) {
			$basename = basename( $def_file, '.php' );
			$raw      = file_get_contents( $def_file );

			$consts     = $this->parse_block_consts( $raw );
			$found_slug = $consts['slug'] ? $consts['slug'] : strtolower( $basename );

			if ( $found_slug === $slug ) {
				$match_file  = $def_file;
				$match_class = $basename;
				$match_name  = $consts['name'] ? $consts['name'] : $basename;
				$match_raw   = $raw;
				break;
			}
		}

		if ( ! $match_file ) {
			return new WP_Error( 'sw_block_not_found', "Block '{$slug}' not found in {$paths['defs']}.", [ 'status' => 404 ] );
		}

		$files = [];

		// View file: theme/block/{slug}.php
		$view_rel  = 'block/' . $slug . '.php';
		$view_full = $paths['theme'] . '/' . $view_rel;
		if ( is_file( $view_full ) ) {
			$files[] = $this->file_payload( $view_full, $view_rel );
		}

		// Definition file: theme/Victoria/Blocks/{Class}.php
		$def_rel  = 'Victoria/Blocks/' . $match_class . '.php';
		$def_full = $paths['theme'] . '/' . $def_rel;
		if ( is_file( $def_full ) ) {
			$files[] = $this->file_payload( $def_full, $def_rel );
		}

		// Parse enqueue_assets closure for additional paths + absolute URLs.
		$extra_rel_paths = $this->extract_enqueued_paths_from_definition( $match_raw );
		$external_urls   = $this->extract_external_urls_from_definition( $match_raw );

		// Add local extra assets (relative to theme).
		foreach ( $extra_rel_paths as $rel ) {
			$full = $paths['theme'] . '/' . $rel;
			if ( is_file( $full ) && ! $this->has_path( $files, $rel ) ) {
				$files[] = $this->file_payload( $full, $rel );
			}
		}

		// Fetch and embed external assets into assets/vendor/<host>/<path>.
		foreach ( $external_urls as $url ) {
			$dest_rel = $this->external_url_to_dest_rel( $url );
			if ( ! $dest_rel ) {
				continue;
			}
			$payload = $this->download_external_to_payload( $url, $dest_rel );
			if ( $payload ) {
				// Avoid duplicates if somehow same rel already added.
				if ( ! $this->has_path( $files, $dest_rel ) ) {
					$files[] = $payload;
				}
			}
		}

		// Shared supporting files.
		$shared_files = [
			'Victoria/Abstracts/Fields_Builder.php',
			'Victoria/Fields/FieldsBuilder.php',
		];
		foreach ( $shared_files as $rel ) {
			$full = $paths['theme'] . '/' . $rel;
			if ( is_file( $full ) && ! $this->has_path( $files, $rel ) ) {
				$files[] = $this->file_payload( $full, $rel );
			}
		}

		// Ensure required files exist.
		if ( ! $this->has_path( $files, $view_rel ) || ! $this->has_path( $files, $def_rel ) ) {
			return new WP_Error(
				'sw_block_incomplete',
				"Block '{$slug}' is missing required files. Expecting '{$view_rel}' and '{$def_rel}'.",
				[ 'status' => 422 ]
			);
		}

		$payload = [
			'slug'  => $slug,
			'name'  => $match_name,
			'class' => $match_class,
			'files' => $files,
		];

		return new WP_REST_Response( $payload, 200 );
	}

	// ---------------------------------------------------------------------
	// Helpers

	private function theme_paths() {
		$root = get_stylesheet_directory();

		// If the theme dir already ends with /theme, use it; otherwise append /theme.
		if ( substr( $root, -6 ) === '/theme' ) {
			$theme = $root;
		} else {
			$theme = rtrim( $root, '/' ) . '/theme';
		}

		return [
			'theme' => $theme,
			'defs'  => $theme . '/Victoria/Blocks',
			'views' => $theme . '/block',
		];
	}

	private function parse_block_consts( $php_source ) {
		$slug = null;
		$name = null;

		// const BLOCK_SLUG = 'cards';
		if ( preg_match( "/const\s+BLOCK_SLUG\s*=\s*'([^']+)'/i", $php_source, $m ) ) {
			$slug = $m[1];
		} elseif ( preg_match( '/const\s+BLOCK_SLUG\s*=\s*"([^"]+)"/i', $php_source, $m ) ) {
			$slug = $m[1];
		}

		// const BLOCK_NAME = 'Cards';
		if ( preg_match( "/const\s+BLOCK_NAME\s*=\s*'([^']+)'/i", $php_source, $m ) ) {
			$name = $m[1];
		} elseif ( preg_match( '/const\s+BLOCK_NAME\s*=\s*"([^"]+)"/i', $php_source, $m ) ) {
			$name = $m[1];
		}

		return [
			'slug' => $slug,
			'name' => $name,
		];
	}

	/**
	 * Extract relative theme paths from the enqueue_assets closure inside the class file.
	 *
	 * We scan for patterns like:
	 *   get_template_directory_uri() . '/block-js/community-teaser.js'
	 *   get_template_directory()      . '/assets/js/slick/slick.css'
	 *   get_stylesheet_directory_uri() . '/something'
	 *   trailingslashit(get_template_directory()) . 'path'
	 *
	 * Returns a list of *relative* paths under the theme directory, e.g.:
	 *   ['block-js/community-teaser.js', 'assets/js/slick/slick.css']
	 */
	private function extract_enqueued_paths_from_definition( $php_source ) {
		$rel_paths = [];

		if ( ! is_string( $php_source ) || $php_source === '' ) {
			return $rel_paths;
		}

		// Restrict to the enqueue_assets closure body (best-effort).
		$body = $php_source;
		if ( preg_match( '/[\'"]enqueue_assets[\'"]\s*=>\s*function\s*\([^)]*\)\s*\{(.*?)\}\s*,/s', $php_source, $m ) ) {
			$body = $m[1];
		}

		$patterns = [
			'/get_template_directory_uri\(\)\s*\.\s*[\'"]([^\'"]+)[\'"]/',
			'/get_template_directory\(\)\s*\.\s*[\'"]([^\'"]+)[\'"]/',
			'/get_stylesheet_directory_uri\(\)\s*\.\s*[\'"]([^\'"]+)[\'"]/',
			'/get_stylesheet_directory\(\)\s*\.\s*[\'"]([^\'"]+)[\'"]/',
			'/trailingslashit\(\s*get_(?:template|stylesheet)_directory\(\)\s*\)\s*\.\s*[\'"]([^\'"]+)[\'"]/',
		];

		foreach ( $patterns as $re ) {
			if ( preg_match_all( $re, $body, $mm ) ) {
				foreach ( $mm[1] as $p ) {
					$p = ltrim( $p, '/' );
					if ( $p !== '' ) {
						$rel_paths[] = $p;
					}
				}
			}
		}

		// Deduplicate (case-insensitive).
		$seen = [];
		$uniq = [];
		foreach ( $rel_paths as $p ) {
			$key = strtolower( $p );
			if ( isset( $seen[ $key ] ) ) {
				continue;
			}
			$seen[ $key ] = true;
			$uniq[] = $p;
		}

		return $uniq;
	}

	/**
	 * Extract hardcoded absolute URLs (http/https) inside the enqueue_assets closure.
	 * Example matches:
	 *   'https://cdn.example.com/x.js'
	 *   "http://cdn.example.com/lib.css"
	 */
	private function extract_external_urls_from_definition( $php_source ) {
		$urls = [];

		if ( ! is_string( $php_source ) || $php_source === '' ) {
			return $urls;
		}

		// Restrict to the enqueue_assets closure body (best-effort).
		$body = $php_source;
		if ( preg_match( '/[\'"]enqueue_assets[\'"]\s*=>\s*function\s*\([^)]*\)\s*\{(.*?)\}\s*,/s', $php_source, $m ) ) {
			$body = $m[1];
		}

		if ( preg_match_all( '/[\'"](https?:\/\/[^\'"]+)[\'"]/', $body, $mm ) ) {
			foreach ( $mm[1] as $url ) {
				$urls[] = $url;
			}
		}

		// Deduplicate.
		$urls = array_values( array_unique( $urls ) );

		return $urls;
	}

	/**
	 * Map an external URL to a deterministic relative destination under the theme.
	 * assets/vendor/<host>/<path>   (query string is stripped)
	 */
	private function external_url_to_dest_rel( $url ) {
		$parts = wp_parse_url( $url );
		if ( empty( $parts['host'] ) ) {
			return null;
		}
		$host = $parts['host'];
		$path = isset( $parts['path'] ) ? trim( $parts['path'], '/' ) : '';
		if ( $path === '' ) {
			$path = 'index';
		}
		// Prevent directory traversal shenanigans.
		$path = preg_replace( '/\.\.+/', '.', $path );

		return 'assets/vendor/' . $host . '/' . $path;
	}

	/**
	 * Download an external file and return a file payload (path + contents/contents_b64).
	 * Returns null on failure.
	 */
	private function download_external_to_payload( $url, $dest_rel ) {
		$args = [
			'timeout'   => 60,
			'sslverify' => ( getenv( 'WP_ENV' ) === 'production' ),
		];

		$res = wp_remote_get( $url, $args );
		if ( is_wp_error( $res ) ) {
			return null;
		}
		$code = wp_remote_retrieve_response_code( $res );
		if ( $code < 200 || $code >= 300 ) {
			return null;
		}
		$body = wp_remote_retrieve_body( $res );

		// Guess text/binary from extension.
		$ext   = strtolower( pathinfo( parse_url( $url, PHP_URL_PATH ) ?? '', PATHINFO_EXTENSION ) );
		$is_text_ext = in_array( $ext, [ 'js', 'css', 'json', 'txt', 'map' ], true );

		if ( $is_text_ext ) {
			return [
				'path'     => str_replace( '\\', '/', $dest_rel ),
				'contents' => (string) $body,
			];
		}

		return [
			'path'         => str_replace( '\\', '/', $dest_rel ),
			'contents_b64' => base64_encode( (string) $body ),
		];
	}

	private function file_payload( $full, $rel ) {
		$ext = strtolower( pathinfo( $full, PATHINFO_EXTENSION ) );
		$raw = file_get_contents( $full );

		if ( $this->is_text( $ext ) ) {
			return [
				'path'     => str_replace( '\\', '/', $rel ),
				'contents' => $raw,
			];
		}

		return [
			'path'         => str_replace( '\\', '/', $rel ),
			'contents_b64' => base64_encode( $raw ),
		];
	}

	private function has_path( $files, $needle_rel ) {
		foreach ( $files as $f ) {
			if ( isset( $f['path'] ) && $f['path'] === $needle_rel ) {
				return true;
			}
		}
		return false;
	}

	private function is_text( $ext ) {
		$text = [ 'php', 'json', 'js', 'css', 'scss', 'sass', 'md', 'txt', 'html', 'twig' ];
		return in_array( $ext, $text, true );
	}

	private function env( $key ) {
		$val = getenv( $key );
		if ( false !== $val && $val !== '' ) {
			return $val;
		}
		return isset( $_ENV[ $key ] ) ? $_ENV[ $key ] : null;
	}
}
