<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://statenweb.com
 * @since      1.0.0
 *
 * @package    Facet_Wp_Remote_Data
 * @subpackage Facet_Wp_Remote_Data/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Facet_Wp_Remote_Data
 * @subpackage Facet_Wp_Remote_Data/includes
 * @author     Marko Bader <marko@statenweb.com>
 */
class Facet_Wp_Remote_Data {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Facet_Wp_Remote_Data_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'FACET_WP_REMOTE_DATA_VERSION' ) ) {
			$this->version = FACET_WP_REMOTE_DATA_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'facet-wp-remote-data';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_public_hooks();
		$this->define_admin_hooks();
		$this->define_neighborhood_pages_hooks();
		$this->define_facetwp_hooks();
		$this->define_acf_hooks();
		$this->define_facetwp_map_addon_hooks();
		$this->define_cron_hooks();
		$this->define_rest_api_hooks();
		$this->define_shortcodes();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Facet_Wp_Remote_Data_Loader. Orchestrates the hooks of the plugin.
	 * - Facet_Wp_Remote_Data_i18n. Defines internationalization functionality.
	 * - Facet_Wp_Remote_Data_Admin. Defines all hooks for the admin area.
	 * - Facet_Wp_Remote_Data_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facet-wp-remote-data-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facet-wp-remote-data-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-facet-wp-remote-data-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-facet-wp-remote-data-public.php';

		/**
		 * The class responsible for defining helper functions.
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facet-wp-remote-data-utils.php';

		/**
		 * The class responsible for defining REST API endpoints.
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facet-wp-remote-data-rest-api.php';

		/**
		 * The class responsible for defining all actions related to FacetWP plugin.
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facet-wp-remote-data-handler.php';

		/**
		 * The class responsible for defining all shortcodes.
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facet-wp-remote-data-shortcodes.php';

		/**
		 * The class responsible for defining functionality for neighborhood pages.
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facet-wp-remote-data-neighborhood-page.php';

		/**
		 * The class responsible for handling cache functionality (transients).
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facet-wp-remote-data-cache.php';

		/**
		 * FacetWP Remote Data Functions.
		 * Provides functions for using plugin features without classes.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/sw-facet-wp-remote-data-functions.php';

		$this->loader = new Facet_Wp_Remote_Data_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Facet_Wp_Remote_Data_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Facet_Wp_Remote_Data_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all the hooks related to the FacetWP functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_facetwp_hooks() {

		$facetwp_handler = new Facet_Wp_Remote_Data_Handler();

		$this->loader->add_filter( 'facetwp_template_html', $facetwp_handler, 'facetwp_template_html', 50, 2 );
		$this->loader->add_filter( 'facetwp_facet_render_args', $facetwp_handler, 'facetwp_facet_render_args', 50, 1 );
		$this->loader->add_filter( 'facetwp_pager_args', $facetwp_handler, 'facetwp_pager_args', 50, 2 );
		$this->loader->add_action( 'facetwp_autocomplete_load', $facetwp_handler, 'forward_facetwp_ajax_load_to_main_site', 1, 1 );

		$this->loader->add_action( 'init', $facetwp_handler, 'property_page_rewrite_rule', 50, 1 );
		$this->loader->add_filter( 'query_vars', $facetwp_handler, 'property_page_rewrite_rule_query_vars', 50, 1 );
		$this->loader->add_action( 'template_redirect', $facetwp_handler, 'property_page_template', 50, 1 );

		$this->loader->add_filter( 'facetwp_facet_dropdown_show_counts', $facetwp_handler, 'facetwp_facet_dropdown_show_counts' );

	}

	/**
	 * Register all the hooks related to the ACF functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_acf_hooks() {

		$facetwp_handler = new Facet_Wp_Remote_Data_Handler();

		$this->loader->add_action( 'wp_ajax_acf/fields/post_object/query', $facetwp_handler, 'forward_ajax_acf_post_object_query_to_main_site', 1, 1 );
		$this->loader->add_filter( 'posts_pre_query', $facetwp_handler, 'posts_pre_query_for_property_cpt', 10, 2 );

	}

	/**
	 * Register all the hooks related to the FacetWP map addon functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_facetwp_map_addon_hooks() {

		$facetwp_handler = new Facet_Wp_Remote_Data_Handler();

		if ( class_exists( 'FacetWP_Facet_Map_Addon' ) ) {
			$this->loader->add_filter( 'facetwp_map_init_args', $facetwp_handler, 'facetwp_map_init_args', PHP_INT_MAX, 1 );
			$this->loader->add_filter( 'facetwp_ajax_response', $facetwp_handler, 'facetwp_map_add_locations_on_refresh_event', PHP_INT_MAX, 2 );
			$this->loader->add_filter( 'facetwp_gmaps_api_key', $facetwp_handler, 'facetwp_gmaps_api_key', PHP_INT_MAX, 2 );
			$this->loader->add_action( 'facetwp_scripts', $facetwp_handler, 'facetwp_scripts', PHP_INT_MAX );
		}
	}

	/**
	 * Register all the hooks related to the Cron functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_cron_hooks() {

		$cache_handler = new Facet_Wp_Remote_Data_Cache();

		$this->loader->add_action( 'init', $cache_handler, 'facetwp_cache_warmup_event' );
		$this->loader->add_filter( 'cron_schedules', $cache_handler, 'cron_schedules' );
		$this->loader->add_action( 'facetwp_cache_warmup_event', $cache_handler, 'warmup_cache' );
	}

	/**
	 * Register all the hooks related to the REST API functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_rest_api_hooks() {

		$rest_api = new Facet_Wp_Remote_Data_Rest_Api();

		$this->loader->add_action( 'rest_api_init', $rest_api, 'register_rest_api_routes' );
	}

	/**
	 * Register all the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Facet_Wp_Remote_Data_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_plugin_admin_menu' );
		$this->loader->add_action( 'admin_init', $plugin_admin, 'register_settings' );
		$this->loader->add_action( 'add_meta_boxes', $plugin_admin, 'add_meta_boxes' );

		$this->loader->add_action( 'save_post', $plugin_admin, 'save_post_action' );

		$this->loader->add_action( 'wp_ajax_sync_facetwp_settings', $plugin_admin, 'wp_ajax_sync_facetwp_settings_callback' );
		$this->loader->add_action( 'wp_ajax_delete_facetwp_cache', $plugin_admin, 'wp_ajax_delete_facetwp_cache_callback' );
	}

	/**
	 * Register all the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Facet_Wp_Remote_Data_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

	}

	/**
	 * Register all the hooks related to the neighborhood pages functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	public function define_neighborhood_pages_hooks() {

		$neighborhood_pages = new Facet_Wp_Remote_Data_Neighborhood_Page();

		$this->loader->add_action( 'add_meta_boxes', $neighborhood_pages, 'add_facetwp_choices_meta_boxes' );
		$this->loader->add_action( 'save_post', $neighborhood_pages, 'save_custom_post_meta' );
		$this->loader->add_action( 'template_redirect', $neighborhood_pages, 'redirect_to_facetwp_filtered_url' );

		$this->loader->add_filter( 'facetwp_facet_html', $neighborhood_pages, 'add_facetwp_hidden_class', 90, 2 );
	}

	/**
	 * Register all the shortcodes of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_shortcodes() {

		$plugin_shortcodes = new Facet_Wp_Remote_Data_Shortcodes( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_shortcode( 'property_google_map', $plugin_shortcodes, 'property_google_map' );
		$this->loader->add_shortcode( 'property_payment_calculator', $plugin_shortcodes, 'property_payment_calculator' );

	}

	/**
	 * Run the loader to execute all the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Facet_Wp_Remote_Data_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
