<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://statenweb.com
 * @since      1.0.0
 *
 * @package    Facet_Wp_Remote_Data
 * @subpackage Facet_Wp_Remote_Data/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Facet_Wp_Remote_Data
 * @subpackage Facet_Wp_Remote_Data/admin
 * @author     Marko Bader <marko@statenweb.com>
 */
class Facet_Wp_Remote_Data_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Facet_Wp_Remote_Data_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Facet_Wp_Remote_Data_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/facet-wp-remote-data-admin.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Facet_Wp_Remote_Data_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Facet_Wp_Remote_Data_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/facet-wp-remote-data-admin.js', array( 'jquery' ), $this->version, false );

		wp_localize_script( $this->plugin_name,
            'swFacetWp',
            [
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'nonce'    => wp_create_nonce( 'facetwp_sw_facetwp_admin_ajax_nonce' ),
		    ]
        );

	}

	public function add_plugin_admin_menu() {
		add_options_page(
			__( 'FacetWP Remote Site Settings' ),
			__( 'FacetWP Remote Settings' ),
			'manage_options',
			'facetwp-remote-data-settings',
			[ $this, 'display_settings_page' ]
		);
	}

	public function display_settings_page() {
		?>
        <div class="wrap">
            <h1><?php esc_html_e( 'FacetWP Remote Site Settings' ); ?></h1>
            <form method="post" action="options.php">
				<?php
                    settings_fields( 'facetwp_remote_data_settings_group' );
                    do_settings_sections( 'facetwp-remote-data-settings' );
                    submit_button();
				?>
            </form>

            <!-- Sync Button -->
            <p>
                Retrieve and sync FacetWP settings from the remote (main) site.
                <br>
                <strong>Warning:</strong> This will overwrite the current FacetWP settings on this site.
            </p>
            <button id="sync-facetwp-settings" class="button button-primary">
				<?php esc_html_e( 'Sync FacetWP Settings' ); ?>
            </button>

            <p id="sync-message" style="display: none; color: green;"></p>

            <div style="margin-top: 30px;"></div>

            <!-- Clear Cache -->
            <p>
                Clear cache (delete all transients).
                <br>
                <strong>Warning:</strong> This will clear the cached FacetWP data on this site.
            </p>
            <button id="delete-facetwp-cache" class="button button-primary">
		        <?php esc_html_e( 'Clear FacetWP Cache' ); ?>
            </button>

            <p id="delete-message" style="display: none; color: green;"></p>
        </div>
		<?php
	}

	public function register_settings() {
		register_setting( 'facetwp_remote_data_settings_group', 'facetwp_remote_data_remote_site_username' );
		register_setting( 'facetwp_remote_data_settings_group', 'facetwp_remote_data_remote_site_password' );
		register_setting( 'facetwp_remote_data_settings_group', 'facetwp_remote_data_remote_site_url' );
		register_setting( 'facetwp_remote_data_settings_group', 'facetwp_remote_data_enable_auto_sync' );
		register_setting(
            'facetwp_remote_data_settings_group',
            'facetwp_remote_data_selected_pages',
            [
                'type' => 'array',
                'sanitize_callback' => function( $input ) {
                    return array_map( 'absint', (array) $input );
                }
            ]
        );
		register_setting(
			'facetwp_remote_data_settings_group',
			'facetwp_remote_data_selected_cpts',
			[
				'type' => 'array',
				'sanitize_callback' => function( $input ) {
					return array_map( 'sanitize_text_field', (array) $input );
				}
			]
		);
		register_setting( 'facetwp_remote_data_settings_group', 'facetwp_remote_data_google_api_key' );
		register_setting(
			'facetwp_remote_data_settings_group',
			'facetwp_remote_data_skipped_local_facets',
			[
				'type' => 'array',
			]
		);

		add_settings_section(
			'facetwp_remote_data_settings_section',
			__( 'API Credentials', 'plugin-textdomain' ),
			null,
			'facetwp-remote-data-settings'
		);

		add_settings_field(
			'facetwp_remote_data_remote_site_username',
			__( 'Username' ),
			[$this, 'username_field_callback'],
			'facetwp-remote-data-settings',
			'facetwp_remote_data_settings_section'
		);

		add_settings_field(
			'facetwp_remote_data_remote_site_password',
			__( 'Password' ),
			[$this, 'password_field_callback'],
			'facetwp-remote-data-settings',
			'facetwp_remote_data_settings_section'
		);

		add_settings_field(
			'facetwp_remote_data_remote_site_url',
			__( 'Remote Site URL' ),
			[$this, 'site_url_field_callback'],
			'facetwp-remote-data-settings',
			'facetwp_remote_data_settings_section'
		);

		add_settings_field(
			'facetwp_remote_data_enable_auto_sync',
			__( 'Enable Automatic Syncing' ),
			[$this, 'auto_sync_field_callback'],
			'facetwp-remote-data-settings',
			'facetwp_remote_data_settings_section'
		);

		add_settings_field(
			'facetwp_remote_data_selected_pages',
			__( 'Select Pages' ),
			[$this, 'selected_pages_field_callback'],
			'facetwp-remote-data-settings',
			'facetwp_remote_data_settings_section'
		);

        add_settings_field(
			'facetwp_remote_data_selected_cpts',
			__( 'Select CPTs' ),
			[$this, 'selected_cpts_field_callback'],
			'facetwp-remote-data-settings',
			'facetwp_remote_data_settings_section'
		);

		add_settings_field(
			'facetwp_remote_data_google_api_key',
			__( 'Google API Key' ),
			[$this, 'google_api_key_field_callback'],
			'facetwp-remote-data-settings',
			'facetwp_remote_data_settings_section'
		);

		add_settings_field(
			'facetwp_remote_data_skipped_local_facets',
			__( 'Facets That Should Not Be Synced' ),
			[$this, 'skipped_local_facets_field_callback'],
			'facetwp-remote-data-settings',
			'facetwp_remote_data_settings_section'
		);
	}

	public function username_field_callback() {
		$username = get_option( 'facetwp_remote_data_remote_site_username', '' );

		echo '<input type="text" name="facetwp_remote_data_remote_site_username" value="' . esc_attr( $username ) . '" class="regular-text">';
	}

	public function password_field_callback() {
		$password = get_option( 'facetwp_remote_data_remote_site_password', '' );

		echo '<input type="password" name="facetwp_remote_data_remote_site_password" value="' . esc_attr( $password ) . '" class="regular-text">';
	}

	public function site_url_field_callback() {
		$site_url = get_option( 'facetwp_remote_data_remote_site_url', '' );

		echo '<input type="url" name="facetwp_remote_data_remote_site_url" value="' . esc_attr( $site_url ) . '" class="regular-text">';
	}

	public function auto_sync_field_callback() {
		$checked = get_option( 'facetwp_remote_data_enable_auto_sync', false );

		echo '<input type="checkbox" name="facetwp_remote_data_enable_auto_sync" value="1" ' . checked( 1, $checked, false ) . '>';
		echo '<p class="description">When checked, if FacetWP settings on the main site are updated, the settings on this subsite will be automatically updated.</p>';
	}

	public function selected_pages_field_callback() {
		$selected_pages = get_option( 'facetwp_remote_data_selected_pages' );

		if ( ! is_array( $selected_pages ) ) {
			$selected_pages = wp_list_pluck( get_pages(), 'ID' );
		}

		$pages = get_pages();

		if ( empty( $pages ) ) {
			echo '<p>No pages found.</p>';

			return;
		}

		echo '<ul>';

		foreach ( $pages as $page ) {
			$checked = in_array( $page->ID, (array) $selected_pages ) ? 'checked' : '';
			echo '<li>';
			echo '<label>';
			echo '<input type="checkbox" name="facetwp_remote_data_selected_pages[]" value="' . esc_attr( $page->ID ) . '" ' . $checked . '> ';
			echo esc_html( $page->post_title );
			echo '</label>';
			echo '</li>';
		}

		echo '</ul>';
		echo '<p class="description">If a page is selected, the plugin will use FacetWP data from the main site for that page.</p>';
	}

    public function selected_cpts_field_callback() {
		$selected_cpts = get_option( 'facetwp_remote_data_selected_cpts' );

		$cpts = get_post_types( ['public' => true, '_builtin' => false] );

		if ( empty( $cpts ) ) {
			echo '<p>No CPTs found.</p>';

			return;
		}

		echo '<ul>';

		foreach ( $cpts as $cpt ) {
			$checked = in_array( $cpt, (array) $selected_cpts ) ? 'checked' : '';
			echo '<li>';
			echo '<label>';
			echo '<input type="checkbox" name="facetwp_remote_data_selected_cpts[]" value="' . esc_attr( $cpt ) . '" ' . $checked . '> ';
			echo esc_html( $cpt );
			echo '</label>';
			echo '</li>';
		}

		echo '</ul>';
		echo '<p class="description">If a CPT is selected, the plugin will use FacetWP data from the main site for that CPT page.</p>';
	}

    public function google_api_key_field_callback() {
	    $google_api_key = get_option( 'facetwp_remote_data_google_api_key', '' );

	    echo '<input type="password" name="facetwp_remote_data_google_api_key" value="' . esc_attr( $google_api_key ) . '" class="regular-text">';
	    echo '<p class="description">Add your Google API key to display a Google Map with the property location on the single property page template.</p>';
    }

	public function skipped_local_facets_field_callback() {
		$facets_to_skip_syncing = get_option( 'facetwp_remote_data_skipped_local_facets' );

		$facet_settings = get_option( 'facetwp_settings' );
		$facet_settings_decoded = ! empty( $facet_settings ) ? json_decode( $facet_settings, true ) : [];

		$facets = ! empty( $facet_settings_decoded['facets'] ) ? $facet_settings_decoded['facets'] : [];

		if ( empty( $facets ) ) {
			echo '<p>No facets found. Please sync facets.</p>';

			return;
		}

		echo '<ul>';

		foreach ( $facets as $facet ) {
			$checked = in_array( $facet['name'], (array) $facets_to_skip_syncing ) ? 'checked' : '';
			echo '<li>';
			echo '<label>';
			echo '<input type="checkbox" name="facetwp_remote_data_skipped_local_facets[]" value="' . esc_attr( $facet['name'] ) . '" ' . $checked . '> ';
			echo esc_html( $facet['label'] );
			echo '</label>';
			echo '</li>';
		}

		echo '</ul>';
		echo '<p class="description">If a facet is selected, the plugin will skip syncing data from the main site for that facet, allowing you to maintain custom facets locally.</p>';
	}

    public function add_meta_boxes() {
	    add_meta_box(
		    'sw_facetwp_page_listing_template',
		    'FacetWP Listing Template',
		    function ( $post ) {
			    $value = get_post_meta( $post->ID, 'sw_facetwp_page_listing_template', true );
			    wp_nonce_field( 'sw_facetwp_page_listing_template_nonce', 'sw_facetwp_page_listing_template_nonce' );
			    ?>
                <p>
                    <label for="sw_facetwp_page_listing_template"><?php echo esc_html( 'Enter template name:' ); ?></label>
                </p>
                <input type="text" id="sw_facetwp_page_listing_template" name="sw_facetwp_page_listing_template"
                       value="<?php echo esc_attr( $value ); ?>" class="widefat">
                <p class="description"><?php echo esc_html( 'Example: Enter "ny-page-listings" for /themes/[your-theme]/theme/facet-wp-remote-data/ny-page-listings.php' ); ?></p>
                <p class="description"><?php echo esc_html( 'Leave blank to use default template.' ); ?></p>
			    <?php
		    },
		    ['post', 'page'],
		    'side'
	    );
    }

    public function save_post_action( $post_id ) {
	    if ( ! isset( $_POST['sw_facetwp_page_listing_template_nonce'] )
             || ! wp_verify_nonce( $_POST['sw_facetwp_page_listing_template_nonce'], 'sw_facetwp_page_listing_template_nonce' )
        ) {
		    return;
	    }

	    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		    return;
	    }

	    if ( isset( $_POST['sw_facetwp_page_listing_template'] ) ) {
		    update_post_meta( $post_id, 'sw_facetwp_page_listing_template', sanitize_text_field( $_POST['sw_facetwp_page_listing_template'] ) );
	    } else {
		    delete_post_meta( $post_id, 'sw_facetwp_page_listing_template' );
	    }
    }

    public function wp_ajax_sync_facetwp_settings_callback() {
        check_ajax_referer( 'facetwp_sw_facetwp_admin_ajax_nonce', 'nonce' );

	    $result = self::sync_facetwp_settings();

	    if ( is_wp_error( $result ) ) {
		    wp_send_json_error( $result->get_error_message() );
	    } else {
		    wp_send_json_success( $result );
	    }

        exit();
    }

	public static function sync_facetwp_settings() {
		$remote_site_url = get_option( 'facetwp_remote_data_remote_site_url' );

		if ( empty( $remote_site_url ) ) {
			return new WP_Error( 'no_remote_url', 'Remote site URL is not set.', ['status' => 400] );
		}

		$auth_header = Facet_Wp_Remote_Data_Utils::get_auth_header();

		if ( empty( $auth_header ) ) {
			return new WP_Error( 'no_credentials', 'Remote site credentials are not set.', ['status' => 403] );
		}

		$response = wp_remote_get(
			$remote_site_url . '/wp-json/sw-facetwp/v1/get-facetwp-settings',
			[
				'headers' => [
					'Authorization' => 'Basic ' . $auth_header,
					'Subsite-Url' => get_bloginfo( 'url' )
				],
				'timeout' => 30,
				'sslverify' => Facet_Wp_Remote_Data_Utils::should_ssl_verify()
			]
		);

		if ( is_wp_error( $response ) ) {
			error_log( "ERROR fetching remote site FacetWP settings data. " . $response->get_error_message() );

			return new WP_Error( 'http_error', $response->get_error_message(), ['status' => 500] );
		}

		$http_code = wp_remote_retrieve_response_code( $response );
		$body_json = wp_remote_retrieve_body( $response );
		$body = json_decode( $body_json, true );

		if ( $http_code !== 200 ) {
			error_log( "ERROR fetching remote site FacetWP settings data. " . print_r( $body, true ) );

			return new WP_Error( 'http_error', "Received HTTP status code {$http_code} with message '{$body['message']}'", ['status' => $http_code] );
		}

		if ( empty( $body ) || ! is_string( $body ) ) {
			error_log(
				"ERROR: FacetWP settings response is empty or not a string. Raw decoded data: " .
				print_r( $body, true )
			);

			return new WP_Error( 'invalid_data', 'Invalid response from main site.', ['status' => 500] );
		}

		$facetwp_remote_site_settings = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			error_log(
				"ERROR: Failed to decode JSON from remote site.
                JSON Error: " . json_last_error_msg() .
				"Raw response: " . print_r( $body, true )
			);

			return new WP_Error( 'invalid_data', 'Invalid JSON response from main site.', ['status' => 500] );
		}

		if ( empty( $facetwp_remote_site_settings ) || ! is_array( $facetwp_remote_site_settings ) ) {
			error_log(
				"ERROR: FacetWP settings response is empty or not an array. Raw decoded data: " .
				print_r( $facetwp_remote_site_settings, true )
			);

			return new WP_Error( 'invalid_data', 'Invalid response from main site.', ['status' => 500] );
		}

		$settings_updated = self::update_facetwp_settings( $body );

		return is_wp_error( $settings_updated )
			? [ 'success' => false, 'message' => $settings_updated->get_error_message() ]
			: [ 'success' => true, 'message' => 'FacetWP settings synced successfully.' ];
	}

	private static function update_facetwp_settings( $settings ) {
		if ( empty( $settings ) ) {
			return new WP_Error( 'empty_data', 'No settings from remote site.', ['status' => 204] );
		}

		$option_name = 'facetwp_settings';
		$skip_option_name = 'facetwp_remote_data_skipped_local_facets';

		$current_settings = json_decode( get_option( $option_name, '{}' ), true ) ?: [];
		$current_settings_facets = ! empty( $current_settings ) ? $current_settings['facets'] : [];

		$facets_new_settings = json_decode( $settings, true ) ?: [];

		if ( empty( $facets_new_settings['facets'] ) ) {
			return new WP_Error( 'empty_data', 'No facet settings from remote site.', ['status' => 204] );
		}

		$skipped_facets = get_option( $skip_option_name, [] );
		$skipped_facets = is_array( $skipped_facets ) ? $skipped_facets : [];

		if (
			empty( $skipped_facets )
			|| empty( $current_settings_facets )
		) {
			return update_option( $option_name, $settings );
		}

		array_walk(
			$skipped_facets,
			function ( $skipped_facet ) use ( &$facets_new_settings, $current_settings_facets ) {
				$skipped_facet_settings_from_name = array_filter(
					$current_settings_facets,
					fn ( $current_setting_facet ) => $current_setting_facet['name'] === $skipped_facet
				);

				if ( ! empty( $skipped_facet_settings_from_name ) ) {
					$skipped_facet_settings_from_name = array_shift( $skipped_facet_settings_from_name );

					foreach ( $facets_new_settings['facets'] as $key => $facets_new_setting ) {
						if ( $facets_new_setting['name'] === $skipped_facet ) {

							// Skip syncing facet (use current setting)
							$facets_new_settings['facets'][ $key ] = [ ...$skipped_facet_settings_from_name ];

							return;
						}
					}

					// If not found in new settings, we'll add it
					$facets_new_settings['facets'][] = $skipped_facet_settings_from_name;
				}
			}
		);

		return update_option( $option_name, json_encode( $facets_new_settings ) );
	}

	public function wp_ajax_delete_facetwp_cache_callback() {
		check_ajax_referer( 'facetwp_sw_facetwp_admin_ajax_nonce', 'nonce' );

		$result = Facet_Wp_Remote_Data_Cache::delete_cache();

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		} else {
			wp_send_json_success( [ 'message' => "FacetWP cache cleared!", 'deleted' => true ] );
		}

		exit();
	}

}
