<?php

use Victoria\Utilities\RE_Helpers;

class Facetwp_Remote_Data_Main_Site_Rest_Api {

	public function register_rest_api_routes() {
		register_rest_route(
			'sw-facetwp/v1',
			'/get-facetwp-settings',
			[
				'methods'  => 'GET',
				'callback' => [$this, 'facetwp_get_settings'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/get-all-locations',
			[
				'methods'  => 'POST',
				'callback' => [$this, 'facetwp_get_all_locations'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/forward-facetwp-autocomplete-ajax-call',
			[
				'methods'  => 'POST',
				'callback' => [$this, 'facetwp_subsite_forwarded_autocomplete_ajax_call'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/forward-acf-post-object-query-ajax-call',
			[
				'methods'  => 'POST',
				'callback' => [$this, 'facetwp_subsite_forwarded_acf_post_object_query_ajax_call'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/properties',
			[
				'methods'  => 'GET',
				'callback' => [$this, 'get_properties_data'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/property/(?P<post_id>\d+)',
			[
				'methods'  => 'GET',
				'callback' => [$this, 'get_property_page_data'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/latest-listings',
			[
				'methods'  => 'GET',
				'callback' => [$this, 'get_latest_listings_data'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/current-listings',
			[
				'methods'  => 'GET',
				'callback' => [$this, 'get_current_listings_data'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);
	}

	public function facetwp_get_settings( WP_REST_Request $request ) {
		global $wpdb;

		$facetwp_settings = get_option( 'facetwp_settings' );

		if ( empty( $facetwp_settings ) ) {
			return new WP_Error( 'no_settings', 'FacetWP settings not found', ['status' => 404] );
		}

		$subsite_url = $request->get_header( 'Subsite-Url' );

		if ( ! empty( $subsite_url ) ) {
			$sanitized_origin = esc_url_raw( $subsite_url );

			$subsites = get_option( 'facetwp_remote_data_subsites_urls', [] );

			if ( ! is_array( $subsites ) ) {
				$subsites = [];
			}

			if ( ! in_array( $sanitized_origin, $subsites, true ) ) {
				$subsites[] = $sanitized_origin;

				update_option( 'facetwp_remote_data_subsites_urls', $subsites );
			}
		}

		$batch      = (int) $request->get_param( 'batch' ) ?: 1;
		$batch_size = 5000;
		$offset     = ( $batch - 1 ) * $batch_size;

		$index_rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}facetwp_index LIMIT %d OFFSET %d",
				$batch_size,
				$offset
			),
			ARRAY_A
		);

		$total_rows = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}facetwp_index" );
		$total_batches = ceil( $total_rows / $batch_size );

		$response_data = [
			'facetwp_settings' => $facetwp_settings,
			'facetwp_index'    => $index_rows,
			'batch'            => $batch,
			'total_batches'    => $total_batches,
		];

		return rest_ensure_response( $response_data );
	}

	public function facetwp_get_all_locations( WP_REST_Request $request ) {
		$raw_body = $request->get_body();

		$decoded_body = json_decode( $raw_body, true );

		$results = ( new Facetwp_Remote_Data_Main_Site_Facetwp_Handler )->get_all_posts_for_map_facet( $decoded_body['post_ids'] );

		return rest_ensure_response( $results );
	}

	public function facetwp_subsite_forwarded_autocomplete_ajax_call( WP_REST_Request $request ) {
		$raw_body = $request->get_body();

		$decoded_body = json_decode( $raw_body, true );

		$_POST = is_array( $decoded_body ) ? $decoded_body : [];

		if ( class_exists( 'FacetWP_Facet_Autocomplete' ) ) {
			( new \FacetWP_Facet_Autocomplete() )->ajax_load();

			exit();
		}
	}

	public function facetwp_subsite_forwarded_acf_post_object_query_ajax_call( WP_REST_Request $request ) {
		$results = [];

		$raw_body = $request->get_body();

		$decoded_body = json_decode( $raw_body, true );

		if ( ! empty( $decoded_body ) ) {
			$args = wp_parse_args(
				$decoded_body,
				array(
					'posts_per_page' => 20,
					'paged' => 0,
					'post_type' => 'property',
					'orderby' => 'menu_order title',
					'order' => 'ASC',
					'post_status' => 'publish',
					'suppress_filters' => false,
					'update_post_meta_cache' => false,
					'fields' => 'ids',
				)
			);

			$post_ids = get_posts( $args );

			if ( ! empty( $post_ids ) ) {
				$results = array_map(
					function ( $post_id ) {
						return [
							'id' => (int) $post_id,
							'text' => get_the_title( $post_id ),
						];
					},
					$post_ids
				);
			}
		}

		return rest_ensure_response( $results );
	}

	public function get_properties_data( WP_REST_Request $request ) {
		$post_ids = $request->get_param( 'post_ids' );

		if ( empty( $post_ids ) ) {
			return new WP_Error( 'missing_post_id', 'Post IDs are required.', ['status' => 400] );
		}

		if ( ! class_exists( RE_Helpers::class ) ) {
			return new WP_Error( 'missing_helper_class', 'RE_Helpers class does not exist.', ['status' => 400] );
		}

		$post_ids = explode( ',', $post_ids );

		$transient_key = Facet_Wp_Remote_Data_Cache::properties_listing_site_data( $post_ids );

		$skip_cache = apply_filters( 'facetwp_remote_data_skip_cache', false );

		$properties_data = ! $skip_cache
			? get_transient( $transient_key )
			: false;

		if ( false === $properties_data ) {
			$properties_data = RE_Helpers::get_properties_listing( $post_ids );

			Facet_Wp_Remote_Data_Cache::set( $transient_key, $properties_data );
		}

		return rest_ensure_response( $properties_data );
	}

	public function get_property_page_data( WP_REST_Request $request ) {
		$post_id = $request->get_param( 'post_id' );

		if ( empty( $post_id ) ) {
			return new WP_Error( 'missing_post_id', 'Post ID is required.', ['status' => 400] );
		}

		if ( ! is_numeric( $post_id ) || intval( $post_id ) != $post_id ) {
			return new WP_Error( 'invalid_post_id', 'Post ID must be an integer.', ['status' => 400] );
		}

		$post_id = (int) $post_id;

		if ( ! get_post( $post_id ) ) {
			return new WP_Error( 'post_not_found', 'Post does not exist.', ['status' => 404] );
		}

		if ( get_post_type( $post_id ) !== 'property' ) {
			return new WP_Error( 'invalid_post_type', 'Post is not of type "property".', ['status' => 400] );
		}

		if ( ! class_exists( RE_Helpers::class ) ) {
			return new WP_Error( 'missing_helper_class', 'RE_Helpers class does not exist.', ['status' => 400] );
		}

		$transient_key = Facet_Wp_Remote_Data_Cache::property_site_data( $post_id );

		$skip_cache = apply_filters( 'facetwp_remote_data_skip_cache', false );

		$property_data = ! $skip_cache
			? get_transient( $transient_key )
			: false;

		if ( false === $property_data ) {
			$property_data = [
				'property_id' => $post_id,
				'property_template_tags' => RE_Helpers::get_property_template_tags( $post_id, 'api' ),
				'property_calculator_data' => RE_Helpers::get_calculator_data( $post_id ),
			];

			Facet_Wp_Remote_Data_Cache::set( $transient_key, $property_data, HOUR_IN_SECONDS );
		}

		return rest_ensure_response( $property_data );
	}

	public function get_listings_data( $posts_query_args ) {
		if ( ! class_exists( RE_Helpers::class ) ) {
			return new WP_Error( 'missing_helper_class', 'RE_Helpers class does not exist.', ['status' => 400] );
		}

		$query = new WP_Query( $posts_query_args );

		$post_ids = $query->posts;

		if ( empty( $post_ids ) ) {
			return new WP_Error( 'posts_not_found', 'Posts not found.', ['status' => 204] );
		}

		$transient_key = Facet_Wp_Remote_Data_Cache::properties_listing_site_data( $post_ids );

		$skip_cache = apply_filters( 'facetwp_remote_data_skip_cache', false );

		$properties_data = ! $skip_cache
			? get_transient( $transient_key )
			: false;

		if ( false === $properties_data ) {
			$properties_data = RE_Helpers::get_properties_listing( $post_ids );

			Facet_Wp_Remote_Data_Cache::set( $transient_key, $properties_data );
		}

		return rest_ensure_response( $properties_data );
	}

	public function get_current_listings_data( WP_REST_Request $request ) {
		$posts_per_page = $request->get_param( 'posts_per_page' );
		$paged = $request->get_param( 'page' );

		$now = new DateTimeImmutable();
		$current_year = (int) $now->format( 'Y' );
		$current_month = (int) $now->format( 'n' );

		$years = [ $current_year ];

		if ( $current_month === 1 ) {
			$years[] = $current_year - 1;
		}

		$args = [
			'post_type' => 'property',
			'post_status' => 'publish',
			'posts_per_page' => (int) $posts_per_page,
			'paged' => (int) $paged,
			'orderby' => 'date',
			'order' => 'DESC',
			'fields' => 'ids',
			'meta_query' => [
				[
					'key' => 'fave_property_year',
					'value' => $years,
					'compare' => 'IN',
				],
			],
		];

		return $this->get_listings_data( $args );
	}

	public function get_latest_listings_data( WP_REST_Request $request ) {
		$posts_per_page = $request->get_param( 'posts_per_page' );
		$paged = $request->get_param( 'page' );

		$args = [
			'post_type' => 'property',
			'post_status' => 'publish',
			'posts_per_page' => (int) $posts_per_page,
			'paged' => (int) $paged,
			'orderby' => 'date',
			'order' => 'DESC',
			'fields' => 'ids',
		];

		return $this->get_listings_data( $args );
	}

	public function facetwp_check_permissions() {
		return current_user_can( 'manage_options' );
	}

}