<?php

use Victoria\Utilities\RE_Helpers;

class Facetwp_Remote_Data_Main_Site_Rest_Api {

	public function register_rest_api_routes() {
		register_rest_route(
			'sw-facetwp/v1',
			'/get-facetwp-settings',
			[
				'methods'  => 'GET',
				'callback' => [$this, 'facetwp_get_settings'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/forward-autocomplete-ajax-call',
			[
				'methods'  => 'POST',
				'callback' => [$this, 'facetwp_subsite_forwarded_autocomplete_ajax_call'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/properties',
			[
				'methods'  => 'GET',
				'callback' => [$this, 'get_properties_data'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);

		register_rest_route(
			'sw-facetwp/v1',
			'/property/(?P<post_id>\d+)',
			[
				'methods'  => 'GET',
				'callback' => [$this, 'get_property_page_data'],
				'permission_callback' => [$this, 'facetwp_check_permissions'],
			]
		);
	}

	public function facetwp_get_settings( WP_REST_Request $request ) {
		$facetwp_settings = get_option( 'facetwp_settings' );

		if ( empty( $facetwp_settings ) ) {
			return new WP_Error( 'no_settings', 'FacetWP settings not found', ['status' => 404] );
		}

		$subsite_url = $request->get_header( 'Subsite-Url' );

		if ( ! empty( $subsite_url ) ) {
			$sanitized_origin = esc_url_raw( $subsite_url );

			$subsites = get_option( 'facetwp_remote_data_subsites_urls', [] );

			if ( ! is_array( $subsites ) ) {
				$subsites = [];
			}

			if ( ! in_array( $sanitized_origin, $subsites, true ) ) {
				$subsites[] = $sanitized_origin;

				update_option( 'facetwp_remote_data_subsites_urls', $subsites );
			}
		}

		return rest_ensure_response( $facetwp_settings );
	}

	public function facetwp_subsite_forwarded_autocomplete_ajax_call( WP_REST_Request $request ) {
		$raw_body = $request->get_body();

		$decoded_body = json_decode( $raw_body, true );

		$_POST = is_array( $decoded_body ) ? $decoded_body : [];

		if ( class_exists( 'FacetWP_Facet_Autocomplete' ) ) {
			( new \FacetWP_Facet_Autocomplete() )->ajax_load();

			exit();
		}
	}

	public function get_properties_data( WP_REST_Request $request ) {
		$post_ids = $request->get_param( 'post_ids' );

		if ( empty( $post_ids ) ) {
			return new WP_Error( 'missing_post_id', 'Post IDs are required.', ['status' => 400] );
		}

		if ( ! class_exists( RE_Helpers::class ) ) {
			return new WP_Error( 'missing_helper_class', 'RE_Helpers class does not exist.', ['status' => 400] );
		}

		$post_ids = explode( ',', $post_ids );

		$transient_key = Facet_Wp_Remote_Data_Cache::properties_listing_site_data( $post_ids );

		$skip_cache = apply_filters( 'facetwp_remote_data_skip_cache', false );

		$properties_data = ! $skip_cache
			? get_transient( $transient_key )
			: false;

		if ( false === $properties_data ) {
			$properties_data = RE_Helpers::get_properties_listing( $post_ids );

			set_transient( $transient_key, $properties_data, 10 * MINUTE_IN_SECONDS );
		}

		return rest_ensure_response( $properties_data );
	}

	public function get_property_page_data( WP_REST_Request $request ) {
		$post_id = $request->get_param( 'post_id' );

		if ( empty( $post_id ) ) {
			return new WP_Error( 'missing_post_id', 'Post ID is required.', ['status' => 400] );
		}

		if ( ! is_numeric( $post_id ) || intval( $post_id ) != $post_id ) {
			return new WP_Error( 'invalid_post_id', 'Post ID must be an integer.', ['status' => 400] );
		}

		$post_id = (int) $post_id;

		if ( ! get_post( $post_id ) ) {
			return new WP_Error( 'post_not_found', 'Post does not exist.', ['status' => 404] );
		}

		if ( get_post_type( $post_id ) !== 'property' ) {
			return new WP_Error( 'invalid_post_type', 'Post is not of type "property".', ['status' => 400] );
		}

		if ( ! class_exists( RE_Helpers::class ) ) {
			return new WP_Error( 'missing_helper_class', 'RE_Helpers class does not exist.', ['status' => 400] );
		}

		$transient_key = Facet_Wp_Remote_Data_Cache::property_site_data( $post_id );

		$skip_cache = apply_filters( 'facetwp_remote_data_skip_cache', false );

		$property_data = ! $skip_cache
			? get_transient( $transient_key )
			: false;

		if ( false === $property_data ) {
			$property_data = [
				'property_id' => $post_id,
				'property_template_tags' => RE_Helpers::get_property_template_tags( $post_id, 'api' ),
				'property_calculator_data' => RE_Helpers::get_calculator_data( $post_id ),
			];

			set_transient( $transient_key, $property_data, HOUR_IN_SECONDS );
		}

		return rest_ensure_response( $property_data );
	}

	public function facetwp_check_permissions() {
		return current_user_can( 'manage_options' );
	}

}