<?php

class Facetwp_Remote_Data_Main_Site_Facetwp_Handler {

	protected $background_process;

	public const POST_IDS_TO_SKIP_FACETWP_INDEXING = 'facetwp_post_ids_to_skip';

	public function __construct() {
		$this->background_process = new Facetwp_Remote_Data_Background_Process();
	}

	public function facetwp_api_can_access() {
		return current_user_can( 'manage_options' );
	}

	public function update_option_facetwp_settings( $old_value, $new_value ) {
		if ( $old_value === $new_value ) return;

		$this->trigger_sync_on_subsites( 'sync-settings' );
	}

	public function facetwp_indexer_post_facet( $return, $params ) {
		$invalid_post_ids = $this->get_invalid_post_ids();

		$post_id = $params['defaults']['post_id'];

		if ( in_array( $post_id, $invalid_post_ids ) ) {
			return true; // Skip indexing this post
		}

		return $return;
	}

	private function get_invalid_post_ids() {
		$transient_key = self::POST_IDS_TO_SKIP_FACETWP_INDEXING;

		$cached_ids = get_transient( $transient_key );

		if ( false !== $cached_ids ) {
			return $cached_ids;
		}

		global $wpdb;

		$required_meta_keys = [
			'fave_property_price',
			'fave_property_address',
			'fave_property_id',
			'houzez_geolocation_lat',
			'houzez_geolocation_long',
		];

		$placeholders = implode(',', array_fill( 0, count( $required_meta_keys ), '%s' ) );

		$sql = $wpdb->prepare( "
			SELECT p.ID
			FROM {$wpdb->posts} p
			LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
				AND pm.meta_key IN ($placeholders)
				AND pm.meta_value != ''
			WHERE p.post_type = 'property'
				AND p.post_status = 'publish'
			GROUP BY p.ID
			HAVING COUNT(DISTINCT pm.meta_key) < %d
	    ", array_merge( $required_meta_keys, [ count( $required_meta_keys ) ] ) );

		$invalid_post_ids = $wpdb->get_col( $sql );

		set_transient( $transient_key, $invalid_post_ids, 20 * MINUTE_IN_SECONDS );

		return $invalid_post_ids;
	}

	public function facetwp_indexer_complete() {
		delete_transient( self::POST_IDS_TO_SKIP_FACETWP_INDEXING );

		Facet_Wp_Remote_Data_Cache::delete_cache();

		$this->trigger_sync_on_subsites( 'sync-index-table' );
	}

	private function trigger_sync_on_subsites( $sync_target ) {
		$subsites = get_option( 'facetwp_remote_data_subsites_urls', [] );

		if ( empty( $subsites ) || ! is_array( $subsites ) ) return;

		foreach ( $subsites as $subsite_url ) {
			$this->background_process->push_to_queue(
				[
					'subsite_url' => $subsite_url,
					'sync_target' => $sync_target,
					'tries' => 0,
				]
			);
		}

		$this->background_process->save();
	}

	public function facetwp_api_output( $output ) {
		$filtered_post_ids = FWP()->filtered_post_ids;

		if ( ! empty( $filtered_post_ids ) ) {
			$all_locations_data = $this->get_all_posts_for_map_facet( $filtered_post_ids );

			if ( ! empty( $all_locations_data ) ) {
				$output['facets']['map']['all_locations'] = $all_locations_data;
			}
		}

		return $output;
	}

	public static function trigger_facetwp_sync( $subsite_url, $sync_target ) {
		$sync_url = trailingslashit( $subsite_url ) . 'wp-json/sw-facetwp/v1/' . $sync_target;

		error_log( "FacetWP Remote Data: Triggering sync for: {$sync_url}, syncing {$sync_target}" );

		$response = wp_remote_get(
			$sync_url, [
				'headers' => [
					'Main-Site-Url' => get_bloginfo( 'url' )
				],
				'timeout' => 30,
				'sslverify' => Facetwp_Remote_Data_Main_Site_Utils::should_ssl_verify()
			]
		);

		if ( is_wp_error( $response ) ) {
			error_log( "ERROR: FacetWP Remote Data - syncing with {$subsite_url}: " . $response->get_error_message() );
			return false;
		}

		$http_code = wp_remote_retrieve_response_code( $response );

		if ( $http_code !== 200 ) {
			error_log( "ERROR: FacetWP Remote Data - received HTTP status code {$http_code} from {$subsite_url}." );
			return false;
		}

		error_log( "SUCCESS: FacetWP Remote Data - FacetWP settings synced successfully for {$subsite_url}." );

		return true;
	}

	public function get_all_posts_for_map_facet( $post_ids ) {
		if ( empty( $post_ids ) || ! is_array( $post_ids ) ) return [];

		$transient_key = Facet_Wp_Remote_Data_Cache::properties_listing_map_data( $post_ids );

		$skip_cache = apply_filters( 'facetwp_remote_data_skip_cache', false );

		$properties_map_data = ! $skip_cache
			? get_transient( $transient_key )
			: false;

		if ( false === $properties_map_data ) {
			global $wpdb;

			$query = $wpdb->prepare(
				"SELECT p.ID as property_id,
	                CAST(pm_lat.meta_value AS DECIMAL(10,6)) AS latitude,
	                CAST(pm_lng.meta_value AS DECIMAL(10,6)) AS longitude,
	                pm_address.meta_value AS address,
	                pm_price.meta_value AS price,
	                pm_bathrooms.meta_value AS bathrooms,
	                pm_bedrooms.meta_value AS bedrooms,
	                pm_size.meta_value AS square_feet,
	                wp_attachment.guid AS featured_image
		        FROM {$wpdb->posts} p
		        LEFT JOIN {$wpdb->postmeta} pm_lat ON p.ID = pm_lat.post_id AND pm_lat.meta_key = 'houzez_geolocation_lat'
		        LEFT JOIN {$wpdb->postmeta} pm_lng ON p.ID = pm_lng.post_id AND pm_lng.meta_key = 'houzez_geolocation_long'
		        LEFT JOIN {$wpdb->postmeta} pm_thumbnail ON p.ID = pm_thumbnail.post_id AND pm_thumbnail.meta_key = '_thumbnail_id'
		        LEFT JOIN {$wpdb->postmeta} pm_address ON p.ID = pm_address.post_id AND pm_address.meta_key = 'fave_property_address'
		        LEFT JOIN {$wpdb->postmeta} pm_price ON p.ID = pm_price.post_id AND pm_price.meta_key = 'fave_property_price'
		        LEFT JOIN {$wpdb->postmeta} pm_bathrooms ON p.ID = pm_bathrooms.post_id AND pm_bathrooms.meta_key = 'fave_property_bathrooms'
		        LEFT JOIN {$wpdb->postmeta} pm_bedrooms ON p.ID = pm_bedrooms.post_id AND pm_bedrooms.meta_key = 'fave_property_bedrooms'
		        LEFT JOIN {$wpdb->postmeta} pm_size ON p.ID = pm_size.post_id AND pm_size.meta_key = 'fave_property_size'
		        LEFT JOIN {$wpdb->posts} wp_attachment ON wp_attachment.ID = pm_thumbnail.meta_value AND wp_attachment.post_type = 'attachment'
		        WHERE p.post_status = 'publish'
					AND p.ID IN (" . implode( ',', array_fill( 0, count( $post_ids ), '%d' ) ) . ")
					AND NOT (
					    CAST(pm_lat.meta_value AS DECIMAL(10,6)) = 0
					    AND CAST(pm_lng.meta_value AS DECIMAL(10,6)) = 0
					)",
				$post_ids
			);

			$properties_map_data = $wpdb->get_results( $query, ARRAY_A );

			Facet_Wp_Remote_Data_Cache::set( $transient_key, $properties_map_data );
		}

		return $properties_map_data;
	}

}