<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://statenweb.com
 * @since      1.0.0
 *
 * @package    Facetwp_Remote_Data_Main_Site
 * @subpackage Facetwp_Remote_Data_Main_Site/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Facetwp_Remote_Data_Main_Site
 * @subpackage Facetwp_Remote_Data_Main_Site/includes
 * @author     Marko Bader <marko@statenweb.com>
 */
class Facetwp_Remote_Data_Main_Site {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Facetwp_Remote_Data_Main_Site_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'FACETWP_REMOTE_DATA_MAIN_SITE_VERSION' ) ) {
			$this->version = FACETWP_REMOTE_DATA_MAIN_SITE_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'facetwp-remote-data-main-site';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_background_process_hooks();
		$this->define_facetwp_hooks();
		$this->define_rest_api_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Facetwp_Remote_Data_Main_Site_Loader. Orchestrates the hooks of the plugin.
	 * - Facetwp_Remote_Data_Main_Site_i18n. Defines internationalization functionality.
	 * - Facetwp_Remote_Data_Main_Site_Admin. Defines all hooks for the admin area.
	 * - Facetwp_Remote_Data_Main_Site_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facetwp-remote-data-main-site-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facetwp-remote-data-main-site-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-facetwp-remote-data-main-site-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-facetwp-remote-data-main-site-public.php';

		/**
		 * The class responsible for defining helper functions.
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facetwp-remote-data-main-site-utils.php';

		/**
		 * The class responsible for defining REST API endpoints.
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facetwp-remote-data-main-site-rest-api.php';

		/**
		 * The class responsible for defining all actions related to background process.
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facetwp-remote-data-main-site-background-process.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/background-process/facetwp-remote-data-background-process.php';

		/**
		 * The class responsible for defining all actions related to FacetWP plugin.
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facetwp-remote-data-main-site-facetwp-handler.php';

		/**
		 * The class responsible for handling cache functionality (transients).
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-facet-wp-remote-data-cache.php';

		$this->loader = new Facetwp_Remote_Data_Main_Site_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Facetwp_Remote_Data_Main_Site_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Facetwp_Remote_Data_Main_Site_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Facetwp_Remote_Data_Main_Site_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		$this->loader->add_action( 'save_post_property', $plugin_admin, 'save_post_property' );

		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_plugin_admin_menu' );

		$this->loader->add_action( 'wp_ajax_delete_facetwp_cache', $plugin_admin, 'wp_ajax_delete_facetwp_cache_callback' );

	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Facetwp_Remote_Data_Main_Site_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

	}

	/**
	 * Register all of the hooks related to the FacetWP functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_facetwp_hooks() {

		$facetwp_handler = new Facetwp_Remote_Data_Main_Site_Facetwp_Handler();

		$this->loader->add_filter( 'facetwp_api_can_access', $facetwp_handler, 'facetwp_api_can_access', 50, 2 );
		$this->loader->add_action( 'update_option_facetwp_settings', $facetwp_handler, 'update_option_facetwp_settings', 10, 2 );
		$this->loader->add_action( 'facetwp_api_output', $facetwp_handler, 'facetwp_api_output', 30, 1 );
	}

	/**
	 * Register all of the hooks related to the REST API functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_rest_api_hooks() {

		$rest_api = new Facetwp_Remote_Data_Main_Site_Rest_Api();

		$this->loader->add_action( 'rest_api_init', $rest_api, 'register_rest_api_routes' );
	}

	/**
	 * Register all of the hooks related to the background process functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_background_process_hooks() {

		$background_process = new Facetwp_Remote_Data_Main_Site_Background_Process();

		$this->loader->add_action( 'plugins_loaded', $background_process, 'init' );
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Facetwp_Remote_Data_Main_Site_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
