<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\SEO
 * @author  StudioPress
 * @license GPL-2.0-or-later
 * @link    https://my.studiopress.com/themes/genesis/
 */

use StudioPress\Genesis\SEO\Open_Graph;

/**
 * Detect plugins that emit Open Graph tags.
 *
 * Helps to determine if the Open Graph Customizer panel should be visible when
 * Genesis SEO is active.
 *
 * Excludes plugins tested by `genesis_detect_seo_plugins()`. Genesis Open
 * Graph is already disabled if those are active.
 *
 * @since 3.2.0
 *
 * @return bool True if Open Graph Plugins are detected, false otherwise.
 */
function genesis_detect_open_graph_plugins() {

	/**
	 * Alter the Open Graph plugins Genesis will deactivate Open Graph support for.
	 *
	 * @since 3.2.0
	 *
	 * @param array $open_graph_plugins Open Graph classes, functions, and constants to detect.
	 */
	$open_graph_plugins = apply_filters(
		'genesis_detect_open_graph_plugins',
		[
			'classes'   => [
				'Iworks_Opengraph', // https://wordpress.org/plugins/og/.
			],
			'functions' => [
				'jetpack_og_tags', // https://wordpress.org/plugins/jetpack/.
				'opengraph_default_metadata', // https://wordpress.org/plugins/opengraph/.

			],
			'constants' => [
				'WEBDADOS_FB_VERSION', // https://wordpress.org/plugins/wonderm00ns-simple-facebook-open-graph-tags/.
				'COMPLETE_OPEN_GRAPH_OPTIONS_PREFIX', // https://wordpress.org/plugins/complete-open-graph/.
			],
		]
	);

	return genesis_detect_plugin( $open_graph_plugins );
}

/**
 * Determines if Open Graph settings should be visible in the Customizer.
 *
 * @since 3.2.0
 *
 * @return bool True if Genesis Open Graph options should be visible.
 */
function genesis_open_graph_available() {
	/**
	 * Filter to disable Genesis Open Graph.
	 *
	 * ```php
	 * add_filter( 'genesis_open_graph_enabled', '__return_false' );
	 * ```
	 *
	 * @since 3.2.0
	 *
	 * @param bool $genesis_open_graph_enabled True if enabled, false otherwise.
	 */
	return apply_filters( 'genesis_open_graph_enabled', true )
		&& genesis_seo_active()
		&& ! genesis_detect_open_graph_plugins();
}

add_action( 'after_setup_theme', 'genesis_open_graph' );
/**
 * Initializes Genesis Open Graph support if enabled.
 *
 * @since 3.2.0
 */
function genesis_open_graph() {
	if ( ! genesis_open_graph_available() ) {
		return;
	}

	$genesis_open_graph_config = apply_filters(
		'genesis_open_graph_config',
		[
			'icon_size' => 512,
		]
	);

	$og = new Open_Graph( $genesis_open_graph_config );
	$og->add_hooks();
}
