<?php
/**
 * Custom functions
 *
 * @package Ascend Theme
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Page titles
 */
function ascend_title() {
	if ( is_home() ) {
		if ( get_option( 'page_for_posts', true ) ) {
				$title = get_the_title( get_option( 'page_for_posts', true ) );
		} else {
				$title = __( 'Latest Posts', 'ascend' );
		}
	} elseif ( is_search() ) {
		$title = sprintf( __( 'Search Results for %s', 'ascend' ), get_search_query() );
	} else if ( is_post_type_archive( 'course' ) && function_exists( 'llms_get_page_id' ) ) {
		$title = get_the_title( llms_get_page_id( 'courses' ) );
	} else if ( is_post_type_archive( 'llms_membership' ) && function_exists( 'llms_get_page_id' ) ) {
		$title = get_the_title( llms_get_page_id( 'memberships' ) );
	} elseif ( is_404() ) {
		$title = __( 'Not Found', 'ascend' );
	} else {
		$title = get_the_title();
	}
	return apply_filters( 'ascend_title', $title );
}
/**
 * Archive Titles
 */
function ascend_title_archive() {
	if ( is_home() ) {
		if ( get_option( 'page_for_posts', true ) ) {
			$title = get_the_title( get_option( 'page_for_posts', true ) );
		} else {
			$title = __( 'Latest Posts', 'ascend' );
		}
	} elseif ( is_archive() ) {
		$title = get_the_archive_title();
	} elseif ( is_search() ) {
		$title = sprintf( __( 'Search Results for %s', 'ascend' ), get_search_query() );
	} elseif ( is_404() ) {
		$title = __( 'Not Found', 'ascend' );
	} else {
		$title = get_the_title();
	}
	return apply_filters( 'ascend_title', $title );
}
add_filter( 'get_the_archive_title', 'ascend_filter_archive_title' );
function ascend_filter_archive_title( $title ) {
		$term = get_term_by( 'slug', get_query_var( 'term' ), get_query_var( 'taxonomy' ) );
	if ( is_category() ) {
		$title = single_cat_title( '', false );
	} elseif ( is_tag() ) {
		$title = single_tag_title( '', false );
	} elseif ( is_author() ) {
		$title = sprintf( __( 'Author: %s', 'ascend' ), get_the_author() );
	} elseif ( is_day() ) {
		$title = sprintf( __( 'Day: %s', 'ascend' ), get_the_date() );
	} elseif ( is_month() ) {
		$title = sprintf( __( 'Month: %s', 'ascend' ), get_the_date( 'F Y' ) );
	} elseif ( is_year() ) {
		$title = sprintf( __( 'Year: %s', 'ascend' ), get_the_date( 'Y' ) );
	} elseif ( is_tax( array( 'product_cat', 'product_tag' ) ) ) {
		$title = single_term_title( '', false );
	} else if ( $term ) {
		$title = $term->name;
	} else if ( function_exists( 'is_bbpress' ) ) {
		if ( is_bbpress() ) {
			$title = bbp_title();
		}
	} else if ( function_exists( 'tribe_is_month' ) && ( tribe_is_month() || tribe_is_past() || tribe_is_upcoming() || tribe_is_day() ) ) {
		$title = tribe_get_event_label_plural();
	} else if ( function_exists( 'tribe_is_photo' ) && ( tribe_is_map() || tribe_is_photo() || tribe_is_week() ) ) {
		$title = tribe_get_event_label_plural();
	} else if ( is_post_type_archive( 'course' ) && function_exists( 'llms_get_page_id' ) ) {
		$title = get_the_title( llms_get_page_id( 'courses' ) );
	} else if ( is_post_type_archive( 'llms_membership' ) && function_exists( 'llms_get_page_id' ) ) {
		$title = get_the_title( llms_get_page_id( 'memberships' ) );
	} elseif ( is_post_type_archive() ) {
		$title = post_type_archive_title( '', false );
	}
	return $title;
}


/**
 * Setup Portfolio Permalinks.
 */
function ascend_permalinks() {

	global $wp_rewrite, $ascend;
	if ( ! empty( $ascend['portfolio_permalink'] ) ) {
		$port_rewrite = $ascend['portfolio_permalink'];
	} else {
		$port_rewrite = 'portfolio';
	}

	$portfolio_structure = '/' . $port_rewrite . '/%portfolio%';
	$wp_rewrite->add_rewrite_tag( '%portfolio%', '([^/]+)', 'portfolio=' );
	$wp_rewrite->add_permastruct( 'portfolio', $portfolio_structure, false );

	add_filter( 'post_type_link', 'ascend_portfolio_permalink', 10, 3 );
}
add_action( 'after_setup_theme', 'ascend_permalinks' );

/**
 * Setup Portfolio Permalinks.
 *
 * @param string  $permalink the permalink.
 * @param integer $post_id   the post id.
 * @param string  $leavename the post leavename.
 */
function ascend_portfolio_permalink( $permalink, $post_id, $leavename ) {
	$post = get_post( $post_id );
	$rewritecode = array(
		'%year%',
		'%monthnum%',
		'%day%',
		'%hour%',
		'%minute%',
		'%second%',
		$leavename ? '' : '%postname%',
		'%post_id%',
		'%category%',
		'%author%',
		$leavename ? '' : '%pagename%',
	);

	if ( '' != $permalink && ! in_array( $post->post_status, array( 'draft', 'pending', 'auto-draft' ) ) ) {
		$unixtime = strtotime( $post->post_date );

		$category = '';
		if ( strpos( $permalink, '%category%' ) !== false ) {
			$cats = wp_get_post_terms(
				$post->ID,
				'portfolio-type',
				array(
					'orderby' => 'parent',
					'order' => 'DESC',
				)
			);
			if ( $cats ) {
				$category = $cats[0]->slug;
			}
			// show default category in permalinks, without
			// having to assign it explicitly.
			if ( empty( $category ) ) {
				$category = 'portfolio-category';
			}
		}

		$author = '';
		if ( strpos( $permalink, '%author%' ) !== false ) {
			$authordata = get_userdata( $post->post_author );
			$author = $authordata->user_nicename;
		}

		$date = explode( ' ', date( 'Y m d H i s', $unixtime ) );
		$rewritereplace =
		array(
			$date[0],
			$date[1],
			$date[2],
			$date[3],
			$date[4],
			$date[5],
			$post->post_name,
			$post->ID,
			$category,
			$author,
			$post->post_name,
		);
		$permalink = str_replace( $rewritecode, $rewritereplace, $permalink );
	}

	return $permalink;
}

/**
 * Flush rules for custom post types.
 */
function ascend_reflush_rules() {
	global $wp_rewrite;
	$wp_rewrite->flush_rules();
}
add_action( 'after_switch_theme', 'ascend_reflush_rules' );

/**
 * Custom 404 template.
 *
 * @param string $template the page template.
 */
function ascend_custom_404_filter_template( $template ) {

	global $ascend;

	if ( isset( $ascend['enable_custom_404'] ) && '1' === $ascend['enable_custom_404'] ) {
		if ( isset( $ascend['custom_404_page'] ) && ! empty( $ascend['custom_404_page'] ) ) {
			global $wp_query;
			global $post;

			$post = get_post( $ascend['custom_404_page'] ); // phpcs:ignore

			$wp_query->posts             = array( $post );
			$wp_query->queried_object_id = $post->ID;
			$wp_query->queried_object    = $post;
			$wp_query->post_count        = 1;
			$wp_query->found_posts       = 1;
			$wp_query->max_num_pages     = 0;
			$wp_query->is_404            = false;
			$wp_query->is_page           = true;

			return get_page_template();

		} else {

			return $template;
		}
	} else {
		return $template;
	}
}
add_filter( '404_template', 'ascend_custom_404_filter_template' );

/**
 * Hex to RGB
 *
 * @param string $hex string hex code.
 */
function ascend_hex2rgb( $hex ) {
	$hex = str_replace( '#', '', $hex );

	if ( strlen( $hex ) == 3 ) {
		$r = hexdec( substr( $hex, 0, 1 ) . substr( $hex, 0, 1 ) );
		$g = hexdec( substr( $hex, 1, 1 ) . substr( $hex, 1, 1 ) );
		$b = hexdec( substr( $hex, 2, 1 ) . substr( $hex, 2, 1 ) );
	} else {
		$r = hexdec( substr( $hex, 0, 2 ) );
		$g = hexdec( substr( $hex, 2, 2 ) );
		$b = hexdec( substr( $hex, 4, 2 ) );
	}
	$rgb = array( $r, $g, $b );

	return $rgb;
}
/**
 * Get Primary Term
 *
 * @param string $taxonomy string term name.
 * @param string $post_id string post id.
 */
function ascend_get_taxonomy_main( $taxonomy, $post_id ) {
	$main_term = '';
	$terms     = wp_get_post_terms(
		$post_id,
		$taxonomy,
		array(
			'orderby' => 'parent',
			'order'   => 'DESC',
		)
	);
	if ( $terms && ! is_wp_error( $terms ) ) {
		if ( is_array( $terms ) ) {
			$main_term = $terms[0];
		}
	}
	return $main_term;
}

/**
 * Get Primary Term
 *
 * @param string $post_id string post id.
 * @param string $term string term name.
 */
function ascend_get_primary_term( $post_id, $term ) {
	$main_term = '';
	if ( class_exists( 'WPSEO_Primary_Term' ) ) {
		$wpseo_term = new WPSEO_Primary_Term( $term, $post_id );
		$wpseo_term = $wpseo_term->get_primary_term();
		$wpseo_term = get_term( $wpseo_term );
		if ( is_wp_error( $wpseo_term ) ) {
			$main_term = ascend_get_taxonomy_main( $term, $post_id );
		} else {
			$main_term = $wpseo_term;
		}
	} elseif ( class_exists( 'RankMath' ) ) {
		$wpseo_term = get_post_meta( $post_id, 'rank_math_primary_category', true );
		if ( $wpseo_term ) {
			$wpseo_term = get_term( $wpseo_term );
			if ( is_wp_error( $wpseo_term ) ) {
				$main_term = ascend_get_taxonomy_main( $term, $post_id );
			} else {
				$main_term = $wpseo_term;
			}
		} else {
			$main_term = ascend_get_taxonomy_main( $term, $post_id );
		}
	} else {
		$main_term = ascend_get_taxonomy_main( $term, $post_id );
	}

	return $main_term;
}
/**
 * Ascend RGB to HSL
 *
 * @param array $rgb the color array.
 */
function ascend_rgb2hsl( $rgb ) {
	$r = $rgb[0] / 255.0;
	$g = $rgb[1] / 255.0;
	$b = $rgb[2] / 255.0;

	$max_c = max( $r, $g, $b );
	$min_c = min( $r, $g, $b );

	$l = ( $max_c + $min_c ) / 2.0;

	if ( $max_c == $min_c ) {
		$s = 0;
		$h = 0;
	} else {
		if ( $l < .5 ) {
			$s = ( $max_c - $min_c ) / ( $max_c + $min_c );
		} else {
			$s = ( $max_c - $min_c ) / ( 2.0 - $max_c - $min_c );
		}
		if ( $r == $max_c ) {
			$h = ( $g - $b ) / ( $max_c - $min_c );
		}
		if ( $g == $max_c ) {
			$h = 2.0 + ( $b - $r ) / ( $max_c - $min_c );
		}
		if ( $b == $max_c ) {
			$h = 4.0 + ( $r - $g ) / ( $max_c - $min_c );
		}
		$h = $h / 6.0;
	}

	$h = (int) round( 255.0 * $h );
	$s = (int) round( 255.0 * $s );
	$l = (int) round( 255.0 * $l );

	return (object) array(
		'hue' => $h,
		'saturation' => $s,
		'lightness' => $l,
	);
}

/**
 * Hex adjust brightness
 *
 * @param string $hex string hex code.
 * @param int    $steps adjustment steps.
 */
function ascend_adjust_brightness( $hex, $steps ) {
	// Steps should be between -255 and 255. Negative = darker, positive = lighter.
	$steps = max( -255, min( 255, $steps ) );

	// Normalize into a six character long hex string.
	$hex = str_replace( '#', '', $hex );
	if ( strlen( $hex ) == 3 ) {
		$hex = str_repeat( substr( $hex, 0, 1 ), 2 ) . str_repeat( substr( $hex, 1, 1 ), 2 ) . str_repeat( substr( $hex, 2, 1 ), 2 );
	}

	// Split into three parts: R, G and B.
	$color_parts = str_split( $hex, 2 );
	$return      = '#';

	foreach ( $color_parts as $color ) {
		$color   = hexdec( $color ); // Convert to decimal.
		$color   = max( 0, min( 255, $color + $steps ) ); // Adjust color.
		$return .= str_pad( dechex( $color ), 2, '0', STR_PAD_LEFT ); // Make two char hex code.
	}

	return $return;
}

/**
 * Page Navigation
 */
function ascend_wp_pagenav() {

	global $wp_query, $wp_rewrite;

	$pages   = '';
	$big     = 999999999; // need an unlikely integer.
	$max     = $wp_query->max_num_pages;
	$current = get_query_var( 'paged' );
	if ( ! $current ) {
		$current = 1;
	}

	$args['base']     = str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) );
	$args['total']    = $max;
	$args['current']  = $current;
	$args['add_args'] = false;

	$total             = 1;
	$args['mid_size']  = 3;
	$args['end_size']  = 1;
	if ( '2' === ascend_get_option( 'minimal_icons' ) ) {
		$args['prev_text'] = '&larr;';
	} else {
		$args['prev_text'] = '<i class="kt-icon-chevron-left"></i>';
	}
	if ( '2' === ascend_get_option( 'minimal_icons' ) ) {
		$args['next_text'] = '&rarr;';
	} else {
		$args['next_text'] = '<i class="kt-icon-chevron-right"></i>';
	}

	if ( $max > 1 ) {
		echo '<div class="scroller-status"><div class="loader-ellips infinite-scroll-request"><span class="loader-ellips__dot"></span><span class="loader-ellips__dot"></span><span class="loader-ellips__dot"></span><span class="loader-ellips__dot"></span></div></div>';
		echo '<div class="wp-pagenavi">';
	}
	if ( 1 === $total && $max > 1 ) {
		echo paginate_links( $args ); // phpcs:ignore
	}
	if ( $max > 1 ) {
		echo '</div>';
	}
}

/**
 * Custom Excerpt by length
 *
 * @param string $limit the length.
 */
function ascend_excerpt( $limit ) {
	// get Read more text.
	$excerpt = explode( ' ', get_the_excerpt(), $limit );
	if ( count( $excerpt ) >= $limit ) {
		array_pop( $excerpt );
		$excerpt = implode( ' ', $excerpt ) . '...';
	} else {
		$excerpt = implode( ' ', $excerpt );
	}
	$excerpt = preg_replace( '`\[[^\]]*\]`', '', $excerpt );
	$excerpt = str_replace( $readmore, '><', $excerpt );

	return $excerpt;
}
/**
 * Custom Content by length
 *
 * @param string $limit the length.
 */
function ascend_content( $limit ) {
	$content = explode( ' ', get_the_content(), $limit );
	if ( count( $content ) >= $limit ) {
		array_pop( $content );
		$content = implode( ' ', $content ) . '...';
	} else {
		$content = implode( ' ', $content );
	}

	return $content;
}

if ( ! function_exists( 'wp_body_open' ) ) {
	/**
	 * WP Body Open
	 */
	function wp_body_open() {
		do_action( 'wp_body_open' );
	}
}
/**
 * Assign template path for comments.
 *
 * @param string $template the template path.
 */
function ascend_comments_template( $template ) {
	$template = get_template_directory() . '/templates/comments.php';
	return $template;
}
add_filter( 'comments_template', 'ascend_comments_template' );

