<?php

namespace WpTypek\Core;

use WpTypek\Platform\Plugin;

defined( '_WPTYPEK_EXEC' ) or die( 'Restricted access' );

/**
 * Class WPTypek_Assets_Manager
 * ## <plugin_mod@mirek> ##
 */
class WPTypekAssetsManager {
    const EXCLUDE_URL_RULES = [
        'e'  => 'Equals',
        'ne' => 'Not equals',
        'c'  => 'Contains',
        'nc' => 'Not contains',
    ];

    private static $assets = [];
    private static $mainDomain;
    private static $currentPageUrl;
    private static $currentPageSlug;

    private static $exclude = [
        'css' => [
            'files'   => [],
            'plugins' => [],
        ],
        'js'  => [
            'files'   => [],
            'plugins' => [],
        ],
    ];

    public static function initialize() {
        $params                            = Plugin::getPluginParams();
        static::$exclude['css']['files']   = $params->get( 'excludeCss', [] );
        static::$exclude['css']['plugins'] = $params->get( 'excludeCssComponents', [] );
        static::$exclude['js']['files']    = $params->get( 'excludeJs', [] );
        static::$exclude['js']['plugins']  = $params->get( 'excludeJsComponents', [] );

        // main domain
        static::$mainDomain = get_bloginfo( 'url' );
        static::$mainDomain = str_replace( array( 'http://', 'https://' ), '', static::$mainDomain );

        // current page url
        static::$currentPageUrl  = $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
        static::$currentPageUrl  = trim( static::$currentPageUrl, "/" );
        static::$currentPageSlug = trim( str_replace( static::$mainDomain, '', static::$currentPageUrl ), '/' );
    }

    public static function attachAssetsExclude() {
        static::getAssets();

        add_action( 'wp_print_styles', function () {
            foreach ( static::$exclude['css'] as $fileOrPlugin => $assets ) {
                foreach ( $assets as $assetName => $excludeRules ) {
                    if ( static::shouldDequeue( $excludeRules ) ) {
                        global $wp_styles;
                        $styleWasRegistered = ! empty( $wp_styles->registered[ $assetName ] );

                        wp_dequeue_style( $assetName );

                        if ( ! $styleWasRegistered ) {
                            //@TODO some fallback action if major dequeue will not work
                        }
                    }
                }
            }
        }, 1 );

        add_action( 'wp_print_scripts', function () {
            foreach ( static::$exclude['js'] as $fileOrPlugin => $assets ) {
                foreach ( $assets as $assetName => $excludeRules ) {
                    if ( static::shouldDequeue( $excludeRules ) ) {
                        global $wp_scripts;
                        $scriptWasRegistered = ! empty( $wp_scripts->registered[ $assetName ] );

                        wp_dequeue_script( $assetName );

                        if ( ! $scriptWasRegistered ) {
                            //@TODO some fallback action if major dequeue will not work
                        }
                    }
                }
            }
        }, 1 );
    }

    public static function shouldDequeue( $rules ) {
        if ( isset( $rules['do_exclude'] ) && $rules['do_exclude'] > 0 ) {
            $currentId    = get_the_ID();
            $reverseLogic = isset( $rules['reverse_exclude'] ) ? $rules['reverse_exclude'] > 0 : false;

            $check = function ( $rules ) use ( $currentId ) {
                if ( ! empty($currentId )) {
                    if ( ! empty( $rules['exclude_pages'] ) && is_page() && in_array( $currentId, $rules['exclude_pages'] ) ) {
                        return true;
                    }

                    if ( ! empty( $rules['exclude_posts'] ) && is_single() && in_array( $currentId, $rules['exclude_posts'] ) ) {
                        return true;
                    }
                }

                if ( ! empty( $rules['exclude_urls'] ) ) {
                    foreach ( $rules['exclude_urls'] as $excludeUrlRule ) {
                        if ( ! empty( $excludeUrlRule['url'] ) ) {
                            $url = static::formatUrl( $excludeUrlRule['url'] );

                            if (
                                ( $excludeUrlRule['rule'] === 'e' && $url === static::$currentPageUrl )
                                ||
                                ( $excludeUrlRule['rule'] === 'ne' && $url !== static::$currentPageUrl )
                                ||
                                ( $excludeUrlRule['rule'] === 'c' && strpos( static::$currentPageUrl, $url ) !== false )
                                ||
                                ( $excludeUrlRule['rule'] === 'nc' && strpos( static::$currentPageUrl, $url ) === false )
                            ) {
                                return true;
                            }
                        }
                    }
                }

                return false;
            };

            $shouldExclude = $check( $rules );

            if ( $reverseLogic ) {
                return ! $shouldExclude;
            } else {
                return $shouldExclude;
            }
        }

        return false;
    }

    /**
     * @TODO that function is really slow - needs optimization
     */
    public static function getAssets() {
        return [
            'plugins' => [
                'css' => [],
                'js'  => [],
            ],
            'other'   => [
                'css' => [],
                'js'  => [],
            ],
        ];

        if ( empty( static::$assets ) ) {
            $assets          = [
                'plugins' => [
                    'css' => [],
                    'js'  => [],
                ],
                'other'   => [
                    'css' => [],
                    'js'  => [],
                ],
            ];
            $wp_typek_plugin = 'wptypek-performance/wp-typek-performance.php';

            //plugins
            $activePlugins = get_option( 'active_plugins' );

            $dirs = [];
            foreach ( $activePlugins as $plugin ) {
                if ( $plugin == $wp_typek_plugin ) {
                    continue;
                }
                $plugin          = explode( DIRECTORY_SEPARATOR, $plugin );
                $plugin          = $plugin[0];
                $dirs[ $plugin ] = static::scanDir( WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $plugin );
            }

            foreach ( $dirs as $pluginName => $plugin ) {
                foreach ( $plugin as $items ) {

                    if ( is_array( $items ) ) {
                        $items = static::array_flat( $items );
                        $items = array_filter(
                            $items,
                            function ( $item ) {
                                if ( strpos( $item, '.php' ) !== false ) {
                                    return $item;
                                }
                            }
                        );
                    } else {
                        $items = [ $items ];
                    }

                    foreach ( $items as $item ) {
                        $js = static::findAssets( $item, 'js' );
                        if ( sizeof( $js ) > 0 ) {

                            if ( ! isset( $assets['plugins']['js'][ $pluginName ] ) ) {
                                $assets['plugins']['js'][ $pluginName ] = [];
                            }

                            $assets['plugins']['js'][ $pluginName ] = array_unique( array_merge( $js, $assets['plugins']['js'][ $pluginName ] ) );
                        }

                        $css = static::findAssets( $item, 'css' );
                        if ( sizeof( $css ) > 0 ) {

                            if ( ! isset( $assets['plugins']['css'][ $pluginName ] ) ) {
                                $assets['plugins']['css'][ $pluginName ] = [];
                            }

                            $assets['plugins']['css'][ $pluginName ] = array_unique( array_merge( $css, $assets['plugins']['css'][ $pluginName ] ) );
                        }
                    }


                }
            }


            //other
            $items = static::scanDir( get_template_directory() );
            if ( get_stylesheet_directory_uri() != get_template_directory_uri() ) {
                $items = array_merge( $items, static::scanDir( get_stylesheet_directory() ) );
            }

            $items = static::array_flat( $items );
            $items = array_filter(
                $items,
                function ( $item ) {
                    if ( strpos( $item, '.php' ) !== false ) {
                        return $item;
                    }
                }
            );

            foreach ( $items as $item ) {
                $js = static::findAssets( $item, 'js' );
                if ( sizeof( $js ) > 0 ) {

                    if ( ! isset( $assets['other']['js'] ) ) {
                        $assets['other']['js'] = [];
                    }

                    $assets['other']['js'] = array_unique( array_merge( $js, $assets['other']['js'] ) );
                }

                $css = static::findAssets( $item, 'css' );
                if ( sizeof( $css ) > 0 ) {

                    if ( ! isset( $assets['other']['css'] ) ) {
                        $assets['other']['css'] = [];
                    }

                    $assets['other']['css'] = array_unique( array_merge( $css, $assets['other']['css'] ) );
                }
            }

            if ( get_stylesheet_directory_uri() != get_template_directory_uri() ) {
                $items = static::scanDir( get_stylesheet_directory() );

                foreach ( $items as $item ) {
                    $js = static::findAssets( $item, 'js' );
                    if ( sizeof( $js ) > 0 ) {

                        if ( ! isset( $assets['other']['js'] ) ) {
                            $assets['other']['js'] = [];
                        }

                        $assets['other']['js'] = array_unique( array_merge( $js, $assets['other']['js'] ) );
                    }

                    $css = static::findAssets( $item, 'css' );
                    if ( sizeof( $css ) > 0 ) {

                        if ( ! isset( $assets['other']['css'] ) ) {
                            $assets['other']['css'] = [];
                        }

                        $assets['other']['css'] = array_unique( array_merge( $css, $assets['other']['css'] ) );
                    }
                }
            }
            static::$assets = $assets;
        }

        return static::$assets;
    }

    private static function array_flat( $data ) {
        $return = [];
        array_walk_recursive( $data, function ( $a ) use ( &$return ) {
            $return[] = $a;
        } );

        return $return;
    }

    private static function findAssets( $file, $type = null ) {
        if ( $type == null ) {
            return [];
        }

        $content = str_replace( " ", '', file_get_contents( $file ) );
        if ( $type == 'js' ) {
            $assets = preg_match_all( "/wp_enqueue_script\((\s)?([\" | \'](\S)+[\"|\'])/", $content, $matches );
        } else {
            $assets = preg_match_all( "/wp_enqueue_style\((\s)?([\" | \'](\S)+[\"|\'])/", $content, $matches );
        }
        $assets = $matches[2];

        if ( sizeof( $assets ) > 0 ) {
            $assets = array_map(
                function ( $asset ) {
                    $asset = explode( ",", $asset );
                    $asset = str_replace( "'", '', str_replace( '"', '', $asset[0] ) );

                    return $asset;
                },
                $assets
            );
        }

        return $assets;
    }

    private static function scanDir( $path ) {
        $results = [];

        $paths = glob( $path . DIRECTORY_SEPARATOR . "*" );
        if ( sizeof( $paths ) > 0 ) {
            foreach ( $paths as $item ) {
                if ( strpos( $item, 'vendor' ) !== false || strpos( $item, 'assets' ) !== false ) {
                    continue;
                }
                if ( is_dir( $item ) ) {
                    $results[ $item ] = static::scanDir( $item );
                } else {
                    $extension = explode( '.', $item );
                    $extension = end( $extension );
                    if ( $extension == 'php' ) {
                        $results[] = $item;
                    }
                }
            }
        }

        return $results;
    }

    private static function formatUrl( $url ) {
        return trim( str_replace( array( 'http://', 'https://' ), '', $url ), '/' );
    }

}
