<?php

namespace Wpo\Sync;

use WP_Error;
use Wpo\Core\Domain_Helpers;
use Wpo\Core\Extensions_Helpers;
use Wpo\Core\Permissions_Helpers;
use Wpo\Core\Url_Helpers;
use Wpo\Core\Version;
use Wpo\Core\WordPress_Helpers;
use Wpo\Core\Wpmu_Helpers;
use Wpo\Services\Ajax_Service;
use Wpo\Services\Graph_Service;
use Wpo\Services\Log_Service;
use Wpo\Services\Options_Service;
use Wpo\Services\Request_Service;
use Wpo\Services\User_Create_Update_Service;
use Wpo\Services\User_Service;
use Wpo\Sync\Sync_Db;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

if ( ! class_exists( '\Wpo\Sync\SyncV2_Service' ) ) {

	class SyncV2_Service {


		/**
		 * Will schedule user synchronization using the schedule configured by the user.
		 *
		 * @since 15.0
		 *
		 * @return boolean|WP_Error
		 *
		 * @throws \Exception Argument exception.
		 */
		public static function schedule( $job_id ) {
			$can_custom_log = version_compare( Version::$current, '36.2' ) > 0;
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				return $job;
			}

			// Delete all scheduled jobs
			Sync_Helpers::get_scheduled_events( $job_id, true );

			try {

				if ( ! isset( $job['schedule'] ) || ! isset( $job['schedule']['scheduledOn'] ) || ! isset( $job['schedule']['scheduledAt'] ) ) {
					throw new \Exception( 'Too few arguments to schedule user synchronization job were supplied' );
				}

				$now = time();

				$day_of_the_week      = intval( gmdate( 'N', $now ) );
				$sel_day_of_the_week  = intval( intval( $job['schedule']['scheduledOn'] ) );
				$hours_of_the_day     = intval( gmdate( 'H', $now ) );
				$sel_hours_of_the_day = intval( intval( $job['schedule']['scheduledAt'] ) );
				$recurrence           = $sel_day_of_the_week < 7 ? 'wpo_weekly' : 'wpo_daily';
				$diff_days            = 0;
				$diff_hours           = 0;

				$seconds_in_an_hour = 60 * 60;
				$seconds_in_a_day   = 24 * $seconds_in_an_hour;
				$treshold           = 0;

				if ( $sel_day_of_the_week < 7 ) {

					if ( $sel_day_of_the_week > $day_of_the_week ) {
						$diff_days = $sel_day_of_the_week - $day_of_the_week;
					}

					if ( $sel_day_of_the_week < $day_of_the_week ) {
						$diff_days = ( 7 + $sel_day_of_the_week - $day_of_the_week );
					}
				}

				if ( $sel_hours_of_the_day > 0 && $sel_hours_of_the_day !== $hours_of_the_day ) {
					$diff_hours = $sel_hours_of_the_day - $hours_of_the_day;
				}

				if ( $diff_days === 0 && $sel_hours_of_the_day > 0 && $sel_hours_of_the_day < $hours_of_the_day ) {
					$diff_days = $sel_day_of_the_week === 7 ? 1 : 7;
				}

				$first_time = ( floor( time() / 3600 ) * 3600 ) + ( $diff_days * $seconds_in_a_day ) + ( $diff_hours * $seconds_in_an_hour ) + $treshold;
			} catch ( \Exception $e ) {
				$message = sprintf( ' %s -> Atttempting to schedule user synchronization but failed to parse time and recurrence values', __METHOD__ );
				$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
				Log_Service::write_log( 'ERROR', $message );
				return new \WP_Error( 'ScheduleParseError', $message );
			}

			$job['next'] = $first_time;
			$job['last'] = null;
			$updated     = Sync_Helpers::update_user_sync_job( $job );

			if ( is_wp_error( $updated ) ) {
				return $updated;
			}

			$result = wp_schedule_event( $first_time, $recurrence, 'wpo_sync_v2_users_start', array( $job_id ) );
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> Result of scheduling a new event: %d', __METHOD__, $result ), 'sync' );

			if ( $result === false ) {
				$message = sprintf( '%s -> Could not schedule a new cron job to synchronize users for the first time at %d and then %s', __METHOD__, $first_time, $recurrence );
				Log_Service::write_log( 'ERROR', $message );
				return new WP_Error( 'WpScheduleEventError', $message );
			}

			return true;
		}

		/**
		 * Starts the user synchronization by calling the first collection / page of
		 * users from Office 365 and then recursively continues until finished. The
		 * results are stored in custom WordPress table.
		 *
		 * @since 15.0
		 *
		 * @return mixed(bool|WP_Error) true if synchronization was successful otherwise WP_Error
		 */
		public static function sync_users( $job_id ) {
			$can_custom_log = version_compare( Version::$current, '36.2' ) > 0;
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				return new WP_Error( 'NotFoundException', sprintf( '%s -> %s', __METHOD__, $job->get_error_message() ) );
			}

			// Validate job
			$validated = self::user_sync_job_is_valid( $job );

			if ( is_wp_error( $validated ) ) {
				$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> WPO365 User Synchronization job validation failed [Error: %s]', __METHOD__, $validated->get_error_message() ), 'sync' );
				Log_Service::write_log( 'ERROR', $validated->get_error_message() );
				return $validated;
			}

			// Delete all scheduled jobs
			if ( $job['trigger'] !== 'schedule' ) {
				Sync_Helpers::get_scheduled_events( $job_id, true );
			}

			// Check if there is an instance of this job still running
			$cached_next_link = self::get_cached_next_link( $job_id );

			if ( ! empty( $cached_next_link ) ) {
				$message = sprintf(
					'%s -> Could not start a new instance of user synchronization job with ID %s because another instance did not yet finish (please stop the job in progress before starting a new one)',
					__METHOD__,
					$job_id
				);

				$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
				Log_Service::write_log( 'ERROR', $message );
				return new \WP_Error( 'NotFinished', $message );
			}

			Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> A new user synchronization job is starting and the old job data will be deleted' );

			// Verify that the table has been created
			Sync_Db::user_sync_table_exists( true );

			// Delete previous log
			Sync_Db::delete_job_data( $job_id );

			// Generate new job.last
			$job['last'] = array(
				'id'        => $job['id'] . '-' . uniqid(),
				'date'      => time(),
				'error'     => null,
				'processed' => 0,
				'total'     => -1,
			);

			// Ensure the $count parameter to be able to keep track on progress.
			$job['query'] = self::add_query_count_param( $job['query'] );

			// Ensure the $count parameter to be able to keep track on progress.
			$job['query'] = self::add_query_top_param( $job['query'] );

			$updated = Sync_Helpers::update_user_sync_job( $job );

			if ( is_wp_error( $updated ) ) {
				return $updated;
			}

			// Start
			return self::fetch_users( $job_id, '/' . $job['query'] );
		}

		/**
		 * Fetches users from Microsoft Graph using the query supplied. Can be called recursively.
		 *
		 * @since 15.0
		 *
		 * @param   string $job_id ID of the job.
		 * @param   string $graph_query Query to call Microsoft Graph.
		 * @return  boolean|WP_Error True if no errors occured otherwise an error will be returned
		 */
		public static function fetch_users( $job_id, $graph_query = null ) {
			$can_custom_log = version_compare( Version::$current, '36.2' ) > 0;
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			$stop = function ( $error ) use ( $job, $job_id, $can_custom_log ) {
				$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> fetch_users::stop [Error: %s]', ( is_wp_error( $error ) ? $error->get_error_message() : 'N.A.' ) ), 'sync' );

				// Remove memoized job info.
				self::set_cached_next_link( $job_id );

				// Update job.last before summary and sending log.
				if ( ! is_wp_error( $job ) ) {
					$job['last']['error']   = $error->get_error_message();
					$job['last']['stopped'] = true;
					Sync_Helpers::update_user_sync_job( $job );
				}

				// Send mail.
				$summary = self::get_results_summary( $job_id );
				self::sync_completed_notification( $job_id, $summary );

				// Cancel any scheduled job.
				Sync_Helpers::get_scheduled_events( $job_id, true );

				// Trigger hook.
				do_action( 'wpo365/sync/error', $job );
			};

			if ( is_wp_error( $job ) ) {
				$stop( $job );
				return;
			}

			if ( empty( $graph_query ) ) {
				$graph_query = self::get_cached_next_link( $job_id );

				if ( empty( $graph_query ) ) {
					$stop(
						new \WP_Error(
							'NextLinkExpired',
							sprintf(
								'%s -> User sync job with ID %s cannot continue because the next-link has expired',
								__METHOD__,
								$job_id
							)
						)
					);
					return;
				}
			}

				// Trigger hook
				do_action( 'wpo365/sync/before', $job );

				$fetch_result = self::process_fetch_result( Graph_Service::fetch( $graph_query, 'GET', false, array( 'Accept: application/json;odata.metadata=minimal' ) ), $job_id );

			if ( is_wp_error( $fetch_result ) ) {
				$stop( $fetch_result );
			}
		}

		/**
		 * Processes a collection of Office 365 users returned from the corresponding Microsoft Graph query. Recursively
		 * calls for the next collection when finished processing with the current collection.
		 *
		 * @since 15.0
		 *
		 * @param stdClass $response   Response returned by the MS Graph client that needs to be processed.
		 * @param string   $job_id     ID of the job.
		 *
		 * @return boolean|WP_Error
		 */
		private static function process_fetch_result( $response, $job_id ) {
			$can_custom_log = version_compare( Version::$current, '36.2' ) > 0;
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				return $job;
			}

			$unscheduled_job_id = \get_option( 'wpo_sync_v2_users_unscheduled' );

			// Check if administrator requested current job to be stopped
			if ( $unscheduled_job_id !== false ) {

				if ( ! empty( $job['last'] ) && ! empty( $job['last']['id'] ) && $unscheduled_job_id === $job['last']['id'] ) {
					delete_option( 'wpo_sync_v2_users_unscheduled' );
					return new \WP_Error( 'UserSyncStopped', __METHOD__ . ' -> Administrator requested user synchronization to stop' );
				}
			}

			// Remember for the duration of this request that users are being synchronized
			$request_service = Request_Service::get_instance();
			$request         = $request_service->get_request( $GLOBALS['WPO_CONFIG']['request_id'] );
			$request->set_item( 'user_sync', true );

			if ( ! Graph_Service::is_fetch_result_ok( $response, __METHOD__ . ' -> Could not fetch users for synchronization from Microsoft Graph' ) ) {
				return new \WP_Error( 'GraphFetchError', __METHOD__ . ' -> Could not fetch users for synchronization from Microsoft Graph [check log]' );
			}

			if ( ! is_array( $response['payload']['value'] ) ) {
				return new \WP_Error( 'GraphFetchError', __METHOD__ . ' -> Could not fetch users for synchronization from Microsoft Graph [no value returned]' );
			}

			if ( ! empty( $job['last'] ) ) {
				$job['last']['processed'] = intval( $job['last']['processed'] ) + count( $response['payload']['value'] );
				$job['last']['date']      = time();

				if ( Options_Service::get_global_boolean_var( 'use_b2c' ) || Options_Service::get_global_boolean_var( 'use_ciam' ) ) {
					// For B2C the $count parameter is not supported at the moment
					$job['last']['total'] = ( $job['last']['processed'] + 1 );
				} elseif ( ! empty( $response['payload']['@odata.count'] ) ) {
					$job['last']['total'] = $response['payload']['@odata.count'];
				}

				$updated = Sync_Helpers::update_user_sync_job( $job );

				if ( is_wp_error( $updated ) ) {
					return $updated;
				}
			}

			foreach ( $response['payload']['value'] as $o365_user ) {
				// make sure the object is a user
				if ( ! empty( $o365_user['@odata.type'] ) && WordPress_Helpers::stripos( $o365_user['@odata.type'], 'user' ) === false ) {
					$message = sprintf( '%s ->Not processing a directory object that is not a user', __METHOD__ );
					Log_Service::write_log( 'WARN', $message );
					$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
					continue;
				}

				// transform user to our own internal format
				$wpo_usr = User_Service::user_from_graph_user( $o365_user );

				// Trigger hook
				do_action( 'wpo365/sync/user', $wpo_usr );

				// Azure AD user without upn cannot be processed
				if ( ! isset( $wpo_usr->upn ) ) {
					$message = sprintf( '%s -> O365 user without userPrincipalName [oid: %s]', __METHOD__, $wpo_usr->oid );
					self::write_log( $job['last']['id'], 'unknown', 'skipped', $wpo_usr, $message, -1 );
					Log_Service::write_log( 'WARN', $message );
					$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
					continue;
				}

				// Azure AD guest user can only be processed if explicitely requested
				if ( $job['membersOnly'] && WordPress_Helpers::stripos( $wpo_usr->upn, '#ext#' ) !== false ) {
					$message = sprintf( '%s -> User is not an internal user: %s', __METHOD__, $wpo_usr->upn );
					self::write_log( $job['last']['id'], 'unknown', 'skipped', $wpo_usr, $message, -1 );
					Log_Service::write_log( 'WARN', $message );
					$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
					continue;
				}

				/**
				 * @since   21.0    Domain check has become optional.
				 */

				if ( empty( $job['skipDomainCheck'] ) ) {
					$domain = Domain_Helpers::get_smtp_domain_from_email_address( $wpo_usr->upn );

					if ( ( empty( $domain ) || ! Domain_Helpers::is_tenant_domain( $domain ) ) && ! empty( $wpo_usr->email ) ) {
						$domain = Domain_Helpers::get_smtp_domain_from_email_address( $wpo_usr->email );
					}

					if ( empty( $domain ) || ! Domain_Helpers::is_tenant_domain( $domain ) ) {
						$message = sprintf( '%s -> UPN (%s) and email (%s) domain are not in list of custom domains', __METHOD__, $wpo_usr->upn, $wpo_usr->email );
						self::write_log( $job['last']['id'], 'unknown', 'skipped', $wpo_usr, $message, -1 );
						Log_Service::write_log( 'DEBUG', $message );
						$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
						continue;
					}
				}

				$wp_user = User_Service::try_get_user_by( $wpo_usr );

				$user_created = false;
				$user_updated = false;

				// found a new Office 365 user
				if ( empty( $wp_user ) ) {

					$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> Processing new WP user [preferred_username: %s]', __METHOD__, $wpo_usr->preferred_username ), 'sync' );

					if ( $job['actionCreateUser'] ) {
						$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> Creating new WP user [preferred_username: %s]', __METHOD__, $wpo_usr->preferred_username ), 'sync' );
						$wp_id = User_Create_Update_Service::create_user( $wpo_usr, true, false );

						if ( empty( $wp_id ) ) {
							$message = sprintf( '%s -> Could not create WordPress user [preferred_username: %s]', __METHOD__, $wpo_usr->preferred_username );
							self::write_log( $job['last']['id'], 'new_domain_user', 'error', $wpo_usr, $message, -1 );
							Log_Service::write_log( 'WARN', $message );
							$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
							continue;
						} else {
							$message = sprintf( ' %s -> Successfully created new WordPress user [preferred_username: %s]', __METHOD__, $wpo_usr->preferred_username );
							self::write_log( $job['last']['id'], 'new_domain_user', 'created', $wpo_usr, $message, $wp_id );
							Log_Service::write_log( 'DEBUG', $message );
							$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );

							$wp_user      = \get_user_by( 'ID', $wp_id );
							$user_created = true;
						}
					}
				}

				// update new and / or existing wp users with group and user info
				if ( ! empty( $wp_user ) ) {
					update_user_meta( $wp_user->ID, 'wpo_sync_users_job_id', $job['last']['id'] );
					update_user_meta( $wp_user->ID, 'wpo_sync_users_last_sync', $job['last']['date'] );

					if ( $user_created || $job['actionUpdateUser'] ) {
						$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> Updating WP user [preferred_username: %s]', __METHOD__, $wpo_usr->preferred_username ), 'sync' );
						// When updating a user we want to make sure he / she is (no longer) deactivated
						delete_user_meta( $wp_user->ID, 'wpo365_active' );

						// Update role(s) assignment and extra user details
						User_Create_Update_Service::update_user( $wp_user->ID, $wpo_usr, true );

						$message = sprintf( '%s -> Successfully updated %s WordPress user [preferred_username: %s]', __METHOD__, $user_created ? 'new' : 'existing', $wpo_usr->preferred_username );
						Log_Service::write_log( 'DEBUG', $message );
						$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );

						if ( ! $user_created ) {
							self::write_log( $job['last']['id'], 'existing_domain_user', 'updated', $wpo_usr, $message, $wp_user->ID );
						}

						$user_updated = true;
					}
				}

				// User not created / updated therefore logged instead
				if ( ! $user_created && ! $user_updated ) {
					$message = sprintf( '%s -> Successfully logged %s WordPress user [preferred_username: %s]', __METHOD__, ( ! empty( $wp_user ) ? 'existing' : 'new' ), $wpo_usr->preferred_username );
					self::write_log( $job['last']['id'], ! empty( $wp_user ) ? 'existing_domain_user' : 'new_domain_user', 'logged', $wpo_usr, $message, ( empty( $wp_user ) ? -1 : $wp_user->ID ) );
					Log_Service::write_log( 'DEBUG', $message );
					$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
				}

				$message = sprintf( '%s -> Finished processing Entra ID user [preferred_username: %s]', __METHOD__, $wpo_usr->preferred_username );
				Log_Service::write_log( 'DEBUG', $message );
				$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
			}

			// continue with the next batch of users
			if ( ! empty( $response['payload']['@odata.nextLink'] ) ) {
				$graph_version = Options_Service::get_global_string_var( 'graph_version' );
				$graph_version = empty( $graph_version ) || $graph_version === 'current' ? 'v1.0' : $graph_version;
				$tld           = Options_Service::get_aad_option( 'tld' );
				$tld           = ! empty( $tld ) ? $tld : '.com';
				$graph_url     = sprintf( 'https://graph.microsoft%s/%s', $tld, $graph_version );
				$next_link     = str_replace( $graph_url, '', $response['payload']['@odata.nextLink'] );

				self::set_cached_next_link( $job_id, $next_link );
				$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> The following next-link has been saved: %s', __METHOD__, $next_link ), 'sync' );

				// Starting with 37.0 user-synchonization can continue without WP Cron.
				if ( $job['trigger'] !== 'externalCron' ) {
					$random_arg = uniqid(); // Added to prevent WordPress from scheduling a similar event within 10 minutes
					$result     = wp_schedule_single_event( time() - 60, 'wpo_sync_v2_users_next', array( $job_id, null, $random_arg ) );

					if ( $result === false ) {
						$message = sprintf( '%s -> Failed to schedule next event for hook "wpo_sync_users_next"', __METHOD__ );
						Log_Service::write_log( 'ERROR', $message );
						$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
						return new \WP_Error( 'UserSyncStopped', $message );
					} else {
						$message = sprintf( '%s -> Next event for hook "wpo_sync_users_next" has been scheduled', __METHOD__ );
						$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
						Log_Service::write_log( 'DEBUG', $message );
					}
				}
			} else {
				$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> No @odata.nextLink attribute found therefore completing the current run', __METHOD__ ), 'sync' );
				// Remove memoized job info
				self::set_cached_next_link( $job_id );

				$untagged_users_result = self::handle_untagged_users( $job_id );

				if ( is_wp_error( $untagged_users_result ) ) {
					return $untagged_users_result;
				}

				$next_cron_jobs = Sync_Helpers::get_scheduled_events( $job_id );

				if ( ! empty( $next_cron_jobs ) ) {

					foreach ( $next_cron_jobs as $cron_timestamp => $cron_job ) {

						if ( $cron_job['hook'] === 'wpo_sync_v2_users_start' ) {
							$job['next'] = $cron_timestamp;
							Sync_Helpers::update_user_sync_job( $job );
						}
					}
				}

				$summary = self::get_results_summary( $job_id );
				self::sync_completed_notification( $job_id, $summary );

				// Mark job as stopped
				$job['last']['stopped'] = true;
				$job['last']['date']    = time();
				$job['last']['total']   = $job['last']['processed'];
				Sync_Helpers::update_user_sync_job( $job );

				// Trigger hook
				do_action( 'wpo365/sync/after', $job );
			}

			return true;
		}

		/**
		 * Stops user synchronization.
		 *
		 * @since   15.3
		 *
		 * @param   string $job_id     The ID of the job to stop.
		 * @return  boolean|WP_Error
		 */
		public static function stop( $job_id ) {
			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				return $job;
			}

			// Inform the user synchronization processor to stop
			if ( ! empty( $job['last'] ) ) {
				$job['last']['error']   = __METHOD__ . ' -> Administrator requested user synchronization to stop';
				$job['last']['stopped'] = true;
				$job['last']['date']    = time();
				\update_option( 'wpo_sync_v2_users_unscheduled', $job['last']['id'] );
			}

			// Delete the schedule configuration
			$job['schedule'] = null;
			$job['next']     = null;

			// Update
			Sync_Helpers::update_user_sync_job( $job );

			// Delete all scheduled jobs
			Sync_Helpers::get_scheduled_events( $job_id, true );

			// Delete memoized job info
			self::set_cached_next_link( $job_id );

			// Send email
			$summary = self::get_results_summary( $job_id );
			self::sync_completed_notification( $job_id, $summary );

			// Trigger hook
			do_action( 'wpo365/sync/error', $job );

			return true;
		}

		/**
		 * Gets a summary of the user synchronization results (all, by status, by record type).
		 *
		 * @since 15.0
		 *
		 * @param   string $job_id The ID of the job instance.
		 * @return  array Representation of the summarized results (all, by status, by record type).
		 */
		public static function get_results_summary( $job_id, $keyword = null, $status = null ) {
			version_compare( Version::$current, '36.2' ) > 0 && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				return array(
					'all'         => 0,
					'created'     => 0,
					'deleted'     => 0,
					'deactivated' => 0,
					'updated'     => 0,
					'error'       => 0,
					'logged'      => 0,
					'skipped'     => 0,
					'info'        => $job->get_error_message(),
				);
			}

			$job_id_last = $job['last']['id'];

			global $wpdb;

			$table_name     = Sync_Db::get_user_sync_table_name();
			$keyword_clause = ! empty( $keyword ) ? " AND `upn` LIKE '%%$keyword%%' " : '';
			$status_clause  = ! empty( $status ) ? " AND `sync_job_status` = '$status' " : '';

			return array(
				'all'         => $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(upn) FROM %i WHERE `sync_job_id` = %s', $table_name, $job_id_last ) ), // phpcs:ignore
				'created'     => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(upn) FROM %i WHERE `sync_job_id` = %s $keyword_clause $status_clause AND `sync_job_status` = %s", $table_name, $job_id_last, 'created' ) ), // phpcs:ignore
				'deleted'     => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(upn) FROM %i WHERE `sync_job_id` = %s $keyword_clause $status_clause AND `sync_job_status` = %s", $table_name, $job_id_last, 'deleted' ) ), // phpcs:ignore
				'deactivated' => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(upn) FROM %i WHERE `sync_job_id` = %s $keyword_clause $status_clause AND `sync_job_status` = %s", $table_name, $job_id_last, 'deactivated' ) ), // phpcs:ignore
				'updated'     => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(upn) FROM %i WHERE `sync_job_id` = %s $keyword_clause $status_clause AND `sync_job_status` = %s", $table_name, $job_id_last, 'updated' ) ), // phpcs:ignore
				'error'       => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(upn) FROM %i WHERE `sync_job_id` = %s $keyword_clause $status_clause AND `sync_job_status` = %s", $table_name, $job_id_last, 'error' ) ), // phpcs:ignore
				'logged'      => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(upn) FROM %i WHERE `sync_job_id` = %s $keyword_clause $status_clause AND `sync_job_status` = %s", $table_name, $job_id_last, 'logged' ) ), // phpcs:ignore
				'skipped'     => $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(upn) FROM %i WHERE `sync_job_id` = %s $keyword_clause $status_clause AND `sync_job_status` = %s", $table_name, $job_id_last, 'skipped' ) ), // phpcs:ignore
				'info'        => $job['last']['error'],
			);
		}

		/**
		 * Gets a page of the paged results of the user synchronization results.
		 *
		 * @since 15.0
		 *
		 * @param   string $job_last_id    The ID of the job instance.
		 * @param   int    $page_size      Number of results to retrieve.
		 * @param   int    $offset         Number of results to skip before retrieving a page of results.
		 *
		 * @return  array   Array representation of results.
		 */
		public static function get_results( $job_last_id, $page_size, $offset, $keyword = null, $status = null ) {

			global $wpdb;

			$table_name     = Sync_Db::get_user_sync_table_name();
			$keyword_clause = ! empty( $keyword ) ? " AND `upn` LIKE '%%$keyword%%' " : '';
			$status_clause  = ! empty( $status ) ? " AND `sync_job_status` = '$status' " : '';

			return $wpdb->get_results(  // phpcs:ignore
				$wpdb->prepare(
					"SELECT * FROM %i WHERE `sync_job_id` = %s $keyword_clause $status_clause LIMIT %d OFFSET %d", // phpcs:ignore
					$table_name,
					$job_last_id,
					$page_size,
					$offset
				),
				ARRAY_A
			);
		}

		/**
		 * Will get user sync job by job.last.id.
		 *
		 * @since 15.0
		 *
		 * @param   string $job_last_id    The ID of the job.
		 * @return  array|WP_Error          The job found or an error if not found.
		 */
		public static function get_user_sync_job_by_last_id( $job_last_id ) {
			$jobs = Options_Service::get_global_list_var( 'user_sync_jobs', false );
			$job  = 0;

			foreach ( $jobs as $_job ) {

				if ( ! empty( $_job['last'] ) && $_job['last']['id'] === $job_last_id ) {
					$job = $_job;
					break;
				}
			}

			if ( empty( $job ) ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> User synchronization stopped [Job with job.last.ID ' . $job_last_id . ' not found.]' );
				return new \WP_Error( 'NotFound', __METHOD__ . ' -> Job with job.last.ID ' . $job_last_id . ' not found.' );
			}

			return $job;
		}

		/**
		 *
		 * @since   15.0
		 *
		 * @param   string $sync_job_last_id   ID of the job instance.
		 * @param   string $record_type        One of the following options.
		 *                                     unknown                 -> Type of user indeterminate
		 *                                     new_domain_user         -> Azure AD user without WordPress account
		 *                                     existing_domain_user    -> Azure AD user with a WordPress account.
		 * @param   string $action_performed   One of the following options
		 *                                     created                 -> New WordPress user created
		 *                                     deleted                 -> WordPress user deleted
		 *                                     error                   -> WordPress user could not be created or updated
		 *                                     skipped                 -> Azure AD user has not been processed
		 *                                     updated                 -> Existing WordPress user updated.
		 * @param   User   $wpo_usr
		 * @param   string $notes
		 * @param   int    $wp_user_id
		 */
		private static function write_log( $sync_job_last_id, $record_type, $action_performed, $wpo_usr, $notes = '', $wp_user_id = -1 ) {
			global $wpdb;

			$table_name = Sync_Db::get_user_sync_table_name();

			if ( intval(
				$wpdb->get_var( // phpcs:ignore
					$wpdb->prepare(
						'SELECT COUNT(*) as num_rows FROM %i WHERE `upn` = %s AND `sync_job_id` = %s',
						$table_name,
						$wpo_usr->upn,
						$sync_job_last_id
					)
				)
			) === 0 ) {
				$wpdb->insert(
					$table_name,
					array(
						'wp_id'           => $wp_user_id,
						'upn'             => esc_sql( $wpo_usr->upn ),
						'first_name'      => esc_sql( $wpo_usr->first_name ),
						'last_name'       => esc_sql( $wpo_usr->last_name ),
						'full_name'       => esc_sql( $wpo_usr->full_name ),
						'email'           => esc_sql( $wpo_usr->email ),
						'sync_job_id'     => $sync_job_last_id,
						'name'            => esc_sql( $wpo_usr->name ),
						'sync_job_status' => $action_performed,
						'record_type'     => $record_type,
						'notes'           => esc_sql( $notes ),
					)
				);
			} else {
				$message = sprintf(
					'%s -> Trying to create a duplicate log entry for %s [%s]',
					__METHOD__,
					$wpo_usr->preferred_username,
					$notes
				);
				Log_Service::write_log( 'ERROR', $message );
			}

			if ( ! empty( $wpdb->last_error ) ) {
				Log_Service::write_log( 'ERROR', sprintf( '%s -> Error occurred when trying to write a log entry for user with upn %s', __METHOD__, $wpo_usr->upn ) );
			}
		}

		/**
		 * Checks if query contains the $count(=true) parameter and if not adds it.
		 *
		 * @since 15.0
		 *
		 * @param   string $query
		 *
		 * @return  string  Updated query with $count=true param.
		 */
		private static function add_query_count_param( $query ) {
			version_compare( Version::$current, '36.2' ) > 0 && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			if ( WordPress_Helpers::stripos( $query, '$count=' ) === false ) {
				return WordPress_Helpers::stripos( $query, '?' ) !== false
					? $query . '&$count=true'
					: $query . '?$count=true';
			}

			return $query;
		}

		/**
		 * Checks if query contains the $count(=true) parameter and if not adds it.
		 *
		 * @since 24.1
		 *
		 * @param string $query
		 * @return string Updated query with $top=10 param.
		 */
		private static function add_query_top_param( $query ) {
			version_compare( Version::$current, '36.2' ) > 0 && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			if ( WordPress_Helpers::stripos( $query, '$top=' ) === false ) {
				return WordPress_Helpers::stripos( $query, '?' ) !== false
					? $query . '&$top=10'
					: $query . '?$top=10';
			}

			return $query;
		}

		/**
		 * Validates the user sync job.
		 *
		 * @since 15.0
		 *
		 * @param   array $job The user sync job to be validated.
		 * @return  array|WP_Error  The job or WP_Error if invalid.
		 */
		private static function user_sync_job_is_valid( $job ) {
			version_compare( Version::$current, '36.2' ) > 0 && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			$error_fields = array();

			if ( empty( $job['name'] ) ) {
				$error_fields[] = 'name is empty';
			}

			if ( empty( $job['query'] ) ) {
				$error_fields[] = 'query is empty';
			}

			if ( empty( filter_var( $job['queryTested'], FILTER_VALIDATE_BOOLEAN ) === false ) ) {
				$error_fields[] = 'query is not tested';
			}

			if ( $job['trigger'] === 'schedule' && empty( $job['schedule'] ) ) {
				$error_fields[] = 'job schedule is empty';
			}

			if ( $job['sendLog'] && empty( $job['sendLogTo'] ) ) {
				$error_fields[] = 'mail recipient to send log is empty';
			}

			if ( filter_var( $job['actionDeleteUser'], FILTER_VALIDATE_BOOLEAN ) === true && empty( $job['reassignPostsToId'] ) ) {
				$error_fields[] = 'user to re-assign posts to is empty';
			}

			if ( ! empty( $errors ) ) {
				$last_error = __METHOD__ . ' -> User synchronization stopped [User sync job is invalid: ' . join( ', ', $error_fields ) . ']';
				Log_Service::write_log( 'ERROR', $last_error );
				return new \WP_Error( 'ArgumentException', $last_error );
			}

			return $job;
		}

		/**
		 * Queries all users for the current job tag and if not found will add those users
		 * to the user sync table as untagged users (no matching Office 365 user was found).
		 *
		 * @since 15.0
		 *
		 * @return bool|WP_Error
		 */
		private static function handle_untagged_users( $job_id ) {

			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				return $job;
			}

			$untagged_users_query = new \WP_User_Query(
				array(
					'meta_query' => array( // phpcs:ignore
						'relation' => 'OR',
						array(
							'key'     => 'wpo_sync_users_job_id',
							'value'   => $job['last']['id'],
							'compare' => '!=',
						),
						array(
							'key'     => 'wpo_sync_users_job_id',
							'compare' => 'NOT EXISTS',
						),
					),
				)
			);

			require_once ABSPATH . 'wp-admin/includes/user.php';

			$table_name  = Sync_Db::get_user_sync_table_name();
			$soft_delete = ! empty( $job['softDeleteUsers'] );
			$reassign_to = ! empty( $job['reassignPostsToId'] ) ? intval( $job['reassignPostsToId'] ) : null;

			// And fill it with the results of the last run
			$untagged_users = $untagged_users_query->get_results();

			foreach ( $untagged_users as $untagged_user ) {
				$skip_delete     = false;
				$sync_job_status = 'skipped';
				$record_type     = 'wordpress_user';

				$wp_user = get_user_by( 'ID', $untagged_user->ID );
				$notes   = '';

				if ( ! $wp_user ) {
					Log_Service::write_log( 'WARN', __METHOD__ . ' -> Cannot retrieve untagged user with ID ' . $untagged_user->ID . ' because user cannot be found' );
					continue;
				}

				if ( ! Options_Service::get_global_boolean_var( 'update_admins' ) && Permissions_Helpers::user_is_admin( $wp_user ) ) {
					$notes = __METHOD__ . ' -> Skipping user with login ' . $wp_user->user_login . ' because user has administrator capabilities';
					Log_Service::write_log( 'DEBUG', $notes );
					$skip_delete = true;
				} elseif ( empty( $job['skipDomainCheck'] ) ) {
					$domain = Domain_Helpers::get_smtp_domain_from_email_address( $wp_user->user_login );

					if ( empty( $domain ) || ! Domain_Helpers::is_tenant_domain( $domain ) ) {
						$domain = Domain_Helpers::get_smtp_domain_from_email_address( $wp_user->user_email );
					}

					if ( empty( $domain ) || ! Domain_Helpers::is_tenant_domain( $domain ) ) {
						$notes = __METHOD__ . ' -> Skipping user with user name ' . $wp_user->user_login . ' [reason: upn and email domain do not match with any of the configured custom domains]';
						Log_Service::write_log( 'DEBUG', $notes );
						$skip_delete = true;
					}
				} else {
					$aad_object_id = get_user_meta( $wp_user->ID, 'aadObjectId', true );

					if ( empty( $aad_object_id ) ) {
						$notes = __METHOD__ . ' -> Skipping user with user name ' . $wp_user->user_login . ' [reason: user has not been tagged with oid claim]';
						Log_Service::write_log( 'DEBUG', $notes );
						$skip_delete = true;
					}
				}

				// Finally commit deletion of WP users if requested
				if ( ! $skip_delete && $job['actionDeleteUser'] === true ) {

					if ( ! $soft_delete ) {
						$sync_job_status = wp_delete_user( $wp_user->ID, $reassign_to ) ? 'deleted' : 'error';
						$notes           = sprintf( '%s -> User with user name %s has %sbeen deleted', __METHOD__, $wp_user->user_login, $sync_job_status === 'deleted' ? '' : 'NOT ' );
					} else {
						\update_user_meta( $wp_user->ID, 'wpo365_active', 'deactivated' );
						$sync_job_status = 'deactivated';

						// Remove all roles of the deactivated user
						foreach ( $wp_user->roles as $current_user_role ) {
							$wp_user->remove_role( $current_user_role );
						}

						$notes = sprintf( '%s -> User with user name %s has been deactivated', __METHOD__, $wp_user->user_login );
					}
				}

				global $wpdb;

				$results = $wpdb->get_results( // phpcs:ignore
					$wpdb->prepare(
						'SELECT * FROM %i WHERE `upn` = %s AND `sync_job_id` = %s',
						$table_name,
						$wp_user->user_login,
						$job['last']['id']
					)
				);

				if ( $wpdb->num_rows === 0 ) {
					$wpdb->insert( // phpcs:ignore
						$table_name,
						array(
							'wp_id'           => $wp_user->ID,
							'upn'             => esc_sql( $wp_user->user_login ),
							'first_name'      => '', // defined in user meta
							'last_name'       => '', // defined in user meta
							'full_name'       => isset( $wp_user->display_name ) ? esc_sql( $wp_user->display_name ) : '',
							'email'           => isset( $wp_user->user_email ) ? esc_sql( $wp_user->user_email ) : '',
							'sync_job_id'     => $job['last']['id'],
							'name'            => esc_sql( $wp_user->user_login ),
							'sync_job_status' => $sync_job_status,
							'record_type'     => $record_type,
							'notes'           => $notes,
						)
					);
				} else {
					$wpdb->update( // phpcs:ignore
						$table_name,
						array(
							'notes' => sprintf( '[1] %s; [2] %s', $results[0]->notes, $notes ),
						),
						array(
							'upn'         => esc_sql( $results[0]->upn ),
							'sync_job_id' => $results[0]->sync_job_id,
						)
					);
				}

				if ( ! empty( $wpdb->last_error ) ) {
					Log_Service::write_log( 'ERROR', sprintf( '%s -> Error occurred when trying to write a log entry for untagged user %s [Error: %s]', __METHOD__, $wp_user->user_login, $wpdb->last_error ) );
				}
			}

			return true;
		}

		/**
		 * Helper to register custom columns to show a couple of WPO365 User synchronization related fields
		 * on the default WordPress Users screen.
		 *
		 * @since   21.0
		 *
		 * @param   array $columns
		 *
		 * @return  array    Array of colums with a couple of Azure AD related columns added.
		 */
		public static function register_users_sync_columns( $columns ) {
			$columns['wpo365_synchronized'] = __( 'Last sync', 'wpo365-login' );
			$columns['wpo365_deactivated']  = __( 'De-activated', 'wpo365-login' );
			return $columns;
		}

		/**
		 * Helper to render a couple of custom WPO365 User sync columns that are added to the default WordPress Users screen.
		 *
		 * @since   21.0
		 *
		 * @param   string $output         Rendered HTML.
		 * @param   string $column_name    Name of the column being rendered.
		 * @param   string $user_id        ID of the user the column's cell is being rendered for.
		 *
		 * @return  string  Rendered HTML.
		 */
		public static function render_users_sync_columns( $output, $column_name, $user_id ) {
			if ( $column_name === 'wpo365_synchronized' ) {
				$last_sync = get_user_meta( $user_id, 'wpo_sync_users_last_sync', true );

				if ( empty( $last_sync ) ) {
					return $output;
				}

				$formatted = method_exists( '\Wpo\Core\WordPress_Helpers', 'time_zone_corrected_formatted_date' )
					? WordPress_Helpers::time_zone_corrected_formatted_date( $last_sync, 'Y-m-d H:i' )
					: gmdate( 'Y-m-d H:i', $last_sync );

				return sprintf( '<div><span>%s</span></div>', $formatted );
			}

			if ( $column_name === 'wpo365_deactivated' ) {
				$deactivated = get_user_meta( $user_id, 'wpo365_active', true );

				if ( $deactivated === 'deactivated' ) {
					$url = add_query_arg( 'wpo365_reactivate_user', $user_id );
					return sprintf( '<div><span><button type="button" onclick="window.location.href = \'%s\'">Reactivate</button></span></div>', $url );
				}
			}

			return $output;
		}

		/**
		 * Helper to reactivate a user from the WP users list.
		 *
		 * @since 21.0
		 *
		 * @return void
		 */
		public static function reactivate_user() {
			if ( isset( $_GET['wpo365_reactivate_user'] ) ) { // phpcs:ignore
				$wp_user_id = (int) sanitize_text_field( wp_unslash( $_GET['wpo365_reactivate_user'] ) ); // phpcs:ignore
				$wp_user    = get_user_by( 'ID', $wp_user_id );

				if ( ! empty( $wp_user ) && get_user_meta( $wp_user->ID, 'wpo365_active', true ) === 'deactivated' ) {
					delete_user_meta( $wp_user->ID, 'wpo365_active' );

					$role = is_main_site()
						? Options_Service::get_global_string_var( 'new_usr_default_role' )
						: Options_Service::get_global_string_var( 'mu_new_usr_default_role' );

					$wp_user->set_role( $role );
				}

				Url_Helpers::force_redirect( remove_query_arg( 'wpo365_reactivate_user' ) );
			}
		}

		/**
		 * Adds custom bulk actions to the central WordPress user list e.g. to allow administrators to manually reactivate multiple users.
		 *
		 * @since   24.0
		 *
		 * @param   array $bulk_actions
		 * @return  array   The (modified) array of bulk actions
		 */
		public static function users_sync_bulk_actions( $bulk_actions ) {
			if ( ! Options_Service::get_global_boolean_var( 'enable_user_sync' ) ) {
				return $bulk_actions;
			}

			$bulk_actions['re-activate-users'] = __( 'Re-activate users', 'wpo365-login' );
			return $bulk_actions;
		}

		/**
		 * Applies the custom bulk actions e.g. to create / update users in Azure AD (B2C).
		 *
		 * @since   24.0
		 *
		 * @param   string $redirect_url
		 * @param   string $action
		 * @param   array  $user_ids
		 * @return  string  The redirect URL with an additional parameter to sho
		 */
		public static function users_sync_bulk_actions_handler( $redirect_url, $action, $user_ids ) {
			if ( ! Options_Service::get_global_boolean_var( 'enable_user_sync' ) ) {
				return $redirect_url;
			}

			if ( $action === 're-activate-users' ) {

				foreach ( $user_ids as $wp_usr_id ) {
					$wp_usr = get_user_by( 'ID', $wp_usr_id );

					if ( ! empty( $wp_usr ) && get_user_meta( $wp_usr->ID, 'wpo365_active', true ) === 'deactivated' ) {
						delete_user_meta( $wp_usr->ID, 'wpo365_active' );

						$role = is_main_site()
							? Options_Service::get_global_string_var( 'new_usr_default_role' )
							: Options_Service::get_global_string_var( 'mu_new_usr_default_role' );

						$wp_usr->set_role( $role );
					}
				}

				$redirect_url = add_query_arg( 're-activate-users', count( $user_ids ), $redirect_url );
			}

			return $redirect_url;
		}

		public static function test_sync_query() {
			// Verify AJAX request
			$current_user = Ajax_Service::verify_ajax_request( 'to toggle (user synchronization) test-query mode' );

			if ( Permissions_Helpers::user_is_admin( $current_user ) === false ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> User has no permission to toggle test-query mode from AJAX service' );
				wp_die();
			}

			Ajax_Service::verify_posted_data( array( 'endpoint', 'query', 'scope' ), false ); // die

			$scope    = wp_sanitize_redirect( rawurldecode( wp_unslash( $_POST['scope'] ) ) ); // phpcs:ignore
			$endpoint = sanitize_text_field( wp_unslash( $_POST['endpoint'] ) ); // phpcs:ignore
			$query    = Url_Helpers::leadingslashit( $endpoint . Url_Helpers::leadingslashit( str_replace( "\'", "'", sanitize_text_field( rawurldecode( rawurldecode( wp_unslash( $_POST['query'] ) ) ) ) ) ) ); // phpcs:ignore

			$result = Graph_Service::fetch( $query, 'GET', false, array(), false, false, '', $scope );

			if ( \is_wp_error( $result ) ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> Could not fetch data from Microsoft Graph [' . $result->get_error_message() . '].' );
				Ajax_Service::ajax_response( 'NOK', 'GraphFetchError', $result->get_error_message(), null );
			}

			if ( $result['response_code'] < 200 || $result['response_code'] > 299 ) {
				$json_encoded_result = wp_json_encode( $result );
				Log_Service::write_log( 'WARN', __METHOD__ . ' -> Could not fetch data from Microsoft Graph [' . $json_encoded_result . '].' );
				Ajax_Service::ajax_response( 'NOK', $result['response_code'], 'Your request to Microsoft Graph returned an invalid HTTP response code [' . $json_encoded_result . '].', null );
			}

			Ajax_Service::ajax_response( 'OK', $result['response_code'], '', $result['payload'] );
		}

		/**
		 * Gets the cached next link. Returns false not found.
		 *
		 * @param string $job_id
		 * @return false|string
		 */
		public static function get_cached_next_link( $job_id ) {
			$option_name  = sprintf( '%s_wpo365_sync_next', $job_id );
			$option_value = get_option( $option_name );

			if ( is_array( $option_value ) && ! empty( $option_value['next_link'] ) && ! empty( $option_value['expires'] ) ) {

				if ( time() > (int) $option_value['expires'] ) {
					// Next link is expired.
					Log_Service::write_log(
						'WARN',
						sprintf(
							'%s -> User sync job with id "%s" cannot continue because the next-link has expired.',
							__METHOD__,
							$job_id
						)
					);
					delete_option( $option_name );
					return false;
				}

				Log_Service::write_log(
					'DEBUG',
					sprintf(
						'%s -> Found the following next-link: "%s".',
						__METHOD__,
						$option_value['next_link']
					)
				);
				return (string) $option_value['next_link'];
			}

			Log_Service::write_log(
				'WARN',
				sprintf(
					'%s -> Next-link for job with id "%s" not found.',
					__METHOD__,
					$job_id
				)
			);
			return false;
		}

		/**
		 * Sends the admin of the site an email to inform that user synchronization has completed.
		 *
		 * @since 15.0
		 *
		 * @return void
		 */
		private static function sync_completed_notification( $job_id, $summary = array() ) {
			version_compare( Version::$current, '36.2' ) > 0 && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				Log_Service::write_log( 'WARN', __METHOD__ . ' -> Could not find the user synchronization job whilst trying to send user-synchronization-completed email' );
				return;
			}

			if ( empty( $job['sendLog'] ) || empty( $job['sendLogTo'] ) ) {
				Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Sending of a user-synchronization-completed email is not configured' );
				return;
			}

			// Bail out now if the job was successful and admin only wishes to send a notification on failure
			if ( ! empty( $job['sendLogOnError'] ) && empty( $summary['info'] ) && absint( $summary['error'] ) === 0 ) {
				return;
			}

			$blogname = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );

			include Extensions_Helpers::get_active_extension_dir( array( 'wpo365-login-premium/wpo365-login.php', 'wpo365-sync-5y/wpo365-sync-5y.php', 'wpo365-login-intranet/wpo365-login.php', 'wpo365-intranet-5y/wpo365-intranet-5y.php', 'wpo365-customers/wpo365-customers.php', 'wpo365-integrate/wpo365-integrate.php' ) ) . '/templates/sync-mail-text.php';

			$message = empty( $summary['info'] )
				? sprintf( $body, $blogname, $job['name'], $summary['all'], $summary['created'], $summary['deleted'], $summary['deactivated'], $summary['updated'], $summary['error'], $summary['logged'], $summary['skipped'] )
				: sprintf( $body, $blogname, $job['name'], $summary['info'] );

			if ( empty( $summary['info'] ) ) {

				if ( absint( $summary['error'] ) > 0 ) {
					$subject = sprintf( 'WPO365 | User Synchronization COMPLETED WITH ERRORS on your site [%s]', $blogname );
				} else {
					$subject = sprintf( 'WPO365 | User Synchronization SUCCEEDED on your site [%s]', $blogname );
				}
			} else {
				$subject = sprintf( 'WPO365 | User Synchronization FAILED on your site [%s]', $blogname );
			}

			$sync_completed_email_admin = array(
				'to'      => $job['sendLogTo'],
				'subject' => $subject,
				'message' => $message,
				'headers' => array( 'Content-Type: text/plain' ),
			);

			@wp_mail( // phpcs:ignore
				$sync_completed_email_admin['to'],
				$sync_completed_email_admin['subject'],
				$sync_completed_email_admin['message'],
				$sync_completed_email_admin['headers']
			);
		}

		/**
		 * Caches the next link in the wp options table, or deletes it if $next_link is null.
		 *
		 * @param string      $job_id
		 * @param string|null $next_link
		 *
		 * @return void
		 */
		private static function set_cached_next_link( $job_id, $next_link = null ) {
			$option_name = sprintf( '%s_wpo365_sync_next', $job_id );

			if ( $next_link === null ) {
				delete_option( $option_name );
				Log_Service::write_log(
					'WARN',
					sprintf(
						'%s -> Deleted next-link for job with id "%s".',
						__METHOD__,
						$job_id
					)
				);
				return;
			}

			$option_value = array(
				'next_link' => $next_link,
				'expires'   => strtotime(
					'+1 hour',
					time(),
				),
			);

			update_option( $option_name, $option_value );

			Log_Service::write_log(
				'DEBUG',
				sprintf(
					'%s -> Saved the folowing next-link "%s".',
					__METHOD__,
					$next_link
				)
			);
		}
	}
}
