<?php

namespace Wpo\Services;

use Wpo\Core\Url_Helpers;
use Wpo\Services\Error_Service;
use Wpo\Services\Log_Service;
use Wpo\Services\Options_Service;

if ( ! class_exists( '\Wpo\Services\Logout_Service' ) ) {

	class Logout_Service {


		/**
		 * Hooks into a default logout action and additionally logs out the user from Office 365 before sending
		 * the user to the default login page.
		 *
		 * @since 3.1
		 *
		 * @return void
		 */
		public static function logout_o365( $wp_user_id = 0 ) {

			if ( Options_Service::get_global_boolean_var( 'logout_from_o365' ) ) {
				$post_logout_redirect_uri = Options_Service::get_global_string_var( 'post_signout_url' );

				if ( empty( $post_logout_redirect_uri ) ) {
					$post_logout_redirect_uri = Url_Helpers::get_preferred_login_url();
				}

				$request_service = Request_Service::get_instance();
				$request         = $request_service->get_request( $GLOBALS['WPO_CONFIG']['request_id'] );
				$wpo_auth_value  = $request->get_item( 'wpo_auth_value' );

				/**
				 * @since 30.3 Take multiple IdPs into account
				 */

				$wpo_idps = Wp_Config_Service::get_multiple_idps();

				if ( ! empty( $wpo_idps ) && $wp_user_id > 0 ) {
					$wpo_idp_id = get_user_meta( $wp_user_id, 'wpo365_idp_id', true );

					if ( $wpo_idp_id !== false ) {
						$filtered_idps = array_filter(
							$wpo_idps,
							function ( $idp ) use ( $wpo_idp_id ) {
								return ! empty( $idp['id'] ) && strcasecmp( $idp['id'], $wpo_idp_id ) === 0;
							}
						);

						$filtered_idps = array_values( $filtered_idps ); // re-index from 0

						if ( count( $filtered_idps ) === 1 ) {
							$request_service = Request_Service::get_instance();
							$request         = $request_service->get_request( $GLOBALS['WPO_CONFIG']['request_id'] );
							$request->set_item( 'wpo_aad', $filtered_idps[0] );

							if ( isset( $filtered_idps[0]['type'] ) ) {
								$use_saml = $filtered_idps[0]['type'] === 'saml';
								Options_Service::add_update_option( 'use_saml', $use_saml );
							}

							if ( isset( $filtered_idps[0]['tenant_type'] ) ) {
								$use_b2c  = $filtered_idps[0]['tenant_type'] === 'b2c';
								$use_ciam = $filtered_idps[0]['tenant_type'] === 'ciam';
								Options_Service::add_update_option( 'use_b2c', $use_b2c );
								Options_Service::add_update_option( 'use_ciam', $use_ciam );
							}
						}
					}
				}

				if ( ! empty( $wpo_auth_value ) ) {
					if ( Options_Service::get_global_boolean_var( 'use_b2c' ) ) {
						$domain_name = Options_Service::get_aad_option( 'b2c_domain_name' );
						$policy      = Options_Service::get_aad_option( 'b2c_policy_name' );

						/**
						 * @since   20.x    Support for custom b2c login domain e.g. login.contoso.com
						 */

						$b2c_domain = Options_Service::get_aad_option( 'b2c_custom_domain' );

						if ( empty( $b2c_domain ) ) {
							$b2c_domain = sprintf( 'https://%s.b2clogin.com/', $domain_name );
						} else {
							$b2c_domain = sprintf( 'https://%s', trailingslashit( $b2c_domain ) );
						}

						$logout_url = sprintf(
							'%s%s.onmicrosoft.com/%s/oauth2/v2.0/logout?post_logout_redirect_uri=%s',
							$b2c_domain,
							$domain_name,
							$policy,
							$post_logout_redirect_uri
						);
					} elseif ( Options_Service::get_global_boolean_var( 'use_ciam' ) ) {
						$domain_name = Options_Service::get_aad_option( 'b2c_domain_name' );

						/**
						 * @since   20.x    Support for custom b2c login domain e.g. login.contoso.com
						 */

						$ciam_domain = Options_Service::get_aad_option( 'b2c_custom_domain' );

						if ( empty( $ciam_domain ) ) {
							$ciam_domain = sprintf( 'https://%s.ciamlogin.com/', $domain_name );
						} else {
							$ciam_domain = sprintf( 'https://%s', trailingslashit( $ciam_domain ) );
						}

						$logout_url = sprintf(
							'%s%s.onmicrosoft.com/oauth2/v2.0/logout?post_logout_redirect_uri=%s',
							$ciam_domain,
							$domain_name,
							$post_logout_redirect_uri
						);
					} else {
						$tld        = Options_Service::get_aad_option( 'tld' );
						$tld        = ! empty( $tld ) ? $tld : '.com';
						$logout_url = sprintf(
							'https://login.microsoftonline%s/common/oauth2/v2.0/logout?post_logout_redirect_uri=%s',
							$tld,
							$post_logout_redirect_uri
						);
					}

					Url_Helpers::force_redirect( $logout_url );
				}
			}
		}

		/**
		 * Hooks into a default logout action and sends the user to a custom "error" page in case the
		 * Administrator has enabled SSO for the login page.
		 *
		 * @since 12.x
		 *
		 * @param int $wp_user_id   ID of the user logging out.
		 *
		 * @return void
		 */
		public static function send_to_custom_logout_page( $wp_user_id = 0 ) {
			if ( Options_Service::get_global_boolean_var( 'redirect_on_login' ) ) {
				$error_page_url = Options_Service::get_global_string_var( 'error_page_url' );

				if ( empty( $error_page_url ) ) {
					Log_Service::write_log( 'ERROR', __METHOD__ . ' -> Administrator has enabled SSO for the login page but not configured an error page to send the user when he / she logs out' );
					return;
				}

				$redirect_to = apply_filters( 'wpo365/goodbye/error_page_uri', $error_page_url, $wp_user_id, Error_Service::LOGGED_OUT );

				// Only add error information if redirect_to is equal to unmodified error_page_url.
				if ( strcmp( $redirect_to, $error_page_url ) === 0 ) {
					$redirect_to = add_query_arg( 'login_errors', Error_Service::LOGGED_OUT, $redirect_to );
				}

				Url_Helpers::force_redirect( $redirect_to );
			}
		}
	}
}
