<?php

namespace Wpo\Mail;

use Wpo\Core\Permissions_Helpers;
use Wpo\Mail\Mail_Db;
use Wpo\Services\Ajax_Service;
use Wpo\Services\Log_Service;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

if ( ! class_exists( '\Wpo\Mail\Mail_Ajax_Service' ) ) {

	class Mail_Ajax_Service {


		/**
		 * Gets the debug log
		 *
		 * @since 7.11
		 *
		 * @return void
		 */
		public static function get_mail_log() {
			// Verify AJAX request
			$current_user = Ajax_Service::verify_ajax_request( 'to get a portion of the wpo365 mail log' );

			if ( Permissions_Helpers::user_is_admin( $current_user ) === false ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> User has no permission to get a portion of the wpo365 mail log from AJAX service' );
				wp_die();
			}

			Ajax_Service::verify_posted_data( array( 'start_row', 'page_size', 'filter' ) ); // wp_die

			$start_row = isset( $_POST['start_row'] ) && ctype_digit( $_POST['start_row'] ) ? \intval( $_POST['start_row'] ) : 0; // phpcs:ignore
			$page_size = isset( $_POST['page_size'] ) && ctype_digit( $_POST['page_size'] ) ? \intval( $_POST['page_size'] ) : 0; // phpcs:ignore
			$filter    = isset( $_POST['filter'] ) && $_POST['filter'] === 'error' ? 'error' : 'all'; // phpcs:ignore
			$log       = Mail_Db::get_mail_log( $start_row, $page_size, $filter );
			Ajax_Service::ajax_response( 'OK', '', '', wp_json_encode( $log ) );
		}

		/**
		 * Try to send the mail with id equal to $_POST[ 'id' ] again.
		 *
		 * @since   17.0
		 *
		 * @return  void
		 */
		public static function send_mail_again() {
			// Verify AJAX request
			$current_user = Ajax_Service::verify_ajax_request( 'to send mail again' );

			if ( Permissions_Helpers::user_is_admin( $current_user ) === false ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> User has no permission to send mail again from AJAX service' );
				wp_die();
			}

			Ajax_Service::verify_posted_data( array( 'id' ) ); // die

			if ( ! isset( $_POST['id'] ) || ! ctype_digit( $_POST['id'] ) ) { // phpcs:ignore
				$message = 'Cannot convert parameter id to integer when trying to send mail again';
				Log_Service::write_log( 'ERROR', __METHOD__ . " -> $message" );
				Ajax_Service::ajax_response( 'NOK', '', $message, null );
				wp_die();
			}

			$id      = \intval( $_POST['id'] ); // phpcs:ignore
			$success = Mail_Db::send_mail_again( $id );
			Ajax_Service::ajax_response( 'OK', '', '', wp_json_encode( array( 'mail_success' => $success ) ) );
		}

		/**
		 * Truncates the wpo365_mail table.
		 *
		 * @since   17.0
		 *
		 * @return  void
		 */
		public static function truncate_mail_log() {
			// Verify AJAX request
			$current_user = Ajax_Service::verify_ajax_request( 'to truncate the wpo365_mail table' );

			if ( Permissions_Helpers::user_is_admin( $current_user ) === false ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> User has no permission to truncate the wpo365_mail table from AJAX service' );
				wp_die();
			}

			$success = Mail_Db::truncate_mail_log();
			Ajax_Service::ajax_response( 'OK', '', '', wp_json_encode( array( 'mail_success' => $success ) ) );
		}

		/**
		 * WP-AJAX wrapper for Mail_Db::ensure_unsent_messages to remove / add a WP-Cron event to check regularly for unsent messages.
		 *
		 * @since 24.0
		 *
		 * @return void
		 */
		public static function mail_auto_retry() {
			// Verify AJAX request
			$current_user = Ajax_Service::verify_ajax_request( 'to toggle the mail-auto-retry feature' );

			if ( Permissions_Helpers::user_is_admin( $current_user ) === false ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> User has no permission to toggle the mail-auto-retry feature from AJAX service' );
				wp_die();
			}

			$remove = isset( $_POST['unschedule'] ) && true === filter_var( $_POST['unschedule'], FILTER_VALIDATE_BOOLEAN ); // phpcs:ignore

			$result = Mail_Db::ensure_unsent_messages( $remove );

			if ( empty( $result ) ) {
				Ajax_Service::ajax_response( 'NOK', '', 'Could not delete / create the WP Cron Job that automatically resends failed emails', wp_json_encode( array() ) );
			}

			Ajax_Service::ajax_response( 'OK', '', '', wp_json_encode( array() ) );
		}
	}
}
