<?php

namespace Wpo\Insights;

use Wpo\Core\Extensions_Helpers;
use Wpo\Core\WordPress_Helpers;
use Wpo\Insights\Event_Notify_Db_Service;
use Wpo\Services\Log_Service;
use Wpo\Services\Options_Service;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

if ( ! class_exists( 'Wpo\Insights\Event_Notify_Service' ) ) {

	class Event_Notify_Service {

		private $p_attempts  = 0;
		private $p_category  = '';
		private $p_license   = '';
		private $p_message   = '';
		private $p_o_message = '';
		private $p_recipient = '';

		/**
		 * Validates the notification before submitting to insights.wpo365.com.
		 *
		 * @since 38.0
		 *
		 * @param string $message
		 * @param string $category e.g. 'MAIL'.
		 * @param string $recipient e.g. 'user@domain.tld'.
		 * @return void
		 */
		public function notify( $message, $category = 'N/A', $recipient = '' ) {
			$this->p_category = $category;

			// Validate message.
			if ( Options_Service::get_global_boolean_var( 'insights_alerts_no_sensitive_data' ) ) {
				$this->p_message = '[No sensitive data] To view the full alert message, go to your site → WP Admin → WPO365 → Dashboard → Insights.';
			} elseif ( empty( $message ) ) {
				$log_message = 'An error occurred sending a notification to insights.wpo365.com. [Error: Empty message]';
				Log_Service::write_log( 'ERROR', $log_message );
				do_action( 'wpo365/alert/submitted/fail', $log_message, 'ALERT', array( 'message' => $message ) );
				return;
			} else {
				$this->p_message = $message;
			}

			$this->p_o_message = $message; // Keep the original message to show on the Insights Dashboard.

			// Validate recipient or take admin email.
			if ( empty( $recipient ) ) {
				$recipient = get_bloginfo( 'admin_email' );
			}

			if ( filter_var( $recipient, FILTER_VALIDATE_EMAIL ) === false || strcasecmp( $recipient, 'user@example.com' ) === 0 ) {
				$log_message = sprintf(
					'An error occurred sending a notification to insights.wpo365.com. [Error: Invalid recipient "%s"]',
					$recipient
				);
				Log_Service::write_log( 'ERROR', $log_message );
				do_action( 'wpo365/alert/submitted/fail', $log_message, 'ALERT', array( 'message' => $message ) );
				return;
			} else {
				$this->p_recipient = $recipient;
			}

			$this->p_license = $this->get_license_key();

			if ( empty( $this->p_license ) ) {
				$log_message = sprintf(
					'An error occurred sending a notification to insights.wpo365.com. [Error: Invalid license "%s"]',
					$this->p_license
				);
				Log_Service::write_log( 'ERROR', $log_message );
				do_action( 'wpo365/alert/submitted/fail', $log_message, 'ALERT', array( 'message' => $message ) );
				return;
			}

			// Send message to API.
			$this->send();

			// Ensure that the WP Cron job to check for failed notifications has been created.
			if ( ! wp_next_scheduled( 'wpo365_insights_check_failed_notifications' ) ) {
				wp_schedule_event( time(), 'hourly', 'wpo365_insights_check_failed_notifications' );
			}
		}

		/**
		 * Check the database for notifications that failed to submit and try again submitting.
		 *
		 * @since 38.0
		 *
		 * @return void
		 */
		public function check_failed_notifications() {
			$notifications   = Event_Notify_Db_Service::get_instance()->get_failed_notifications();
			$this->p_license = $this->get_license_key();

			foreach ( $notifications as $notification ) {
				$payload = json_decode( $notification->notification_payload );

				if ( empty( $this->p_license ) ) {
					Log_Service::write_log(
						'ERROR',
						sprintf(
							'An error occurred re-sending notification with id %s to insights.wpo365.com. [Error: Invalid license "%s"]',
							$payload->id,
							$this->p_license
						)
					);
					continue;
				}

				$this->p_message   = $payload->message;
				$this->p_recipient = $payload->recipient;

				$this->send( $notification->id );
			}
		}

		/**
		 * Sending the notification to insights.wpo365.com.
		 *
		 * @since 0.1.0
		 *
		 * @param int $id The id of the cached item or empty when sending for the first time.
		 *
		 * @return void
		 */
		private function send( $id = -1 ) {
			++$this->p_attempts;

			$payload = array(
				'goto'      => admin_url( 'admin.php?page=wpo365-wizard#dashboard' ),
				'id'        => sprintf( '%d|%s', (int) $id, $this->p_license ),
				'message'   => $this->p_message,
				'recipient' => $this->p_recipient,
				'site'      => get_bloginfo( 'name' ),
				'url'       => home_url(),
			);

			$log = function ( $error = null, $success = true ) use ( &$payload, $id ) {

				if ( ! empty( $error ) ) {
					Log_Service::write_log( 'ERROR', $error );
					do_action(
						'wpo365/alert/submitted/fail',
						$error,
						'ALERT',
						array(
							'id'      => '|' . $payload['id'] . '|',
							'message' => $this->p_o_message, // In the event - which is maintained locally - show the actual error message.
						)
					);
				}

				$inserted_id = Event_Notify_Db_Service::get_instance()->cache_notification( $payload, $success, $error );

				if ( (int) $id === -1 && $inserted_id !== false ) {
					$payload['id'] = sprintf(
						'%d|%s',
						$inserted_id,
						$this->p_license
					);
				}

				if ( $success ) {
					do_action(
						'wpo365/alert/submitted',
						sprintf(
							'Notification %s was sent successfully to insights.wpo365.com',
							$payload['id']
						),
						'ALERT',
						array(
							'id'      => '|' . $payload['id'] . '|',
							'message' => $this->p_o_message, // In the event - which is maintained locally - show the actual error message.
						)
					);
				}
			};

			if ( (int) $id === -1 ) {
				// Log notification preliminary to get an inserted ID.
				$log( null, false );
			}

			$response = wp_remote_post(
				'https://insights.wpo365.com/wp-json/wpo365/v1/insights/notify',
				array(
					'sslverify' => false,
					'body'      => wp_json_encode( $payload ),
					'headers'   => array(
						'Expect'               => '',
						'Content-Type'         => 'application/json',
						'Wpo-Insights-Api-Key' => base64_encode( // phpcs:ignore
							sprintf(
								'%s|%s',
								$this->p_license,
								home_url()
							)
						),
					),
				)
			);

			if ( is_wp_error( $response ) ) {
				// Log fail.
				$log(
					sprintf(
						'An error occurred sending notification with id %s to insights.wpo365.com. [Error: %s]',
						$payload['id'],
						$response->get_error_message()
					),
					false
				);
				return;
			}

			$response_code = wp_remote_retrieve_response_code( $response );
			$response_json = wp_remote_retrieve_body( $response );
			$body_as_json  = json_decode( $response_json, true );

			if ( $response_code === 403 ) {

				// Retry once whilst trying to refresh the license key.
				if ( $this->p_attempts === 1 ) {
					$this->p_license = $this->get_license_key( true );

					if ( ! empty( $this->p_license ) ) {
						$this->send(); // immediately try again with an updated license.
						return;
					}
				}

				$log(
					sprintf(
						'An error occurred sending notification with id %s to insights.wpo365.com. [Error: %s]',
						$payload['id'],
						isset( $body_as_json['message'] ) ? $body_as_json['message'] : 'License invalid'
					),
					false
				);

				return;
			} elseif ( $response_code === 409 ) {
				// Log fail.
				$log(
					sprintf(
						'An error occurred sending notification with id %s to insights.wpo365.com. [Error: %s]',
						$payload['id'],
						isset( $body_as_json['message'] ) ? $body_as_json['message'] : 'Notification already received'
					),
					true // Report success since the alert appears was sent afterall.
				);
				return;
			} elseif ( $response_code === 429 ) {
				// Log fail.
				$log(
					sprintf(
						'An error occurred sending notification with id %s to insights.wpo365.com. [Error: %s]',
						$payload['id'],
						isset( $body_as_json['message'] ) ? $body_as_json['message'] : 'Daily quota exceeded'
					),
					false // Report success to prevent the system to retry submitting the alert.
				);
				return;
			} elseif ( $response_code !== 204 ) {
				// Log fail.
				$log(
					sprintf(
						'An error occurred sending notification with id %s to insights.wpo365.com. [Status code: %d]',
						$payload['id'],
						$response_code
					),
					false
				);
				return;
			}

			// Log success.
			$log();
		}

		/**
		 * Get the license key for item with ID 41717 from the site options.
		 *
		 * @since 38.0
		 *
		 * @param bool $refresh If true, the cached value for insights_license_key will be deemed invalid and will be refreshed.
		 *
		 * @return string|null
		 */
		private function get_license_key( $refresh = false ) {
			$network_options = get_site_option( 'wpo365_options' );

			$license_key   = '';
			$store_item_id = 0;

			if ( ! empty( $network_options['insights_license_key'] ) ) {
				$license_content = explode( ',', $network_options['insights_license_key'] );
				$license_key     = isset( $license_content[0] ) ? $license_content[0] : '';
				$store_item_id   = isset( $license_content[1] ) ? (int) $license_content[1] : 0;

				if ( ! $refresh && ! empty( $license_key ) && ! empty( $store_item_id ) ) {
					return $network_options['insights_license_key'];
				}
			}

			$license_keys = array_filter(
				$network_options,
				function ( $key ) {
					return str_starts_with( $key, 'license_' ) && strcmp( $key, 'license_key' ) !== 0;
				},
				ARRAY_FILTER_USE_KEY
			);

			// No saved licenses found.
			if ( count( $license_keys ) === 0 ) {
				return null;
			}

			$active_extensions = Extensions_Helpers::get_active_extensions();
			$license_key_name  = '';

			foreach ( $active_extensions as $plugin => $plugin_data ) {

				foreach ( $license_keys as $key_name => $value ) {
					$item_id = (int) str_replace( 'license_', '', $key_name );

					// Ignore this key because is was deemed invalid.
					if ( ! empty( $license_key ) && WordPress_Helpers::stripos( $value, $license_key ) !== false && ! empty( $store_item_id ) && $store_item_id === $item_id ) {
						continue;
					}

					if ( $plugin_data['store_item_id'] === $item_id ) {
						$license_key_name = $key_name;
						$store_item_id    = $item_id;
						break;
					}
				}
			}

			// No saved valid license found for any of the active extensions.
			if ( empty( $license_key_name ) ) {
				return null;
			}

			$license_key = $network_options[ $license_key_name ];

			// Saved license present but empty.
			if ( empty( $license_key ) ) {
				return null;
			}

			if ( stripos( $license_key, '|' ) > -1 ) {
				$exploded    = explode( '|', $license_key );
				$license_key = $exploded[0];
			}

			$license = sprintf( '%s,%d', $license_key, $store_item_id );

			// Cache the license key for the next time, until deemed invalid.
			Options_Service::add_update_option( 'insights_license_key', $license );

			return $license;
		}
	}
}
