<?php

namespace Wpo\Insights;

use Wpo\Services\Log_Service;
use Wpo\Services\Options_Service;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

if ( ! class_exists( '\Wpo\Insights\Event_Notify_Db_Service' ) ) {

	class Event_Notify_Db_Service {

		private static $instance    = null;
		private $p_cache_table_name = '';

		private function __construct() {
		}

		public static function get_instance() {

			if ( empty( $instance ) ) {
				self::$instance = new Event_Notify_Db_Service();
			}

			return self::$instance;
		}

		/**
		 * Looks up cached notification that failed to submit less than 4 times.
		 *
		 * @since 0.1.0
		 *
		 * @return array
		 */
		public function get_failed_notifications() {
			$this->cache_table_exists( true );

			global $wpdb;

			$table_name = $this->get_cache_table_name();
			$like       = '%daily quota%';

			$results = $wpdb->get_results( // phpcs:ignore
				"SELECT * FROM `$table_name` WHERE `notification_success` = 0 AND `notification_attempts` < 3 AND `notification_error` NOT LIKE '$like'" // phpcs:ignore
			);

			$db_last_error = $wpdb->last_error;

			if ( ! empty( $db_last_error ) ) {
				Log_Service::write_log(
					'ERROR',
					sprintf(
						'An error occurred trying to retrieve a failed notifications. [Error: %s]',
						$db_last_error
					)
				);
			}

			return is_array( $results ) ? $results : array();
		}

		/**
		 * Caches the notification sent and returns the id of the record inserted / updated or false if failed to create.
		 *
		 * @since 0.1.0
		 *
		 * @param string $payload The notification formatted as a JSON string.
		 * @param bool   $success
		 * @param string $error
		 * @param int    $attempts Starting by zero because the entry is created before sending.
		 *
		 * @return int|false
		 */
		public function cache_notification( $payload, $success, $error = null, $attempts = 0 ) {
			global $wpdb;

			$this->cache_table_exists( true );
			$table_name = $this->get_cache_table_name();

			$id = isset( $payload['id'] ) && stripos( $payload['id'], '|' ) !== false ? (int) explode( '|', $payload['id'] )[0] : false;

			if ( $id !== false && $id > -1 ) {
				$results = $wpdb->get_results( // phpcs:ignore
					$wpdb->prepare(
						'SELECT * FROM %i WHERE `id` = %d',
						$table_name,
						$id
					)
				);

				$db_last_error = $wpdb->last_error;

				if ( ! empty( $db_last_error ) ) { // DB Error.
					Log_Service::write_log(
						'ERROR',
						sprintf(
							'An error occurred trying to retrieve a previously cached notification with id %d. [Error: %s]',
							$id,
							$db_last_error
						)
					);
				} elseif ( ! is_array( $results ) || count( $results ) === 0 ) { // Cannot find notification for $id.
					Log_Service::write_log(
						'ERROR',
						sprintf(
							'An error occurred trying to retrieve a previously cached notification with id %d. [Error: Not found]',
							$id
						)
					);
				} else { // Cached notification exists.
					$wpdb->update(  // phpcs:ignore
						$table_name,
						array(
							'notification_payload'      => wp_json_encode( $payload ),
							'notification_success'      => $success,
							'notification_error'        => $error,
							'notification_attempts'     => ( $results[0]->notification_attempts + 1 ),
							'notification_last_attempt' => self::time_zone_corrected_formatted_date_string(),
						),
						array( 'id' => $id )
					);

					$db_last_error = $wpdb->last_error;

					if ( ! empty( $db_last_error ) ) {
						Log_Service::write_log(
							'ERROR',
							sprintf(
								'An error occurred trying to cache the notification submission result. [Error: %s]',
								$db_last_error
							)
						);
					}

					return $id;
				}
			}

			$wpdb->insert( // phpcs:ignore
				$table_name,
				array(
					'notification_submitted'    => self::time_zone_corrected_formatted_date_string(),
					'notification_payload'      => wp_json_encode( $payload ),
					'notification_success'      => $success,
					'notification_error'        => $error,
					'notification_attempts'     => $attempts,
					'notification_last_attempt' => self::time_zone_corrected_formatted_date_string(),
				)
			);

			$db_last_error = $wpdb->last_error;

			if ( ! empty( $db_last_error ) ) {
				Log_Service::write_log(
					'ERROR',
					sprintf(
						'An error occurred trying to cache the notification submission result. [Error: %s]',
						$db_last_error
					)
				);

				return false;
			}

			// After every 100th cached item check if entries older than 10 needs deleting.
			if ( $wpdb->insert_id % 100 === 0 ) {
				self::cached_notifications_retention();
			}

			return $wpdb->insert_id;
		}

		/**
		 * Delete any cached notifications older than 10 days.
		 *
		 * @since 38.0
		 *
		 * @return void
		 */
		public static function cached_notifications_retention() {
			$_instance = self::get_instance();

			if ( ! $_instance->cache_table_exists( false ) ) {
				return;
			}

			global $wpdb;

			$table_name = $_instance->get_cache_table_name();

			$result        = $wpdb->query( // phpcs:ignore
				$wpdb->prepare(
					'DELETE FROM %i WHERE `notification_submitted` < CURDATE() - INTERVAL 10 DAY ORDER BY `notification_submitted` DESC',
					$table_name
				)
			);

			$db_last_error = $wpdb->last_error;

			if ( ! empty( $db_last_error ) ) {
				Log_Service::write_log( 'ERROR', sprintf( '%s -> Failed to delete items older than 10 from %s [error: %s]', __METHOD__, $table_name, $db_last_error ) );
			} else {
				Log_Service::write_log( 'DEBUG', sprintf( '%s -> Successfully deleted %d items older than 10 days from %s', __METHOD__, $result, $table_name ) );
			}
		}

		/**
		 * Helper method to create / update the custom Notifications DB table used for caching.
		 *
		 * @since   0.1.0
		 *
		 * @return  void
		 */
		private function create_cache_table() {
			global $wpdb;

			$table_name      = $this->get_cache_table_name();
			$charset_collate = $wpdb->get_charset_collate();
			$sql             = "CREATE TABLE IF NOT EXISTS $table_name (
                    id BIGINT AUTO_INCREMENT PRIMARY KEY,
                    notification_submitted DATETIME DEFAULT NULL,
                    notification_payload JSON,
                    notification_success BOOLEAN,
                    notification_error TEXT,
                    notification_attempts TINYINT DEFAULT 0,
                    notification_last_attempt DATETIME DEFAULT NULL
                    ) $charset_collate;";

			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
			dbDelta( $sql );

			global $wpdb;

			$db_last_error = $wpdb->last_error;

			if ( ! empty( $db_last_error ) ) {
				Log_Service::write_log(
					'ERROR',
					sprintf(
						'An error occurred trying to create the cache table. [Error: %s]',
						$db_last_error
					)
				);
				return;
			}

			Options_Service::add_update_option( 'notifications_sent_table_created', 1 );
		}

		/**
		 * Helper method to centrally provide the custom WordPress table name.
		 *
		 * @since 0.1.0
		 *
		 * @return string
		 */
		private function get_cache_table_name() {

			if ( ! empty( $this->p_cache_table_name ) ) {
				return $this->p_cache_table_name;
			}

			global $wpdb;

			return $wpdb->base_prefix . 'wpo365_notifications_sent';
		}

		/**
		 * Helper method to check whether the custom WordPress table exists.
		 *
		 * @since   0.1.0
		 *
		 * @return boolean
		 */
		private function cache_table_exists( $create = true ) {
			global $wpdb;

			$table_name            = $this->get_cache_table_name();
			$current_table_version = Options_Service::get_global_numeric_var( 'notifications_sent_table_created' );

			if ( $wpdb->get_var( // phpcs:ignore
				$wpdb->prepare(
					'SHOW TABLES LIKE %s',
					$table_name
				)
			) === $table_name && $current_table_version >= 1 ) {
				return true;
			}

			if ( $create ) {
				$this->create_cache_table();
				return true;
			}

			return false;
		}

		/**
		 * Helper to format a date using a helper in the WordPress Helpers class with fallback.
		 *
		 * @since 0.1.0
		 *
		 * @param int|null $time
		 * @return string
		 */
		private static function time_zone_corrected_formatted_date_string( $time = null ) {

			if ( empty( $time ) ) {
				$time = time();
			}

			return gmdate( 'Y-m-d H:i:s', $time );
		}
	}
}
