<?php

namespace Wpo\Sync;

use WP_Error;
use WP_User_Query;
use Wpo\Core\Extensions_Helpers;
use Wpo\Core\Wpmu_Helpers;
use Wpo\Services\Log_Service;
use Wpo\Services\Wp_To_Aad_Create_Update_Service;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

if ( ! class_exists( '\Wpo\Sync\Sync_Wp_To_Aad_Service' ) ) {

	class Sync_Wp_To_Aad_Service {

		public static function schedule( $job_id ) {
			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				return $job;
			}

			// Delete all scheduled jobs
			Sync_Helpers::get_scheduled_events( $job_id, true );

			try {

				if ( ! isset( $job['schedule'] ) || ! isset( $job['schedule']['scheduledOn'] ) || ! isset( $job['schedule']['scheduledAt'] ) ) {
					throw new \Exception( 'Too few arguments to schedule user synchronization job were supplied' );
				}

				$now                  = time();
				$day_of_the_week      = intval( gmdate( 'N', $now ) );
				$sel_day_of_the_week  = intval( intval( $job['schedule']['scheduledOn'] ) );
				$hours_of_the_day     = intval( gmdate( 'H', $now ) );
				$sel_hours_of_the_day = intval( intval( $job['schedule']['scheduledAt'] ) );
				$recurrence           = $sel_day_of_the_week < 7 ? 'wpo_weekly' : 'wpo_daily';
				$diff_days            = 0;
				$diff_hours           = 0;

				$seconds_in_an_hour = 60 * 60;
				$seconds_in_a_day   = 24 * $seconds_in_an_hour;
				$treshold           = 0;

				if ( $sel_day_of_the_week < 7 ) {

					if ( $sel_day_of_the_week > $day_of_the_week ) {
						$diff_days = $sel_day_of_the_week - $day_of_the_week;
					}

					if ( $sel_day_of_the_week < $day_of_the_week ) {
						$diff_days = ( 7 + $sel_day_of_the_week - $day_of_the_week );
					}
				}

				if ( $sel_hours_of_the_day > 0 && $sel_hours_of_the_day !== $hours_of_the_day ) {
					$diff_hours = $sel_hours_of_the_day - $hours_of_the_day;
				}

				if ( $diff_days === 0 && $sel_hours_of_the_day > 0 && $sel_hours_of_the_day < $hours_of_the_day ) {
					$diff_days = $sel_day_of_the_week === 7 ? 1 : 7;
				}

				$first_time = time() + ( $diff_days * $seconds_in_a_day ) + ( $diff_hours * $seconds_in_an_hour ) + $treshold;
			} catch ( \Exception $e ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> Atttempting to schedule user synchronization but failed to parse time and recurrence values' );
				return new \WP_Error( 'ScheduleParseError', __METHOD__ . ' -> Atttempting to schedule user synchronization but failed to parse time and recurrence values' );
			}

			$job['next'] = $first_time;
			$job['last'] = null;
			$updated     = Sync_Helpers::update_user_sync_job( $job );

			if ( is_wp_error( $updated ) ) {
				return $updated;
			}

			$result = wp_schedule_event( $first_time, $recurrence, 'wpo_sync_wp_to_aad_start', array( $job_id ) );

			if ( $result === false ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> Could not schedule a new cron job to synchronize users from WP to AAD for the first time at ' . $first_time . ' and then ' . $recurrence );
			}

			return true;
		}

		public static function stop( $job_id, $error = null ) {
			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				return $job;
			}

			if ( empty( $error ) ) {
				$error = new \WP_Error( 'UserSyncStopped', sprintf( '%s -> Administrator requested user synchronization to stop', __METHOD__ ) );
			}

			// Inform the user synchronization processor to stop
			if ( ! empty( $job['last'] ) ) {
				$job['last']['date']    = time();
				$job['last']['error']   = $error->get_error_message();
				$job['last']['errors']  = self::get_nok_user_count( $job );
				$job['last']['stopped'] = true;
				\update_option( 'wpo_sync_wp_to_aad_unscheduled', $job['last']['id'] );
			}

			// Delete all scheduled jobs
			Sync_Helpers::get_scheduled_events( $job_id, true );

			// Delete the schedule configuration
			$job['schedule'] = null;
			$job['next']     = null;

			// Update
			Sync_Helpers::update_user_sync_job( $job );

			// Delete memoized job info
			$job_info_name = sprintf( '%s_wpo365_sync_next', $job_id );
			Wpmu_Helpers::mu_delete_transient( $job_info_name );

			// Send email
			self::sync_completed_notification( $job_id );

			// Trigger hook
			do_action( 'wpo365/sync_wp_to_aad/error', $job );

			return true;
		}

		public static function sync_users( $job_id ) {
			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			// Try get job from config
			if ( is_wp_error( $job ) ) {
				Log_Service::write_log( 'ERROR', $job->get_error_message() );
				return;
			}

			// Validate job
			$validated = self::user_sync_job_is_valid( $job );

			if ( is_wp_error( $validated ) ) {
				Log_Service::write_log( 'ERROR', $validated->get_error_message() );
				return;
			}

			// Delete all scheduled tasks for this job
			if ( $job['trigger'] !== 'schedule' ) {
				Sync_Helpers::get_scheduled_events( $job_id, true, true );
			}

			// Check if there is an instance of this job still running
			$job_info_name = sprintf( '%s_wpo365_sync_next', $job_id );
			$job_info      = Wpmu_Helpers::mu_get_transient( $job_info_name );

			if ( ! empty( $job_info ) ) {
				Log_Service::write_log(
					'ERROR',
					sprintf(
						'%s -> Could not start a new instance of user synchronization job with ID %s because another instance did not yet finish (please stop the job in progress before starting a new one)',
						__METHOD__,
						$job_id
					)
				);
				return;
			}

			// Generate new job.last
			$job['last'] = array(
				'id'        => $job['id'] . '-' . uniqid(),
				'date'      => time(),
				'error'     => null,
				'processed' => 0,
				'total'     => -1,
			);

			$updated = Sync_Helpers::update_user_sync_job( $job );

			if ( is_wp_error( $updated ) ) {
				Log_Service::write_log( 'ERROR', $updated->get_error_message() );
			}

			// Start
			self::fetch_users( $job_id );
		}

		public static function fetch_users( $job_id, $start_row = 0 ) {
			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			$unscheduled_job_id = \get_option( 'wpo_sync_wp_to_aad_unscheduled' );

			// Check if administrator requested current job to be stopped
			if ( $unscheduled_job_id !== false ) {

				if ( ! empty( $job['last'] ) && ! empty( $job['last']['id'] ) && $unscheduled_job_id === $job['last']['id'] ) {
					delete_option( 'wpo_sync_wp_to_aad_unscheduled' );
					return new \WP_Error( 'UserSyncStopped', __METHOD__ . ' -> Administrator requested user synchronization to stop' );
				}
			}

			$job_info_name = sprintf( '%s_wpo365_sync_next', $job_id );

			if ( is_wp_error( $job ) ) {
				self::stop( $job_id, $job );
				return;
			}

			if ( empty( $start_row ) && $start_row !== 0 ) {
				$start_row = Wpmu_Helpers::mu_get_transient( $job_info_name );

				if ( $start_row === false ) {
					self::stop(
						$job_id,
						new \WP_Error(
							'TimeOutException',
							sprintf(
								'%s -> User sync job with ID %s cannot continue because the previous task was executed more than an hour ago',
								__METHOD__,
								$job_id
							)
						)
					);

					return;
				}
			}

			$page_size      = empty( $job['pageSize'] ) ? 25 : $job['pageSize'];
			$password_reset = false;

			if ( ! empty( $job['data'] ) && ! empty( $job['data']['passwordReset'] ) ) {
				$password_reset = filter_var( $job['data']['passwordReset'], FILTER_VALIDATE_BOOL );
			}

			// Trigger hook
			do_action( 'wpo365/sync_wp_to_aad/before', $job );

			$user_query_args = array(
				'count_total' => true,
				'fields'      => 'ID',
				'number'      => $page_size,
				'offset'      => $start_row,
				'meta_query'  => array( // phpcs:ignore
					'relation' => 'OR',
					array(
						'key'     => 'wpo365_active',
						'value'   => 'deactivated',
						'compare' => '!=',
					),
					array(
						'key'     => 'wpo365_active',
						'compare' => 'NOT EXISTS',
					),
				),
			);

			if ( ! empty( $job['excludedRoles'] ) ) {
				$user_query_args['role__not_in'] = $job['excludedRoles'];
			}

			$user_query = new WP_User_Query( $user_query_args );

			$users     = $user_query->get_results();
			$total     = $user_query->get_total();
			$sizeof    = count( $users );
			$processed = empty( $job['last']['processed'] ) ? 0 : $job['last']['processed'];

			// Process the current result
			if ( $sizeof > 0 ) {

				foreach ( $users as $wp_usr_id ) {
					$wp_usr            = get_user_by( 'ID', $wp_usr_id );
					$wp_usr->user_pass = ''; // The user didn't enter one so we set it to a random password

					$send_result = Wp_To_Aad_Create_Update_Service::send_user_to_azure( $wp_usr, $job['actionUpdateUser'] );
					++$processed;

					if ( is_wp_error( $send_result ) ) {
						Wp_To_Aad_Create_Update_Service::update_user_status( $wp_usr_id, $job['last']['id'], false, $send_result->get_error_message() );
						continue;
					}

					Wp_To_Aad_Create_Update_Service::update_user_status( $wp_usr_id, $job['last']['id'], true );
				}
			}

			// Update totals and processed for the current job for tracking
			if ( $job['last']['total'] === -1 ) {
				$job['last']['total'] = $total;
			}

			$job['last']['processed'] = $processed;
			Sync_Helpers::update_user_sync_job( $job );

			if ( $sizeof === $page_size ) {
				// Equal to $page_size so let's prepare another batch
				Wpmu_Helpers::mu_set_transient( $job_info_name, ( $start_row + $page_size ), 3600 );
				$random_arg = uniqid(); // Added to prevent WordPress from scheduling a similar event within 10 minutes
				$result     = wp_schedule_single_event( time() - 60, 'wpo_sync_wp_to_aad_next', array( $job_id, null, $random_arg ) );

				if ( $result === false ) {
					$error_message = sprintf( '%s -> Failed to schedule next event for hook "wpo_sync_wp_to_aad_next"', __METHOD__ );
					self::stop( $job_id, new WP_Error( 'EventScheduleException', $error_message ) );
					return;
				} else {
					Log_Service::write_log( 'DEBUG', sprintf( '%s -> Next event for hook "wpo_sync_wp_to_aad_next" has been scheduled', __METHOD__ ) );
				}
			} else {
				// Less than $page_size thus this must be the last batch of users
				Wpmu_Helpers::mu_delete_transient( $job_info_name );
				$next_cron_jobs = Sync_Helpers::get_scheduled_events( $job_id, false, true );

				if ( ! empty( $next_cron_jobs ) ) {

					foreach ( $next_cron_jobs as $cron_timestamp => $cron_job ) {

						if ( $cron_job['hook'] === 'wpo_sync_wp_to_aad_start' ) {
							$job['next'] = $cron_timestamp;
							break;
						}
					}
				}

				// Mark job as stopped
				$job['last']['stopped'] = true;
				$job['last']['date']    = time();
				$job['last']['errors']  = self::get_nok_user_count( $job );
				Sync_Helpers::update_user_sync_job( $job );

				// Trigger hook
				do_action( 'wpo365/sync_wp_to_aad/after', $job );

				// Send notification email
				self::sync_completed_notification( $job_id );

				Log_Service::write_log( 'DEBUG', sprintf( '%s -> User synchronization job with ID %s has finished', __METHOD__, $job_id ) );
			}
		}

		/**
		 * Validates the user sync job.
		 *
		 * @since   24.0
		 *
		 * @param   array $job    The user sync job to be validated.
		 * @return  array|WP_Error          The job or WP_Error if invalid.
		 */
		private static function user_sync_job_is_valid( $job ) {
			$error_fields = array();

			if ( empty( $job['name'] ) ) {
				$error_fields[] = 'name is empty';
			}

			if ( $job['trigger'] === 'schedule' && empty( $job['schedule'] ) ) {
				$error_fields[] = 'job schedule is empty';
			}

			if ( $job['sendLog'] && empty( $job['sendLogTo'] ) ) {
				$error_fields[] = 'mail recipient to send log is empty';
			}

			if ( ! empty( $errors ) ) {
				return new \WP_Error( 'JobValidationException', sprintf( '%s -> WP to AAD User sync job is invalid: %s', __METHOD__, join( ', ', $error_fields ) ) );
			}

			return $job;
		}

		/**
		 * Returns the number of errors encountered during the last execution of the wp-to-aad sync job.
		 *
		 * @since   24.0
		 *
		 * @param array $job Corresponds to $job['last']['id'].
		 * @return int Number of users with status NOK for the "last" run of the resverse sync job.
		 */
		private static function get_nok_user_count( $job ) {
			$job_last_id = $job['last']['id'];

			$user_query_args = array(
				'count_total' => true,
				'fields'      => 'ID',
				'meta_query'  => array( // phpcs:ignore
					'relation' => 'AND',
					array(
						'key'     => 'wpo_sync_wp_to_aad_status',
						'value'   => '"status":"NOK"',
						'compare' => 'LIKE',
					),
					array(
						'key'     => 'wpo_sync_wp_to_aad_status',
						'value'   => sprintf( '"job_id":"%s"', $job_last_id ),
						'compare' => 'LIKE',
					),
				),
			);

			if ( ! empty( $job['excludedRoles'] ) ) {
				$user_query_args['role__not_in'] = $job['excludedRoles'];
			}

			$user_query = new WP_User_Query( $user_query_args );
			return $user_query->get_total();
		}

		/**
		 * Sends the admin of the site an email to inform that user synchronization has completed.
		 *
		 * @since 24.0
		 *
		 * @return void
		 */
		private static function sync_completed_notification( $job_id ) {
			$job = Sync_Helpers::get_user_sync_job_by_id( $job_id );

			if ( is_wp_error( $job ) ) {
				Log_Service::write_log( 'WARN', __METHOD__ . ' -> Could not find the user synchronization job whilst trying to send user-synchronization-completed email' );
				return;
			}

			if ( empty( $job['sendLog'] ) || empty( $job['sendLogTo'] ) ) {
				Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Sending of a user-synchronization-completed email is not configured' );
				return;
			}

			$blogname = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );
			$errors   = ! empty( $job['last']['errors'] ) ? absint( $job['last']['errors'] ) : 0;
			$error    = $job['last']['error'];

			// Bail out now if the job was successful and admin only wishes to send a notification on failure
			if ( ! empty( $job['sendLogOnError'] ) && empty( $error ) && $errors === 0 ) {
				return;
			}

			$view_users_tmpl = sprintf( "You can view a list of users affected by this job by visiting %s?wpo365_jobid=%s.\r\n\r\n", get_admin_url( null, 'users.php' ), $job['last']['id'] );

			if ( ! empty( $error ) ) {
				$completed_tmpl   = 'FAILED to complete successfully';
				$view_users_tmpl  = '';
				$view_errors_tmpl = sprintf( "The error encountered is:\r\n\r\n%s\r\n\r\n", $error );
			} elseif ( $errors > 0 ) {
				$completed_tmpl   = sprintf( 'COMPLETED with %s errors', $errors );
				$view_errors_tmpl = sprintf( "You can view a list of users for which synchronization failed by visiting %s?wpo365_filter=wpo365_send_to_azure_nok&wpo365_jobid=%s.\r\n\r\n", get_admin_url( null, 'users.php' ), $job['last']['id'] );
			} else {
				$completed_tmpl   = 'COMPLETED successfully';
				$view_errors_tmpl = '';
			}

			$subject = sprintf( 'WPO365 | User Synchronization WordPress -> Azure AD (B2C) %s', $completed_tmpl );

			include Extensions_Helpers::get_active_extension_dir( array( 'wpo365-customers/wpo365-customers.php' ) ) . '/templates/wp-to-aad-sync-mail-text.php';

			$message = sprintf( $body, $job['name'], $blogname, $job['last']['processed'], $completed_tmpl, $view_users_tmpl, $view_errors_tmpl );

			$sync_completed_email_admin = array(
				'to'      => $job['sendLogTo'],
				'subject' => $subject,
				'message' => $message,
				'headers' => array( 'Content-Type: text/plain' ),
			);

			@wp_mail( // phpcs:ignore
				$sync_completed_email_admin['to'],
				$sync_completed_email_admin['subject'],
				$sync_completed_email_admin['message'],
				$sync_completed_email_admin['headers']
			);
		}
	}
}
