<?php

namespace Wpo\Services;

use Wpo\Core\Extensions_Helpers;
use Wpo\Services\Options_Service;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

if ( ! class_exists( '\Wpo\Services\Mail_Notifications_Service' ) ) {

	class Mail_Notifications_Service {

		/**
		 * Custom filter that hooks up with the wp_new_user_notification_email filter.
		 *
		 * @since 7.5
		 *
		 * @param   array   $email_info New user registration email info: to, subject, message and headers.
		 * @param   WP_user $wp_user The WP_User that was just registered as a new user.
		 * @param   string  $blogname The current blog's title.
		 *
		 * @return array The (possibly updated) new user registration mail info.
		 */
		public static function new_user_notification_email( $email_info, $wp_user, $blogname ) {
			// Do nothing if custom template is not explicitely configured
			if ( Options_Service::get_global_boolean_var( 'new_usr_send_mail_custom' ) === false ) {
				return $email_info;
			}

			// e.g. $subject_template = '[__##BLOG_NAME##__] New user registration';
			$subject_template = Options_Service::get_global_string_var( 'new_usr_mail_subject' );
			$subject          = self::replace_template_tags( $subject_template, $wp_user, $blogname );

			// e.g. $title_template = 'USER REGISTRATION';
			$title_template = Options_Service::get_global_string_var( 'new_usr_mail_title' );
			$title          = self::replace_template_tags( $title_template, $wp_user, $blogname );

			// e.g. $subtitle_template = ''; but currently not in use
			$subtitle = '';

			// e.g. $salutation_template = 'Dear __##USER_DISPLAY_NAME##__';
			$salutation_template = Options_Service::get_global_string_var( 'new_usr_mail_salutation' );
			$salutation          = self::replace_template_tags( $salutation_template, $wp_user, $blogname );

			// e.g. $body_template = 'The website <strong><a href="__##BLOG_URL##__">__##BLOG_NAME##__</a></strong> has been shared with you.'
			// . 'You can now sign in using your Azure AD or Azure AD B2C account <strong>__##USER_LOGIN_NAME##__</strong>.';
			$body_template = Options_Service::get_global_string_var( 'new_usr_mail_body' );
			$body          = self::replace_template_tags( $body_template, $wp_user, $blogname );

			// e.g. $footer_template = '<strong>WPO365 - Connecting WordPress and Microsoft Entra | AAD | 365</strong><br/>'
			// . 'Zurich, Switzerland<br/><br/>'
			// . 'Stay informed on Twitter <a href="https://twitter.com/WPO365" target="_blank">https://twitter.com/WPO365</a><br/>';
			$footer_template = Options_Service::get_global_string_var( 'new_usr_mail_footer' );
			$footer          = self::replace_template_tags( $footer_template, $wp_user, $blogname );

			ob_start();
			include Extensions_Helpers::get_active_extension_dir( array( 'wpo365-login-professional/wpo365-login.php', 'wpo365-login-premium/wpo365-login.php', 'wpo365-sync-5y/wpo365-sync-5y.php', 'wpo365-login-intranet/wpo365-login.php', 'wpo365-intranet-5y/wpo365-intranet-5y.php', 'wpo365-customers/wpo365-customers.php', 'wpo365-integrate/wpo365-integrate.php', 'wpo365-pro/wpo365-pro.php' ) ) . '/templates/new-user.php';
			$message = ob_get_clean();

			$email_info['message'] = $message;
			$email_info['subject'] = $subject;
			$email_info['headers'] = "MIME-Version: 1.0\r\nContent-Type: text/html; charset=ISO-8859-1\r\n";

			return $email_info;
		}

		/**
		 * Used to override the pluggable wp_password_change_notification function with an empty
		 * function to prevent WordPress from sending this notification to admins.
		 *
		 * @return void
		 */
		public static function password_change_notification() {
			if ( ! function_exists( 'wp_password_change_notification' ) ) {
				function wp_password_change_notification() {
				}
			}
		}

		/**
		 * Replaces various new user email template tags with their corresponding values.
		 *
		 * @since 7.5
		 *
		 * @param string  $template  The template with the tags to be replaced.
		 * @param WP_User $wp_user The WP_User object used to obtain diplay and login name plus email.
		 * @param string  $blogname The blog's title.
		 *
		 * @return string Template with tags replaced with their corresponding instance values.
		 */
		private static function replace_template_tags( $template, $wp_user, $blogname ) {
			$template = str_replace( '__##USER_DISPLAY_NAME##__', $wp_user->display_name, $template );
			$template = str_replace( '__##USER_LOGIN_NAME##__', $wp_user->user_login, $template );
			$template = str_replace( '__##USER_EMAIL##__', $wp_user->user_email, $template );
			$template = str_replace( '__##BLOG_NAME##__', $blogname, $template );
			$template = str_replace( '__##BLOG_URL##__', network_site_url(), $template );
			return $template;
		}
	}
}
