<?php

namespace Wpo\Sync;

use Wpo\Core\Wpmu_Helpers;
use Wpo\Services\Log_Service;
use Wpo\Services\Options_Service;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

if ( ! class_exists( '\Wpo\Sync\Sync_Db' ) ) {

	class Sync_Db {


		/**
		 * Helper method to create / update the custom WordPress table.
		 *
		 * @since 3.0
		 *
		 * @return void
		 */
		public static function create_user_sync_table() {

			global $wpdb;

			$table_name = self::get_user_sync_table_name();

			$charset_collate = $wpdb->get_charset_collate();

			$sql = "CREATE TABLE IF NOT EXISTS $table_name (
                    wp_id int(11) DEFAULT 0 NOT NULL,
                    upn varchar(100) NOT NULL,
                    first_name varchar(100) DEFAULT '' NOT NULL,
                    last_name varchar(100) DEFAULT '' NOT NULL,
                    full_name varchar(100) DEFAULT '' NOT NULL,
                    email varchar(100) DEFAULT '' NOT NULL,
                    sync_job_id varchar(50) NOT NULL,
                    name varchar(100) DEFAULT '' NOT NULL,
                    sync_job_status varchar(11) NOT NULL,
                    record_type varchar(20) NOT NULL,
                    notes varchar(500) DEFAULT '',
                    KEY record_type (record_type),
                    KEY sync_job_id (sync_job_id),
                    KEY wp_id (wp_id),
                    KEY sync_job_status (sync_job_status),
                    PRIMARY KEY  (upn, sync_job_id)
                    ) $charset_collate;";

			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
			dbDelta( $sql );
		}

		/**
		 * Helper method to centrally provide the custom WordPress table name.
		 *
		 * @since 3.0
		 *
		 * @return string
		 */
		public static function get_user_sync_table_name() {
			global $wpdb;

			if ( Options_Service::mu_use_subsite_options() && ! Wpmu_Helpers::mu_is_network_admin() ) {
				return $wpdb->prefix . 'wpo365_user_sync';
			}

			return $wpdb->base_prefix . 'wpo365_user_sync';
		}

		/**
		 * Helper method to check whether the custom WordPress table exists.
		 *
		 * @since   3.0
		 *
		 * @return boolean
		 */
		public static function user_sync_table_exists( $force = false ) {

			global $wpdb;

			$table_name = self::get_user_sync_table_name();
			$exists     = $wpdb->get_var( // phpcs:ignore
				$wpdb->prepare(
					'SHOW TABLES LIKE %s',
					$table_name
				)
			) === $table_name;

			if ( ! $exists && ! $force ) {
				return false;
			}

			/**
			 * @since   15.0    Updated to add a notes column and change PRIMARY KEY.
			 */

			if ( $exists ) {
				$has_notes = $wpdb->get_results( // phpcs:ignore
					$wpdb->prepare(
						'SELECT `COLUMN_NAME` FROM INFORMATION_SCHEMA.COLUMNS WHERE `table_name` = %s AND `column_name` = %s',
						$table_name,
						'notes'
					)
				);
				$has_pk    = $wpdb->get_results( // phpcs:ignore
					$wpdb->prepare(
						'SHOW INDEX FROM %i WHERE `Key_name` = %s',
						$table_name,
						'PRIMARY'
					)
				);

				if ( empty( $has_notes ) || empty( $has_pk ) || count( $has_pk ) !== 2 ) {
					$wpdb->query( // phpcs:ignore
						$wpdb->prepare(
							'DROP TABLE IF EXISTS %i', // phpcs:ignore
							$table_name
						)
					);
					$exists = false;
					Log_Service::write_log( 'WARN', __METHOD__ . ' -> Deleted existing table for WPO365 user synchronization [' . $table_name . ']' );
				}
			}

			if ( ! $exists ) {
				self::create_user_sync_table();
				Log_Service::write_log( 'WARN', __METHOD__ . ' -> Created new table for WPO365 user synchronization [' . $table_name . ']' );
			}

			return true;
		}

		/**
		 * Helper method to truncate the table and remove the job id and last run time.
		 *
		 * @since 15.0
		 *
		 * @return void
		 */
		public static function delete_job_data( $job_id ) { // phpcs:ignore

			global $wpdb;

			$table_name = self::get_user_sync_table_name();
			$wpdb->query( // phpcs:ignore
				$wpdb->prepare(
					'DELETE FROM %i WHERE sync_job_id LIKE %s',
					$table_name,
					'job-id-%'
				)
			);
		}
	}
}
