<?php

namespace Wpo\Services;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

use WP_Error;
use Wpo\Core\Permissions_Helpers;
use Wpo\Core\Url_Helpers;
use Wpo\Core\User;
use Wpo\Core\WordPress_Helpers;
use Wpo\Services\Authentication_Service;
use Wpo\Services\Log_Service;
use Wpo\Services\Options_Service;

if ( ! class_exists( '\Wpo\Services\Wp_To_Aad_Create_Update_Service' ) ) {

	class Wp_To_Aad_Create_Update_Service {

		/**
		 * When a new WordPress user is created or an existing one is updated.
		 *
		 * @since   24.0
		 *
		 * @param   array $user_data
		 * @param   bool  $update
		 * @param   int   $wp_usr_id
		 * @param   array $raw
		 *
		 * @return  array   Array with (possibly updated) user data
		 */
		public static function handle_user_registering( $user_data, $update, $wp_usr_id, $raw ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			// Don't interfer if admin didn't configure WP -> AAD for inserts
			if ( ( ! $update || empty( $wp_usr_id ) ) && ! Options_Service::get_global_boolean_var( 'wp_to_aad_create_users' ) ) {
				return $user_data;
			}

			// Don't interfer if admin didn't configure WP -> AAD for updates
			if ( ( $update || ! empty( $wp_usr_id ) ) && ! Options_Service::get_global_boolean_var( 'wp_to_aad_update_users' ) ) {
				return $user_data;
			}

			// For B2C and CIAM each user must posess a valid email address
			if ( empty( $user_data['user_email'] ) || ! filter_var( $user_data['user_email'], FILTER_VALIDATE_EMAIL ) ) {
				Log_Service::write_log( 'ERROR', sprintf( '%s -> Cannot send user to Azure because the email address "%s" not found or invalid', __METHOD__, $user_data['user_email'] ) );
				return $user_data;
			}

			// Check if the user's role is excluded from sending to Azure
			if ( ! empty( $raw['role'] ) ) {
				$wp_to_aad_excluded_roles = Options_Service::get_global_list_var( 'wp_to_aad_excluded_roles' );

				foreach ( $wp_to_aad_excluded_roles as $wp_to_aad_excluded_role ) {

					if ( strcasecmp( $wp_to_aad_excluded_role, $raw['role'] ) === 0 ) {
						Log_Service::write_log( 'DEBUG', sprintf( '%s -> Cannot sent user to Azure because WP user with email %s has role %s which has been excluded', __METHOD__, $user_data['user_email'], $wp_to_aad_excluded_role ) );
						return $user_data;
					}
				}
			}

			// We may need to cache some WP user details that can only be updated after the WP user exists (see handle_user_registered)
			$request_service = Request_Service::get_instance();
			$request         = $request_service->get_request( $GLOBALS['WPO_CONFIG']['request_id'] );
			$usermeta_cache  = $request->get_item( 'usermeta_cache' );

			if ( empty( $usermeta_cache ) ) {
				$usermeta_cache = array( $user_data['user_email'] => array() );
			}

			if ( ! isset( $usermeta_cache[ $user_data['user_email'] ] ) ) {
				$usermeta_cache[ $user_data['user_email'] ] = array();
			}

			if ( ! empty( $usermeta_cache[ $user_data['user_email'] ]['update_user_status'] ) ) {
				Log_Service::write_log( 'WARN', sprintf( '%s -> An attempt to send user %s to Azure has been aborted because an earlier attempt has failed', __METHOD__, $user_data['user_email'] ) );
				return $user_data;
			}

			// Clone $user_data because we need to add some properties when sending it further down stream
			$aad_user_data = array_map(
				function ( $element ) {
					return $element;
				},
				$user_data
			);

			// "Move" some props from $raw to $aad_user_data if found
			if ( ! empty( $raw['first_name'] ) ) {
				$aad_user_data['first_name'] = $raw['first_name'];
			}

			if ( ! empty( $raw['last_name'] ) ) {
				$aad_user_data['last_name'] = $raw['last_name'];
			}

			// If the user has entered a password then lets use that (otherwise set empty)
			if ( ! empty( $raw['user_pass'] ) && $user_data['user_pass'] !== $raw['user_pass'] ) {
				$aad_user_data['user_pass'] = $raw['user_pass'];
			} else {
				$aad_user_data['user_pass'] = '';
			}

			// Add the ID
			$aad_user_data['ID'] = $wp_usr_id ?? 0;

			// Send user to Azure
			$send_user_to_azure_result = self::send_user_to_azure( (object) $aad_user_data, true );

			if ( is_wp_error( $send_user_to_azure_result ) ) {
				Log_Service::write_log(
					'ERROR',
					sprintf(
						'%s -> Failed to create / update user with email %s in Azure AD (B2C) [Error: %s]',
						__METHOD__,
						$user_data['user_email'],
						$send_user_to_azure_result->get_error_message()
					)
				);

				if ( ! empty( $wp_usr_id ) ) {
					self::update_user_status( $wp_usr_id, '', false, $send_user_to_azure_result->get_error_message() );

					if ( Options_Service::get_global_boolean_var( 'wp_to_aad_exit_on_failure' ) ) {
						return null; // Not enough data to create the user
					}
				} else {

					if ( Options_Service::get_global_boolean_var( 'wp_to_aad_exit_on_failure' ) ) {

						// Assumption: Administrator at work
						if ( is_admin() ) {
							return null; // Not enough data to create the user
						}

						// Assumption: The user is interactively registering him / her self
						Authentication_Service::goodbye( Error_Service::CHECK_LOG, false );
					}

					$usermeta_cache[ $user_data['user_email'] ]['update_user_status'] = $send_user_to_azure_result->get_error_message();
					$request->set_item( 'usermeta_cache', $usermeta_cache );
				}

				return $user_data;
			}

			Log_Service::write_log(
				'DEBUG',
				sprintf(
					'%s -> Successfully created / updated user with email %s in Azure AD (B2C)',
					__METHOD__,
					$user_data['user_email']
				)
			);

			// Update some WP user meta for OID, UPN and Send-to-Azure status
			$aad_object_id       = ! empty( $send_user_to_azure_result->oid ) ? $send_user_to_azure_result->oid : '';
			$user_principal_name = ! empty( $send_user_to_azure_result->upn ) ? $send_user_to_azure_result->upn : '';
			$tenant_id           = ! empty( $send_user_to_azure_result->tid ) ? $send_user_to_azure_result->tid : '';

			if ( empty( $wp_usr_id ) ) { // New users
				$usermeta_cache[ $user_data['user_email'] ]['aadObjectId']       = $aad_object_id;
				$usermeta_cache[ $user_data['user_email'] ]['userPrincipalName'] = $user_principal_name;
				$usermeta_cache[ $user_data['user_email'] ]['aadTenantId']       = $tenant_id;

				// The admin may have chosen to reset a user's password so the user cannot sign in / does not know the own password
				if ( Options_Service::get_global_boolean_var( 'wp_to_aad_password_reset' ) ) {
					$password_length = Options_Service::get_global_numeric_var( 'password_length' );

					if ( empty( $password_length ) || $password_length < 16 ) {
						$password_length = 16;
					}

					if ( method_exists( '\Wpo\Core\Permissions_Helpers', 'generate_password' ) ) {
						$user_password = Permissions_Helpers::generate_password( $password_length );
					} else {
						$user_password = wp_generate_password( $password_length, true, false );
					}

					$usermeta_cache[ $user_data['user_email'] ]['wp_pass'] = $user_password;
					$user_password          = wp_hash_password( $user_password );
					$user_data['user_pass'] = $user_password;
				}

				// The admin may have chosen to customize the user's WP username
				$custom_username = '';

				if ( Options_Service::get_global_string_var( 'user_name_preference' ) === 'custom' ) {
					$username_claim = Options_Service::get_global_string_var( 'user_name_claim' );

					if ( ! empty( $username_claim ) && property_exists( $send_user_to_azure_result, $username_claim ) ) {
						$custom_username = $send_user_to_azure_result->$username_claim;
					}
				} elseif ( Options_Service::get_global_boolean_var( 'use_short_login_name' ) || Options_Service::get_global_string_var( 'user_name_preference' ) === 'short' ) {
					$custom_username = \stristr( $user_data['user_email'], '@', true );
				}

				$user_login = ! empty( $custom_username ) ? $custom_username : $user_data['user_login'];

				/**
				 * @since 33.2  Allow developers to filter the user_login.
				 */

				$user_login = apply_filters( 'wpo365/user/user_login', $user_login );

				$user_data['user_login']                                  = $user_login;
				$usermeta_cache[ $user_data['user_email'] ]['user_login'] = $user_login;
			} else { // Existing users
				self::update_user_status( $wp_usr_id, '', true, '' );
			}

			$request->set_item( 'usermeta_cache', $usermeta_cache );
			return $user_data;
		}

		/**
		 * Updates a new WordPress user that was created using the handle_user_registering
		 * handler with some Azure details (oid, upn) and update status.
		 *
		 * @param mixed $wp_usr_id
		 * @return void
		 */
		public static function handle_user_registered( $wp_usr_id ) {
			$request_service = Request_Service::get_instance();
			$request         = $request_service->get_request( $GLOBALS['WPO_CONFIG']['request_id'] );
			$usermeta_cache  = $request->get_item( 'usermeta_cache' );

			if ( empty( $usermeta_cache ) ) {
				return;
			}

			$wp_usr = get_user_by( 'ID', $wp_usr_id );

			if ( ! empty( $wp_usr ) && ! empty( $wp_usr->user_email ) && ! empty( $usermeta_cache[ $wp_usr->user_email ] ) ) {
				$usermeta = $usermeta_cache[ $wp_usr->user_email ];

				if ( ! empty( $usermeta['aadObjectId'] ) ) {
					update_user_meta( $wp_usr_id, 'aadObjectId', $usermeta['aadObjectId'] );
				}

				if ( ! empty( $usermeta['userPrincipalName'] ) ) {
					update_user_meta( $wp_usr_id, 'userPrincipalName', $usermeta['userPrincipalName'] );
				}

				if ( ! empty( $usermeta['aadTenantId'] ) ) {
					update_user_meta( $wp_usr_id, 'aadTenantId', $usermeta['aadTenantId'] );
				}

				$user_status_message = ! empty( $usermeta['update_user_status'] ) ? $usermeta['update_user_status'] : '';
				self::update_user_status( $wp_usr_id, '', empty( $user_status_message ), $user_status_message );
			}
		}

		/**
		 * Given the WordPress user ID, synchronizes that user to Azure AD (B2C) either by creating the user or by updating an existing user.
		 *
		 * @param   object $user_data
		 * @param   bool   $update
		 *
		 * @return  User|bool|WP_Error   Returns true if the user was sent to Azure AD (B2C) successfully and otherwise the WP_Error.
		 */
		public static function send_user_to_azure( $user_data, $update = true ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			// Currently WPO365 only support sending users to Azure AD B2C / For Customers
			if ( ! Options_Service::get_global_boolean_var( 'use_b2c' ) && ! Options_Service::get_global_boolean_var( 'use_ciam' ) ) {
				return new WP_Error( 'ConfigurationException', sprintf( '%s -> Please configure Azure AD B2C / For Customers based SSO', __METHOD__ ) );
			}

			// Check if the user already exists in Azure
			$email             = ! empty( $user_data->user_email ) ? $user_data->user_email : '';
			$oid               = ! empty( $user_data->ID ) ? get_user_meta( $user_data->ID, 'aadObjectId', true ) : '';
			$upn               = ! empty( $user_data->ID ) ? get_user_meta( $user_data->ID, 'userPrincipalName', true ) : '';
			$existing_aad_user = self::get_aad_user( $email, $oid, $upn );

			if ( is_wp_error( $existing_aad_user ) ) {
				return $existing_aad_user;
			}

			// Update of an existing user is not needed
			if ( ! empty( $existing_aad_user ) && ! $update ) {
				return $existing_aad_user;
			}

			$email_changed    = $existing_aad_user !== false && ! empty( $email ) && strcasecmp( $existing_aad_user->email, $email ) !== 0;
			$updated_aad_user = empty( $existing_aad_user ) ? self::create_user_in_azure( $user_data ) : self::update_user_in_azure( $existing_aad_user->oid, $user_data, $email_changed );

			return $updated_aad_user;
		}

		/**
		 * Checks by email (or optionally by Azure AD Object ID or User Principal Name) whether the user already exists in Azure AD.
		 *
		 * @since   24.0
		 *
		 * @param   string $email
		 * @param   string $oid
		 * @param   string $upn
		 *
		 * @return  User|WP_Error|bool  Returns WPO365's internal representation of a user, false if no user was found and otherwise the WP_Error.
		 */
		public static function get_aad_user( $email, $oid = null, $upn = null ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			$_email = $email;
			$_oid   = $oid;
			$_upn   = $upn;

			if ( empty( $email ) ) {
				$_email = '22976358-cffc-41cf-9997-4c8e050458fa@example.com'; // Just some random email since Graph does not like empty values
			}

			if ( empty( $upn ) ) {
				$_upn = '22976358-cffc-41cf-9997-4c8e050458fa@example.com'; // Just some random username since Graph does not like empty values
			}

			if ( empty( $oid ) ) {
				$_oid = 'e373001a-603d-4b05-b969-9277dc2b08c0'; // Just some random guid since Graph does not like empty values
			}

			$headers      = array(
				'Accept: application/json;odata.metadata=minimal',
				'Content-Type: application/json',
			);
			$query        = sprintf( '/users?$filter=mail eq \'%s\' OR id eq \'%s\' OR userPrincipalName eq \'%s\'', $_email, $_oid, rawurlencode( $_upn ) );
			$fetch_result = Graph_Service::fetch( $query, 'GET', false, $headers, false, false, '', 'User.ReadWrite.All' ); // We use anyway readwrite already

			if ( is_wp_error( $fetch_result ) ) {
				return $fetch_result;
			}

			$graph_error = '';
			$on_error    = function ( $log_message ) use ( &$graph_error ) {
				$graph_error = $log_message;
			};

			if ( ! Graph_Service::is_fetch_result_ok( $fetch_result, 'to look up a user in Azure AD using email, oid or upn', 'WARN', $on_error ) ) {
				return new WP_Error( 'GraphFetchException', sprintf( '%s -> %s', __METHOD__, $graph_error ) );
			}

			if ( is_array( $fetch_result['payload'] ) && isset( $fetch_result['payload']['value'] ) && is_array( $fetch_result['payload']['value'] ) ) {

				if ( count( $fetch_result['payload']['value'] ) === 1 ) {
					return User_Service::user_from_graph_user( $fetch_result['payload']['value'][0] );
				}

				if ( count( $fetch_result['payload']['value'] ) > 1 ) {
					return new \WP_Error( 'NonUniqueResultException', sprintf( '%s -> More than one user found. The search for an existing user was performed with the following parameter: email %s, oid %s and upn %s', __METHOD__, $email, $oid, $upn ) );
				}
			}

			return self::get_aad_user_by_issuer_assigned_id( $email );
		}

		/**
		 * A second query to attempt to search user by issuer assigned id. Adding this predicate to the default search query with OR
		 * results in "Complex query on property identities is not supported." error.
		 *
		 * @param mixed $email
		 * @return array|WP_Error|false|User
		 */
		public static function get_aad_user_by_issuer_assigned_id( $email ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			if ( empty( $email ) ) {
				return false;
			}

			$b2c_domain_name = Options_Service::get_aad_option( 'b2c_domain_name' );

			$headers      = array(
				'Accept: application/json;odata.metadata=minimal',
				'Content-Type: application/json',
			);
			$query        = sprintf( '/users?$filter=identities/any(x:x/issuer eq \'%s\' and x/issuerAssignedId eq \'%s\')', sprintf( '%s.onmicrosoft.com', $b2c_domain_name ), $email );
			$fetch_result = Graph_Service::fetch( $query, 'GET', false, $headers, false, false, '', 'User.ReadWrite.All' ); // We use anyway readwrite already

			if ( is_wp_error( $fetch_result ) ) {
				return $fetch_result;
			}

			$graph_error = '';
			$on_error    = function ( $log_message ) use ( &$graph_error ) {
				$graph_error = $log_message;
			};

			if ( ! Graph_Service::is_fetch_result_ok( $fetch_result, 'to look up a user in Azure AD using email, oid or upn', 'WARN', $on_error ) ) {
				return new WP_Error( 'GraphFetchException', sprintf( '%s -> %s', __METHOD__, $graph_error ) );
			}

			if ( is_array( $fetch_result['payload'] ) && isset( $fetch_result['payload']['value'] ) && is_array( $fetch_result['payload']['value'] ) ) {

				if ( count( $fetch_result['payload']['value'] ) === 1 ) {
					return User_Service::user_from_graph_user( $fetch_result['payload']['value'][0] );
				}

				if ( count( $fetch_result['payload']['value'] ) > 1 ) {
					return new \WP_Error( 'NonUniqueResultException', sprintf( '%s -> More than one user found. The search for an existing user was performed with the following parameter: email %s', __METHOD__, $email ) );
				}
			}

			return false;
		}

		/**
		 * Creates a new user in Azure AD (B2C).
		 *
		 * @param   WP_User $wp_user
		 * @return  User|WP_Error   WPO365 internal user representation when the new user was created successfully and otherwise the WP_Error.
		 */
		public static function create_user_in_azure( $wp_user ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			$email = ! empty( $wp_user->user_email ) ? $wp_user->user_email : '';

			if ( empty( $email ) ) {
				return new \WP_Error( 'MissingUserPropertyException', sprintf( '%s -> (WordPress) User [login: %s] does not have a valid email address', __METHOD__, $wp_user->user_login ) );
			}

			$b2c_domain_name = Options_Service::get_aad_option( 'b2c_domain_name' );

			if ( empty( $b2c_domain_name ) ) {
				return new \WP_Error( 'ConfigurationException', sprintf( '%s -> Your Azure AD B2C integration configuration is not complete (B2C domain name is null or empty)', __METHOD__ ) );
			}

			$b2c_local_user = self::get_user_attributes( $wp_user );

			if ( is_wp_error( $b2c_local_user ) ) {
				return $b2c_local_user;
			}

			$authentication_method = Options_Service::get_global_string_var( 'wp_to_aad_identity_provider' );

			if ( strcasecmp( $authentication_method, 'emailOtp' ) === 0 ) {
				$b2c_local_user['identities'] = array(
					array(
						'signInType'       => 'federated',
						'issuer'           => 'mail',
						'issuerAssignedId' => $email,
					),
				);
			} else { // Fallback: Email with password
				$b2c_local_user['identities'] = array(
					array(
						'signInType'       => 'emailAddress',
						'issuer'           => sprintf( '%s.onmicrosoft.com', $b2c_domain_name ),
						'issuerAssignedId' => $email,
					),
				);
			}

			$password_length = Options_Service::get_global_numeric_var( 'password_length' );

			if ( empty( $password_length ) || $password_length < 16 ) {
				$password_length = 16;
			}

			$force_change_pwd = empty( $wp_user->user_pass ) || Options_Service::get_global_boolean_var( 'wp_to_aad_force_change_password_next_sign_in' );

			if ( ! empty( $wp_user->user_pass ) && ! $force_change_pwd ) {
				$password = $wp_user->user_pass;
			} elseif ( method_exists( '\Wpo\Core\Permissions_Helpers', 'generate_password' ) ) {
				$password = Permissions_Helpers::generate_password( $password_length );
			} else {
				$password = wp_generate_password( $password_length, true, false );
			}

			$b2c_local_user['passwordProfile']  = array(
				'password'                      => $password,
				'forceChangePasswordNextSignIn' => $force_change_pwd,
			);
			$b2c_local_user['passwordPolicies'] = 'DisablePasswordExpiration';

			$body_as_json = wp_json_encode( $b2c_local_user );
			$headers      = array(
				'Accept: application/json;odata.metadata=minimal',
				'Content-Type: application/json',
			);

			$fetch_result = Graph_Service::fetch( '/users', 'POST', false, $headers, false, false, $body_as_json, 'User.ReadWrite.All' );

			if ( is_wp_error( $fetch_result ) ) {
				return $fetch_result;
			}

			$graph_error = '';
			$on_error    = function ( $log_message ) use ( &$graph_error ) {
				$graph_error = $log_message;
			};

			if ( ! Graph_Service::is_fetch_result_ok( $fetch_result, 'to create a user in Azure AD (B2C)', 'WARN', $on_error ) ) {
				return new WP_Error( 'GraphFetchException', sprintf( '%s -> %s', __METHOD__, $graph_error ) );
			}

			$graph_user = null;

			if ( is_array( $fetch_result['payload'] ) && count( $fetch_result['payload'] ) === 1 ) {
				$graph_user = User_Service::user_from_graph_user( $fetch_result['payload'][0] );
			} else {
				$graph_user = User_Service::user_from_graph_user( $fetch_result['payload'] );
			}

			if ( ! empty( $wp_user->ID ) ) {
				// Update some WP user meta for OID, UPN and Send-to-Azure status
				$aad_object_id       = ! empty( $graph_user->oid ) ? $graph_user->oid : '';
				$user_principal_name = ! empty( $graph_user->upn ) ? $graph_user->upn : '';
				$tenant_id           = ! empty( $graph_user->tid ) ? $graph_user->tid : '';

				update_user_meta( $wp_user->ID, 'aadObjectId', $aad_object_id );
				update_user_meta( $wp_user->ID, 'userPrincipalName', $user_principal_name );
				update_user_meta( $wp_user->ID, 'aadTenantId', $tenant_id );
			}

			do_action( 'wpo365/aad_user/created', $wp_user, $password );

			return $graph_user;
		}

		/**
		 * Updates an existing user (identified by the $oid parameter) in Azure AD (B2C) using the WP_User provided.
		 *
		 * @param   string  $oid
		 * @param   WP_user $wp_user
		 * @return  WP_Error|bool       Returns true if the update was successful otherwise the WP_Error.
		 */
		public static function update_user_in_azure( $oid, $wp_user, $email_changed ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			if ( empty( $oid ) ) {
				return new \WP_Error( 'ArgumentException', sprintf( '%s -> Azure Object ID is not provided when trying to update user with WordPress ID %s', __METHOD__, $wp_user->ID ) );
			}

			$user_attributes = self::get_user_attributes( $wp_user, true );

			if ( $email_changed ) {
				$b2c_domain_name = Options_Service::get_aad_option( 'b2c_domain_name' );

				if ( empty( $b2c_domain_name ) ) {
					return new \WP_Error( 'ConfigurationException', sprintf( '%s -> Your Azure AD B2C integration configuration is not complete (B2C domain name is null or empty)', __METHOD__ ) );
				}

				$authentication_method = Options_Service::get_global_string_var( 'wp_to_aad_identity_provider' );

				if ( strcasecmp( $authentication_method, 'emailOtp' ) === 0 ) {
					$user_attributes['identities'] = array(
						array(
							'signInType'       => 'federated',
							'issuer'           => 'mail',
							'issuerAssignedId' => $wp_user->user_email,
						),
					);
				} else { // Fallback: Email with password
					$user_attributes['identities'] = array(
						array(
							'signInType'       => 'emailAddress',
							'issuer'           => sprintf( '%s.onmicrosoft.com', $b2c_domain_name ),
							'issuerAssignedId' => $wp_user->user_email,
						),
					);
				}
			}

			if ( is_wp_error( $user_attributes ) ) {
				return $user_attributes;
			}

			$body_as_json = wp_json_encode( $user_attributes );
			$headers      = array(
				'Accept: application/json;odata.metadata=minimal',
				'Content-Type: application/json',
			);
			$query        = sprintf( '/users/%s', $oid );
			$fetch_result = Graph_Service::fetch( $query, 'PATCH', false, $headers, false, false, $body_as_json, 'User.ReadWrite.All' );

			if ( is_wp_error( $fetch_result ) ) {
				return $fetch_result;
			}

			$graph_error = '';
			$on_error    = function ( $log_message ) use ( &$graph_error ) {
				$graph_error = $log_message;
			};

			if ( ! Graph_Service::is_fetch_result_ok( $fetch_result, 'to update a user in Azure AD (B2C)', 'WARN', $on_error ) ) {
				return new WP_Error( 'GraphFetchException', sprintf( '%s -> %s', __METHOD__, $graph_error ) );
			}

			return self::get_aad_user( '', $oid );
		}

		/**
		 * Update's a WordPress user's meta with status info in regard of WP to AAD user synchronization.
		 *
		 * @since   24.0
		 *
		 * @param mixed $wp_user_id
		 * @param mixed $job_id
		 * @param bool  $success
		 * @param mixed $message
		 *
		 * @return WP_Error|bool
		 */
		public static function update_user_status( $wp_user_id, $job_id = '', $success = false, $message = '' ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			$wp_user = get_user_by( 'ID', $wp_user_id );

			if ( empty( $wp_user ) ) {
				return new \WP_Error( 'UserNotFoundException', sprintf( '%s -> (WordPress) User [ID: %s] cannot be resolved', __METHOD__, $wp_user_id ) );
			}

			$status_info = wp_json_encode(
				array(
					'time'    => time(),
					'job_id'  => $job_id,
					'status'  => $success ? 'OK' : 'NOK',
					'message' => str_replace( '\\', '\\\\', $message ),
				)
			);

			update_user_meta( $wp_user_id, 'wpo_sync_wp_to_aad_status', $status_info );

			return true;
		}

		/**
		 * Helper to register custom columns to show a couple of WPO365 WP to AAD User synchronization related fields
		 * on the default WordPress Users screen.
		 *
		 * @since   24.0
		 *
		 * @param   array $columns Array of columns.
		 *
		 * @return  array    Array of colums with a couple of columns added.
		 */
		public static function register_users_sync_wp_to_aad_columns( $columns ) {
			if ( ! self::wp_to_aad_is_enabled() ) {
				return $columns;
			}

			$columns['wpo365_send_to_azure'] = __( 'Send to Azure', 'wpo365-login' );
			return $columns;
		}

		/**
		 * Helper to render a couple of custom WPO365 User sync columns that are added to the default WordPress Users screen.
		 *
		 * @since   24.0
		 *
		 * @param   string $output         Rendered HTML.
		 * @param   string $column_name    Name of the column being rendered.
		 * @param   string $user_id        ID of the user the column's cell is being rendered for.
		 *
		 * @return  string  Rendered HTML.
		 */
		public static function render_users_sync_wp_to_aad_columns( $output, $column_name, $user_id ) {
			if ( ! self::wp_to_aad_is_enabled() ) {
				return $output;
			}

			if ( $column_name === 'wpo365_send_to_azure' ) {
				$url              = add_query_arg( 'wpo365_send_to_azure', $user_id );
				$status_info_json = get_user_meta( $user_id, 'wpo_sync_wp_to_aad_status', true );
				$last             = '<strong>Last</strong> -';
				$status           = '<strong>Status</strong> -';

				if ( ! empty( $status_info_json ) ) {

					$status_info = json_decode( $status_info_json, true );

					$err = json_last_error_msg();

					if ( ! empty( $status_info['time'] ) ) {
						$last = sprintf( '<strong>Last</strong> %s', gmdate( 'Y-m-d H:i', $status_info['time'] ) );
					}

					if ( ! empty( $status_info['status'] ) ) {
						$status = $status_info['status'] === 'NOK'
							? sprintf( '<strong>Status</strong> NOK <span title="%s" class="dashicons dashicons-info" style="padding-left: 7px; color: #428BCA; cursor: pointer;">', $status_info['message'] )
							: '<strong>Status</strong> OK';
					}
				}

				return sprintf(
					'<div><button type="button" onclick="window.location.href = \'%s\'">Send to Azure</button><div>%s</div><div>%s</div></div>',
					$url,
					$last,
					$status
				);
			}

			return $output;
		}

		/**
		 * Renders the WP TO AAD status information for a user on the user's profile page at the bottom of the Personal Options section.
		 *
		 * @since 28.x
		 *
		 * @param WP_User $wp_usr
		 *
		 * @return void
		 */
		public static function render_user_profile_wp_to_aad_info( $wp_usr ) {
			if ( ! self::wp_to_aad_is_enabled() ) {
				return;
			}

			$url              = add_query_arg( 'wpo365_send_to_azure', $wp_usr->ID );
			$status_info_json = get_user_meta( $wp_usr->ID, 'wpo_sync_wp_to_aad_status', true );
			$last             = '<strong>Last</strong> -';
			$status           = '<strong>Status</strong> -';

			if ( ! empty( $status_info_json ) ) {

				$status_info = json_decode( $status_info_json, true );

				$err = json_last_error_msg();

				if ( ! empty( $status_info['time'] ) ) {
					$last = sprintf( '<strong>Last</strong> %s', gmdate( 'Y-m-d H:i', $status_info['time'] ) );
				}

				if ( ! empty( $status_info['status'] ) ) {
					$status = $status_info['status'] === 'NOK'
						? sprintf( '<strong>Status</strong> NOK <span title="%s" class="dashicons dashicons-info" style="padding-left: 7px; color: #428BCA; cursor: pointer;">', $status_info['message'] )
						: '<strong>Status</strong> OK';
				}
			}

			$output = sprintf(
				'<tr><th scope="row">Send to Azure</th><td><p><button type="button" onclick="window.location.href = \'%s\'">Send to Azure</button></p><p>%s</p><p>%s</p></td></tr>',
				$url,
				$last,
				$status
			);

			echo wp_kses( $output, WordPress_Helpers::get_allowed_html() );
		}

		/**
		 * Handler for a "Send to Azure" button click (in which case the user ID parameter is passed as a query string parameter).
		 *
		 * @since   24.0
		 *
		 * @return void
		 */
		public static function send_to_azure() {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			if ( ! self::wp_to_aad_is_enabled() ) {
				return;
			}

			if ( isset( $_GET['wpo365_send_to_azure'] ) ) { // phpcs:ignore
				$wp_usr_id = (int) sanitize_text_field( $_GET['wpo365_send_to_azure'] ); // phpcs:ignore
				$wp_usr    = get_user_by( 'ID', $wp_usr_id );

				if ( ! empty( $wp_usr ) ) {
					$wp_to_aad_excluded_roles = Options_Service::get_global_list_var( 'wp_to_aad_excluded_roles' );

					foreach ( $wp_to_aad_excluded_roles as $wp_to_aad_excluded_role ) {

						if ( in_array( $wp_to_aad_excluded_role, $wp_usr->roles, true ) ) {
							self::update_user_status( $wp_usr_id, '', false, sprintf( '%s -> Not sending user with ID %d to Azure because the user\'s role %s has been added to the exclude-list', __METHOD__, $wp_usr_id, $wp_to_aad_excluded_role ) );
							return;
						}
					}

					$wp_usr->user_pass    = ''; // The user didn't enter one so we set it to a random password
					$create_update_result = self::send_user_to_azure( $wp_usr, true );

					if ( is_wp_error( $create_update_result ) ) {
						$message = sprintf( '%s -> Failed to create / update WordPress user with ID %s in Azure AD (B2C) [Error: %s]', __METHOD__, $wp_usr_id, $create_update_result->get_error_message() );
						Log_Service::write_log( 'ERROR', $message );
						self::update_user_status( $wp_usr_id, '', false, $message );
					} else {
						self::update_user_status( $wp_usr_id, '', true );
					}
				}

				Url_Helpers::force_redirect( remove_query_arg( 'wpo365_send_to_azure' ) );
			}
		}

		/**
		 * Adds custom bulk actions to the central WordPress user list e.g. to allow administrators to manually sync multiple users to Azure AD (B2C).
		 *
		 * @since   24.0
		 *
		 * @param   array $bulk_actions
		 * @return  array   The (modified) array of bulk actions
		 */
		public static function users_sync_wp_to_aad_bulk_actions( $bulk_actions ) {
			if ( ! self::wp_to_aad_is_enabled() ) {
				return $bulk_actions;
			}

			$bulk_actions['send-to-azure'] = __( 'Send to Azure', 'wpo365-login' );
			return $bulk_actions;
		}

		/**
		 * Applies the custom bulk actions e.g. to create / update users in Azure AD (B2C).
		 *
		 * @since   24.0
		 *
		 * @param   string $redirect_url
		 * @param   string $action
		 * @param   array  $user_ids
		 * @return  string  The redirect URL with an additional parameter to sho
		 */
		public static function users_sync_wp_to_aad_bulk_actions_handler( $redirect_url, $action, $user_ids ) {
			if ( ! self::wp_to_aad_is_enabled() ) {
				return $redirect_url;
			}

			if ( $action === 'send-to-azure' ) {
				$wp_to_aad_excluded_roles = Options_Service::get_global_list_var( 'wp_to_aad_excluded_roles' );

				foreach ( $user_ids as $wp_usr_id ) {
					$wp_usr   = get_user_by( 'ID', $wp_usr_id );
					$excluded = false;

					if ( ! empty( $wp_usr ) ) {

						foreach ( $wp_to_aad_excluded_roles as $wp_to_aad_excluded_role ) {

							if ( in_array( $wp_to_aad_excluded_role, $wp_usr->roles, true ) ) {
								self::update_user_status( $wp_usr_id, '', false, sprintf( '%s -> Not sending user with ID %d to Azure because the user\'s role %s has been added to the exclude-list', __METHOD__, $wp_usr_id, $wp_to_aad_excluded_role ) );
								$excluded = true;
								break;
							}
						}

						if ( $excluded ) {
							continue;
						}

						$wp_usr->user_pass    = ''; // The user didn't enter one so we set it to a random password
						$create_update_result = self::send_user_to_azure( $wp_usr, true );

						if ( is_wp_error( $create_update_result ) ) {
							Log_Service::write_log( 'ERROR', sprintf( '%s -> Failed to create / update WordPress user with ID %s in Azure AD (B2C) [Error: %s]', __METHOD__, $wp_usr_id, $create_update_result->get_error_message() ) );
							self::update_user_status( $wp_usr_id, '', false, $create_update_result->get_error_message() );
						} else {
							self::update_user_status( $wp_usr_id, '', true );
						}
					}
				}

				$redirect_url = add_query_arg( 'send-to-azure', count( $user_ids ), $redirect_url );
			}

			return $redirect_url;
		}

		/**
		 * Filters the contents of the regular user list to only show
		 * users with WP to AAD sync issues (optionally for a specific job)
		 * or users that were affected by a specific job.
		 *
		 * @since   24.0
		 *
		 * @param   mixed $query
		 * @return  mixed
		 */
		public static function get_users( $query ) {
			if ( ! self::wp_to_aad_is_enabled() ) {
				return $query;
			}

			global $pagenow;

			if ( ! is_admin() || $pagenow !== 'users.php' || ( ! isset( $_REQUEST['wpo365_filter'] ) && ! isset( $_REQUEST['wpo365_jobid'] ) ) ) { // phpcs:ignore
				return $query;
			}

			if ( isset( $_REQUEST['wpo365_filter'] ) && $_REQUEST['wpo365_filter'] === 'wpo365_send_to_azure_nok' ) { // phpcs:ignore
				$meta_query = array(
					array(
						'key'     => 'wpo_sync_wp_to_aad_status',
						'value'   => '"status":"NOK"',
						'compare' => 'LIKE',
					),
				);
			}

			if ( isset( $_REQUEST['wpo365_jobid'] ) ) { // phpcs:ignore
				$job_id = sanitize_text_field( $_REQUEST['wpo365_jobid'] ); // phpcs:ignore

				if ( empty( $meta_query ) ) {
					$meta_query = array(
						array(
							'key'     => 'wpo_sync_wp_to_aad_status',
							'value'   => sprintf( '"job_id":"%s"', $job_id ),
							'compare' => 'LIKE',
						),
					);
				} else {
					$meta_query['relation'] = 'AND';
					$meta_query[][]         = array(
						'key'     => 'wpo_sync_wp_to_aad_status',
						'value'   => sprintf( '"job_id":"%s"', $job_id ),
						'compare' => 'LIKE',
					);
				}
			}

			if ( ! empty( $meta_query ) ) {
				$query->set( 'meta_query', $meta_query );
			}

			return $query;
		}

		/**
		 * Renders a button in the user list's toolbar to filter for users with issues when syncing to Azure AD (B2C).
		 *
		 * @since   24.0
		 *
		 * @param mixed $which
		 * @return void
		 */
		public static function render_button_nok_users( $which ) { // phpcs:ignore

			if ( ! self::wp_to_aad_is_enabled() ) {
				return;
			}

			$html = sprintf(
				'<input type="button" name="wpoSendToAzureNok" id="wpoSendToAzureNok" onclick="location.href=\'%s\'" class="button" value="Filter: Send to Azure -> NOK" style="margin-left: 6px;">',
				add_query_arg( 'wpo365_filter', 'wpo365_send_to_azure_nok' )
			);

			echo wp_kses( $html, WordPress_Helpers::get_allowed_html() );
		}

		/**
		 * @param WP_User $wp_user
		 * @param string  $user_pass
		 * @return void
		 */
		public static function send_new_customer_notification( $wp_user, $user_pass ) {
			$wp_to_aad_email_notifications = Options_Service::get_global_string_var( 'wp_to_aad_email_notifications' );

			// Admin configured to never send new customer notification
			if ( strcasecmp( $wp_to_aad_email_notifications, 'never' ) === 0 ) {
				return;
			}

			// The user entered a password interactively and the admin did not force to send new customer notification always
			if ( ! empty( $wp_user->user_pass ) && strcasecmp( $wp_to_aad_email_notifications, 'always' ) !== 0 ) {
				return;
			}

			$blogname      = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );
			$customer_url  = Options_Service::get_global_string_var( 'wp_to_aad_customer_endpoint' );
			$domain_name   = Options_Service::get_aad_option( 'b2c_domain_name' );
			$custom_domain = Options_Service::get_aad_option( 'b2c_custom_domain' );

			if ( empty( $custom_domain ) ) {
				$custom_domain = sprintf(
					'%s.%slogin.com',
					Options_Service::get_global_boolean_var( 'use_ciam' ) ? 'ciam' : 'b2c',
					$domain_name
				);
			}

			/* translators: The user's name - Do not translate */
			$message = array( __( 'Hi %s,' ) );
			/* translators: The website's name - Do not translate */
			$message[] = __( 'We have updated our website %s and made it easier for you to log in.' );
			/* translators: The account URL - Do not translate */
			$message[] = __( 'You can access your account to manage your details, subscriptions, licenses and download purchased downloads at %s.' );
			/* translators: The AAD B2C / Entra Ext. ID endpoint - Do not translate */
			$message[] = __( 'Please note that you will automatically redirected to %s, where you can now sign in with your email address.' );
			/* translators: The temporary password - Do not translate */
			$message[] = __( 'We have created a temporary password %s for you and you will be asked to change it when you first sign in.' );
			$message[] = __( 'We look forward to see you soon!' );

			$body = sprintf( '<p>%s</p>', implode( '</p><p>', $message ) );
			$body = sprintf( $body, $wp_user->display_name, $blogname, $customer_url, $custom_domain, $user_pass );

			$wp_new_customer_notification_email = array(
				'to'      => $wp_user->user_email,
				/* translators: The website's name - Do not translate */
				'subject' => __( 'Welcome to %s' ),
				'body'    => $body,
				'headers' => 'Content-Type: text/html; charset=UTF-8',
			);

			wp_mail(
				$wp_new_customer_notification_email['to'],
				wp_specialchars_decode( sprintf( $wp_new_customer_notification_email['subject'], $blogname ) ),
				$wp_new_customer_notification_email['body'],
				$wp_new_customer_notification_email['headers']
			);
		}

		/**
		 * Check to see if the WP to AAD flow is enabled either through user-registration or through user synchronization.
		 *
		 * @since 24.0
		 *
		 * @return bool
		 */
		public static function wp_to_aad_is_enabled() {
			$request_service = Request_Service::get_instance();
			$request         = $request_service->get_request( $GLOBALS['WPO_CONFIG']['request_id'] );

			if ( $request->get_item( 'wp_to_aad' ) ) {
				return true;
			}

			if ( ! Options_Service::get_global_boolean_var( 'use_b2c', false ) && ! Options_Service::get_global_boolean_var( 'use_ciam', false ) ) {
				$request->set_item( 'wp_to_aad', false );
				return false;
			}

			if ( Options_Service::get_global_boolean_var( 'wp_to_aad_create_users', false ) ) {
				$request->set_item( 'wp_to_aad', true );
				return true;
			}

			if ( ! Options_Service::get_global_boolean_var( 'enable_user_sync', false ) ) {
				$request->set_item( 'wp_to_aad', false );
				return false;
			}

			$sync_jobs = Options_Service::get_global_list_var( 'user_sync_jobs', false );

			foreach ( $sync_jobs as $sync_job ) {

				if ( isset( $sync_job['type'] ) && $sync_job['type'] === 'wpToAad' ) {
					$request->set_item( 'wp_to_aad', true );
					return true;
				}
			}

			$request->set_item( 'wp_to_aad', false );
			return false;
		}

		/**
		 * Adds default user attributes (first, last and full name plus email address) and optional
		 * ones that the administrator configured. Adding the attribute will be skipped if a value
		 * appears to be empty.
		 *
		 * @param WP_User $wp_user
		 * @return WP_Error|array
		 */
		private static function get_user_attributes( $wp_user, $update = false ) {
			$user_attributes = Options_Service::get_global_list_var( 'wp_to_aad_user_attribute_mappings' );

			if ( ! filter_var( $wp_user->user_email, FILTER_VALIDATE_EMAIL ) ) {
				return new WP_Error( 'RequiredPropertyException', sprintf( '%s -> Email address for user with ID %d not found or invalid', __METHOD__, $wp_user->ID ) );
			}

			$props = array( 'mail' => $wp_user->user_email );

			if ( ! empty( $wp_user->display_name ) ) {
				$props['displayName'] = $wp_user->display_name;
			}

			if ( ! empty( $wp_user->user_firstname ) ) {
				$props['givenName'] = $wp_user->user_firstname;
			}

			if ( empty( $props['givenName'] ) && ! empty( $wp_user->first_name ) ) {
				$props['givenName'] = $wp_user->first_name;
			}

			if ( ! empty( $wp_user->user_lastname ) ) {
				$props['surname'] = $wp_user->user_lastname;
			}

			if ( empty( $props['surname'] ) && ! empty( $wp_user->last_name ) ) {
				$props['surname'] = $wp_user->last_name;
			}

			if ( count( $user_attributes ) === 0 || ! isset( $wp_user->ID ) ) {
				return $props;
			}

			$user_meta              = get_user_meta( $wp_user->ID );
			$update_sensitive_props = Options_Service::get_global_boolean_var( 'wp_to_aad_update_sensitive_props' );

			foreach ( $user_attributes as $index => $attribute ) {

				if ( $update && ! $update_sensitive_props ) {

					if ( WordPress_Helpers::stripos( $attribute['strC'], 'businessPhones' ) === 0 || WordPress_Helpers::stripos( $attribute['strC'], 'mobilePhone' ) === 0 ) {
						Log_Service::write_log( 'WARN', sprintf( '%s -> Skipping updating user attribute with key %s for user with ID %s because updating sensitive properties has not been configured', __METHOD__, $attribute['strC'], $wp_user->ID ) );
						continue;
					}
				}

				// key
				if ( empty( $attribute['strB'] ) ) {
					continue;
				}

				$key = $attribute['strB'];

				// value
				if ( $attribute['strA'] === 'WP_User' ) {
					$value = isset( $wp_user->$key ) ? $wp_user->$key : '';
				} elseif ( $attribute['strA'] === 'usermeta' ) {
					$value = isset( $user_meta[ $key ] ) ? $user_meta[ $key ] : '';

					// usermeta by default stored as arrays
					if ( is_array( $value ) && count( $value ) >= 1 ) {
						$value = $value[0];
					}
				}

				if ( empty( $value ) ) {
					Log_Service::write_log( 'WARN', sprintf( '%s -> Skipping adding user attribute with key %s for user with ID %s because setting empty values is not supported', __METHOD__, $key, $wp_user->ID ) );
					continue;
				}

				// property
				if ( WordPress_Helpers::stripos( $attribute['strC'], '.' ) !== false ) {
					$members = explode( '.', $attribute['strC'] );

					if ( count( $members ) === 2 ) {

						if ( is_numeric( $members[1] ) ) {
							$props[ $members[0] ] = array( $value );
						} else {
							$props[ $members[0] ] = array( $members[1] => $value );
						}
					} else {
						Log_Service::write_log( 'WARN', sprintf( '%s -> Nesting level %d of the target user property %s is currently not supported', __METHOD__, count( $members ), $attribute->strC ) ); // phpcs:ignore
					}
				} else {
					$props[ $attribute['strC'] ] = $value;
				}
			}

			return $props;
		}

		private static function replace_template_tags( $template, $wp_user, $blogname ) {
			$template = str_replace( '__##USER_DISPLAY_NAME##__', $wp_user->display_name, $template );
			$template = str_replace( '__##USER_LOGIN_NAME##__', $wp_user->user_login, $template );
			$template = str_replace( '__##USER_EMAIL##__', $wp_user->user_email, $template );
			$template = str_replace( '__##BLOG_NAME##__', $blogname, $template );
			$template = str_replace( '__##BLOG_URL##__', network_site_url(), $template );
			return $template;
		}
	}
}
