<?php

namespace Wpo\Services;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

use Wpo\Core\Compatibility_Helpers;
use Wpo\Core\WordPress_Helpers;
use Wpo\Services\Log_Service;
use Wpo\Services\Options_Service;
use Wpo\Services\Graph_Service;
use Wpo\Services\Saml2_Service;
use Wpo\Services\User_Service;
use Wpo\Services\User_Details_Service;

if ( ! class_exists( '\Wpo\Services\User_Custom_Fields_Service' ) ) {

	class User_Custom_Fields_Service {

		public static function update_custom_fields( $wp_usr_id, $wpo_usr ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );
			self::update_custom_fields_from_graph( $wp_usr_id, $wpo_usr );
			self::update_custom_fields_from_id_token( $wp_usr_id, $wpo_usr );
			self::update_custom_fields_from_saml_attributes( $wp_usr_id, $wpo_usr );
		}

		/**
		 * @since 11.0
		 */
		public static function update_custom_fields_from_graph( $wp_usr_id, $wpo_usr ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			if ( empty( $wpo_usr->graph_resource ) ) {
				Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Skipping updating custom user fields [Graph User Resource not found]' );
				return;
			}

			self::process_extra_user_fields(
				function ( $name, $title ) use ( &$wpo_usr, &$wp_usr_id ) {
					$parsed_user_field_key = User_Details_Service::parse_user_field_key( $name );
					$name                  = $parsed_user_field_key[0];
					$wp_user_meta_key      = $parsed_user_field_key[1];

					if ( \method_exists( '\Wpo\Core\Compatibility_Helpers', 'check_user_claim_prefix' ) ) {
						Compatibility_Helpers::check_user_claim_prefix( $name );
					}

					$is_graph_property = WordPress_Helpers::stripos( $name, 'graph::' ) === 0;

					if ( $is_graph_property === true ) {
						$name = str_replace( 'graph::', '', $name );
					}

					// Expand manager property
					if ( is_string( $name ) && ( strcasecmp( $name, 'manager' ) === 0 || WordPress_Helpers::stripos( $name, 'manager.' ) === 0 ) ) {
						$upn = User_Service::try_get_user_principal_name( $wp_usr_id );

						if ( ! empty( $upn ) ) {
							$user_manager = Graph_Service::fetch( '/users/' . \rawurlencode( $upn ) . '/manager', 'GET', false, array( 'Accept: application/json;odata.metadata=minimal' ) );

							// Expand user details
							if ( Graph_Service::is_fetch_result_ok( $user_manager, 'Could not retrieve user manager details for user ' . $upn, 'WARN' ) ) {
								$wpo_usr->graph_resource['manager'] = $user_manager['payload'];
							}
						}
					}

					$name_arr = explode( '.', $name );
					$current  = $wpo_usr->graph_resource;
					$value    = null;

					if ( count( $name_arr ) > 1 ) {

						$found       = false;
						$array_count = count( $name_arr );

						for ( $i = 0; $i < $array_count; $i++ ) {
							/**
							 * Found must be true for the last iteration therefore it resets every cycle
							 */

							$found = false;

							if ( is_array( $current ) && \is_numeric( $name_arr[ $i ] ) && $i < count( $current ) && array_key_exists( $name_arr[ $i ], $current ) ) { // Administrator has specified to get the nth element of an array / object
								$current = $current[ $name_arr[ $i ] ];
								$found   = true;
							} elseif ( is_array( $current ) && array_key_exists( $name_arr[ $i ], $current ) ) { // Administrator has specified to get the named element of an array / object
								$current = $current[ $name_arr[ $i ] ];
								$found   = true;
							}

							/**
							 * Bail out early if one segment of the path isn't found.
							 */

							if ( ! $found ) {
								break;
							}
						}

						if ( $found ) {
							$value = $current;
						}
					} elseif ( array_key_exists( $name, $current ) && ! empty( $current[ $name ] ) ) { // Administrator has specified a simple non-nested property
						$value = $name === 'manager'
						? self::parse_manager_details( $current['manager'] )
						: $current[ $name ];
					}

					// ID token claim was expected but not sent which indicates that the existing meta data should be emptied
					if ( $is_graph_property && $value === null ) {
						$value = '';
					}

					if ( $value !== null ) {
						update_user_meta(
							$wp_usr_id,
							$wp_user_meta_key,
							$value
						);

						if ( function_exists( 'xprofile_set_field_data' ) && Options_Service::get_global_boolean_var( 'use_bp_extended' ) === true ) {
							xprofile_set_field_data( $title, $wp_usr_id, $value );
						}
					}
				}
			);
		}

		/**
		 * Processes the extra user fields and tries to read them from the SAML attributes
		 * and if found saves their value as WordPress user meta.
		 *
		 * @since   20.0
		 *
		 * @param   mixed $wp_usr_id
		 * @param   mixed $wpo_usr
		 * @return  void
		 */
		public static function update_custom_fields_from_saml_attributes( $wp_usr_id, $wpo_usr ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			if ( empty( $wpo_usr->saml_attributes ) ) {
				Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Skipping updating custom user fields [SAML attributes not found]' );
				return;
			}

			self::process_extra_user_fields(
				function ( $name, $title ) use ( &$wpo_usr, &$wp_usr_id ) {

					$parsed_user_field_key = User_Details_Service::parse_user_field_key( $name );
					$claim                 = $parsed_user_field_key[0];
					$wp_user_meta_key      = $parsed_user_field_key[1];

					if ( \method_exists( '\Wpo\Core\Compatibility_Helpers', 'check_user_claim_prefix' ) ) {
						Compatibility_Helpers::check_user_claim_prefix( $claim );
					}

					$is_saml_claim = WordPress_Helpers::stripos( $claim, 'saml::' ) === 0;

					if ( $is_saml_claim === true ) {
						$claim = str_replace( 'saml::', '', $claim );
					}

					if ( strcmp( $claim, $wp_user_meta_key ) === 0 && WordPress_Helpers::stripos( $wp_user_meta_key, '/' ) > 0 ) {
						$key_exploded     = explode( '/', $wp_user_meta_key );
						$wp_user_meta_key = sprintf( 'saml_%s', array_pop( $key_exploded ) );
					}

					$value = Saml2_Service::get_attribute( $claim, $wpo_usr->saml_attributes, false, 'string', true );

					// SAML claim was expected but not sent which indicates that the existing meta data should be emptied
					if ( $is_saml_claim && $value === null ) {
						$value = '';
					}

					if ( $value !== null ) {
						update_user_meta(
							$wp_usr_id,
							$wp_user_meta_key,
							$value
						);

						if ( function_exists( 'xprofile_set_field_data' ) && Options_Service::get_global_boolean_var( 'use_bp_extended' ) === true ) {
							xprofile_set_field_data( $title, $wp_usr_id, $value );
						}
					}
				}
			);
		}

		/**
		 * Processes the custom user fields and tries to read them from the ID token
		 * and if found saves their value as WordPress user meta.
		 *
		 * @since   30.x
		 *
		 * @param   mixed $wp_usr_id
		 * @param   mixed $wpo_usr
		 * @return  void
		 */
		public static function update_custom_fields_from_id_token( $wp_usr_id, $wpo_usr ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			if ( empty( $wpo_usr->id_token ) ) {
				Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Skipping updating custom user fields [ID token not found]' );
				return;
			}

			self::process_extra_user_fields(
				function ( $name, $title ) use ( &$wpo_usr, &$wp_usr_id ) {

					$parsed_user_field_key = User_Details_Service::parse_user_field_key( $name );
					$claim                 = $parsed_user_field_key[0];
					$wp_user_meta_key      = $parsed_user_field_key[1];

					if ( \method_exists( '\Wpo\Core\Compatibility_Helpers', 'check_user_claim_prefix' ) ) {
						Compatibility_Helpers::check_user_claim_prefix( $claim );
					}

					$is_oidc_claim = WordPress_Helpers::stripos( $claim, 'oidc::' ) === 0;

					if ( $is_oidc_claim === true ) {
						$claim = str_replace( 'oidc::', '', $claim );
					}

					if ( strcmp( $claim, $wp_user_meta_key ) === 0 && WordPress_Helpers::stripos( $wp_user_meta_key, '/' ) > 0 ) {
						$key_exploded     = explode( '/', $wp_user_meta_key );
						$wp_user_meta_key = sprintf( 'oidc_%s', array_pop( $key_exploded ) );
						$wp_user_meta_key = str_replace( '.', '_', $wp_user_meta_key );
					}

					$value = isset( $wpo_usr->id_token->$claim ) ? $wpo_usr->id_token->$claim : null;

					// ID token claim was expected but not sent which indicates that the existing meta data should be emptied
					if ( $is_oidc_claim && $value === null ) {
						$value = '';
					}

					if ( $value !== null ) {
						update_user_meta(
							$wp_usr_id,
							$wp_user_meta_key,
							$value
						);

						if ( function_exists( 'xprofile_set_field_data' ) && Options_Service::get_global_boolean_var( 'use_bp_extended' ) === true ) {
							xprofile_set_field_data( $title, $wp_usr_id, $value );
						}
					}
				}
			);
		}

		/**
		 *
		 * @param function $callback
		 *
		 * @return void
		 */
		public static function process_extra_user_fields( $callback ) {
			$extra_user_fields = Options_Service::get_global_list_var( 'extra_user_fields' );

			if ( count( $extra_user_fields ) === 0 ) {
				return;
			}

			foreach ( $extra_user_fields as $kv_pair ) {
				$callback( $kv_pair['key'], $kv_pair['value'] );
			}
		}

		/**
		 * Adds an additional section to the bottom of the user profile page
		 *
		 * @since 2.0
		 *
		 * @param WP_User $user WP User whose profile is being shown.
		 * @return void
		 */
		public static function show_extra_user_fields( $user ) {
			if ( Options_Service::get_global_boolean_var( 'graph_user_details' ) === false ) {
				Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Extra user fields disabled as per configuration' );
				return;
			} elseif ( Options_Service::get_global_boolean_var( 'use_bp_extended' ) === true ) {
				Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Extra user fields will be display on BuddyPress Extended Profile instead' );
				return;
			} else {
				$label = Options_Service::get_global_string_var( 'wpo_title_m365_profile_info' );

				if ( empty( $label ) ) {
					$label = __( 'Microsoft 365 Profile Information', 'wpo365-login' );
				}

				echo wp_kses( "<h3>$label</h3>", WordPress_Helpers::get_allowed_html() );
				echo ( '<table class="form-table">' );

				self::process_extra_user_fields(
					function ( $name, $title ) use ( &$user ) {

						$parsed_user_field_key = User_Details_Service::parse_user_field_key( $name );
						$name                  = $parsed_user_field_key[0];
						$wp_user_meta_key      = $parsed_user_field_key[1];

						// The following may be true for SAML based custom attributes
						if ( strcmp( $name, $wp_user_meta_key ) === 0 && WordPress_Helpers::stripos( $wp_user_meta_key, '/' ) > 0 ) {
							$key_exploded     = explode( '/', $wp_user_meta_key );
							$wp_user_meta_key = sprintf( 'saml_%s', array_pop( $key_exploded ) );
						}

						$value = get_user_meta( $user->ID, $wp_user_meta_key, true );

						echo ( '<tr><th><label for="' . esc_attr( $wp_user_meta_key ) . '">' . esc_html( $title ) . '</label></th>' );

						if ( is_array( $value ) ) {

							echo ( '<td>' );

							foreach ( $value as $idx => $val ) {

								if ( empty( $val ) ) {
									continue;
								}

								echo '<input type="text" name="' . esc_attr( $wp_user_meta_key ) . '__##__' . esc_attr( $idx ) . '" id="' . esc_attr( $wp_user_meta_key ) . esc_attr( $idx ) . '" value="' . esc_attr( $val ) . '" class="regular-text" /><br />';
							}

							echo ( '</td>' );
						} else {

							echo ( '<td><input type="text" name="' . esc_attr( $wp_user_meta_key ) . '" id="' . esc_attr( $wp_user_meta_key ) . '" value="' . esc_attr( $value ) . '" class="regular-text" /><br/></td>' );
						}

						echo ( '</tr>' );
					}
				);

				echo ( '</table>' );
			}
		}

		/**
		 * Allow users to save their updated extra user fields
		 *
		 * @since 4.0
		 *
		 * @return mixed(boolean|void)
		 */
		public static function save_user_details( $user_id ) {
			if ( ! current_user_can( 'edit_user', $user_id ) ) {
				return false;
			}

			self::process_extra_user_fields(
				function ( $name, $title ) use ( &$user_id ) { // phpcs:ignore

					$parsed_user_field_key = User_Details_Service::parse_user_field_key( $name );
					$name                  = $parsed_user_field_key[0];
					$wp_user_meta_key      = $parsed_user_field_key[1];

					// The following may be true for SAML based custom attributes
					if ( strcmp( $name, $wp_user_meta_key ) === 0 && WordPress_Helpers::stripos( $wp_user_meta_key, '/' ) > 0 ) {
						$key_exploded     = explode( '/', $wp_user_meta_key );
						$wp_user_meta_key = sprintf( 'saml_%s', array_pop( $key_exploded ) );
					}

					$lookup = str_replace( '.', '_', $wp_user_meta_key ); // '.' is changed to '_' when sent in a request

					if ( isset( $_POST[ $lookup ] ) ) { // phpcs:ignore

						update_user_meta(
							$user_id,
							$wp_user_meta_key,
							sanitize_text_field( wp_unslash( $_POST[ $lookup ] ) ) // phpcs:ignore
						);
						return;
					}

					$array_of_user_meta = array();

					foreach ( $_POST as $key => $value ) { // phpcs:ignore

						if ( WordPress_Helpers::strpos( $key, $lookup . '__##__' ) !== false ) {
							$array_of_user_meta[ $key ] = $value;
						}
					}

					if ( empty( $array_of_user_meta ) === false ) {

						$array_of_user_meta_values = array_values( $array_of_user_meta );

						update_user_meta(
							$user_id,
							$wp_user_meta_key,
							$array_of_user_meta_values
						);
						return;
					}
				}
			);
		}

		/**
		 * Gets details of a user as an array with displayName, mail, officeLocation, department,
		 * businessPhones, mobilePhone.
		 *
		 * @since   15.0
		 *
		 * @param   int $wp_usr_id
		 * @return  array
		 */
		public static function get_manager_details_from_wp_user( $wp_usr_id ) {
			$wp_usr = get_user_by( 'ID', $wp_usr_id );

			if ( empty( $wp_usr ) ) {
				return array();
			}

			$display_name    = $wp_usr->display_name;
			$mail            = $wp_usr->user_email;
			$office_location = get_user_meta( $wp_usr_id, 'officeLocation', true );
			$department      = get_user_meta( $wp_usr_id, 'department', true );
			$business_phones = get_user_meta( $wp_usr_id, 'businessPhones', true );
			$mobile_phone    = get_user_meta( $wp_usr_id, 'mobilePhone', true );

			return array(
				'displayName'    => $display_name,
				'mail'           => $mail,
				'officeLocation' => ! empty( $office_location ) ? $office_location : '',
				'department'     => ! empty( $department ) ? $department : '',
				'businessPhones' => ! empty( $business_phones ) ? $business_phones : '',
				'mobilePhone'    => ! empty( $mobile_phone ) ? $mobile_phone : '',
			);
		}

		/**
		 * Parses the manager details fetched from Microsoft Graph.
		 *
		 * @since 7.17
		 *
		 * @return array Assoc. array with the most important manager details.
		 */
		private static function parse_manager_details( $manager ) {
			if ( empty( $manager ) ) {
				return array();
			}
			$display_name    = ! empty( $manager['displayName'] )
				? $manager['displayName']
				: '';
			$mail            = ! empty( $manager['mail'] )
				? $manager['mail']
				: '';
			$office_location = ! empty( $manager['officeLocation'] )
				? $manager['officeLocation']
				: '';
			$department      = ! empty( $manager['department'] )
				? $manager['department']
				: '';
			$business_phones = ! empty( $manager['businessPhones'] )
				? $manager['businessPhones'][0]
				: '';
			$mobile_phone    = ! empty( $manager['mobilePhone'] )
				? $manager['mobilePhone'][0]
				: '';
			return array(
				'displayName'    => $display_name,
				'mail'           => $mail,
				'officeLocation' => $office_location,
				'department'     => $department,
				'businessPhones' => $business_phones,
				'mobilePhone'    => $mobile_phone,
			);
		}
	}
}
