<?php

namespace Wpo\Sync;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

use WP_Error;
use Wpo\Core\Permissions_Helpers;
use Wpo\Core\Version;
use Wpo\Services\Log_Service;
use Wpo\Services\Options_Service;

if ( ! class_exists( '\Wpo\Sync\SyncV2_Helpers' ) ) {

	class Sync_Helpers {

		/**
		 * Checks if the user can retrieve an access token for the requested scope.
		 *
		 * @since   24.0            Moved to Sync_Helpers
		 *
		 * @param   WP_REST_Request $request
		 * @return  bool|WP_Error   True if user can retrieve an access token for the requested scope otherwise a WP_Error is returned.
		 */
		public static function check_permissions( $request ) {
			if ( ! wp_verify_nonce( $request->get_header( 'X-WP-Nonce' ), 'wp_rest' ) ) {
				return new \WP_Error( 'UnauthorizedException', 'The request cannot be validated.', array( 'status' => 401 ) );
			}

			$wp_usr = \wp_get_current_user();

			if ( empty( $wp_usr ) ) {
				return new \WP_Error( 'UnauthorizedException', 'Please sign in first before using this API.', array( 'status' => 401 ) );
			}

			if ( ! Permissions_Helpers::user_is_admin( $wp_usr ) ) {
				return new \WP_Error( 'UnauthorizedException', 'Please sign in with administrative credentials before using this API.', array( 'status' => 403 ) );
			}

			return true;
		}

		/**
		 * Will try and retrieve a user sync job by ID from the WPO365 configuration.
		 *
		 * @since   15.0
		 * @since   24.0            Moved to Sync_Helpers
		 *
		 * @param   string $job_id  The ID of the job.
		 * @return  array|WP_Error  he job found or WP_Error if not found.
		 */
		public static function get_user_sync_job_by_id( $job_id ) {
			$can_custom_log = version_compare( Version::$current, '36.2' ) > 0;
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			$jobs = Options_Service::get_global_list_var( 'user_sync_jobs', false );
			$job  = 0;

			foreach ( $jobs as $_job ) {

				if ( $_job['id'] === $job_id ) {
					$job = $_job;
					break;
				}
			}

			if ( empty( $job ) ) {
				$message = sprintf( '%s -> User synchronization job with ID %s not found.', __METHOD__, $job_id );
				$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
				return new WP_Error( 'NotFoundException', $message );
			}

			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> Found a WPO365 User Sync job with ID: %s', __METHOD__, $job['id'] ), 'sync' );

			return $job;
		}

		/**
		 * @since 10.0
		 *
		 * @param string $job_id
		 * @param bool   $delete Whether or not the scheduled events should be deleted.
		 * @param bool   $wp_to_aad
		 *
		 * @return array Collection of scheduled events
		 */
		public static function get_scheduled_events( $job_id, $delete = false, $wp_to_aad = false ) {
			$can_custom_log = version_compare( Version::$current, '36.2' ) > 0;
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> %s [job_id: %s, delete: %d]', __METHOD__, $job_id, $delete ), 'sync' );

			$cron_jobs     = _get_cron_array();
			$wpo_sync_jobs = array();

			foreach ( $cron_jobs as $timestamp => $array_of_jobs ) {

				foreach ( $array_of_jobs as $hook => $jobs ) {

					if ( ! $wp_to_aad && ( $hook === 'wpo_sync_users_start' || $hook === 'wpo_sync_users' || $hook === 'wpo_sync_v2_users_start' || $hook === 'wpo_sync_v2_users_next' ) ) {

						foreach ( $jobs as $id => $job ) {
							$job['hook']                 = $hook; // Add the hook back so it can be used.
							$wpo_sync_jobs[ $timestamp ] = $job;

							if ( $delete ) {

								if ( empty( $job_id ) || $job['args'][0] === $job_id ) {
									$nr_of_unscheduled_events = wp_clear_scheduled_hook( $hook, $job['args'] );
									$message                  = sprintf( '%s -> Unscheduled %d cron jobs [hook: %s]', __METHOD__, $nr_of_unscheduled_events, $hook );
									Log_Service::write_log( 'DEBUG', $message );
									$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
								}
							}
						}
					}

					if ( $wp_to_aad && ( $hook === 'wpo_sync_wp_to_aad_start' || $hook === 'wpo_sync_wp_to_aad_next' ) ) {

						foreach ( $jobs as $id => $job ) {
							$job['hook']                 = $hook; // Add the hook back so it can be used.
							$wpo_sync_jobs[ $timestamp ] = $job;

							if ( $delete ) {

								if ( empty( $job_id ) || $job['args'][0] === $job_id ) {
									$nr_of_unscheduled_events = wp_clear_scheduled_hook( $hook, $job['args'] );
									Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Unscheduled ' . $nr_of_unscheduled_events . ' cron jobs [hook: ' . $hook . ']' );
								}
							}
						}
					}
				}
			}

			return $wpo_sync_jobs;
		}

		/**
		 * Updates the job provided in the array of jobs that are stored
		 * as part of the WPO365 settings.
		 *
		 * @since 15.0
		 *
		 * @param   array $job        The job to be updated in the array.
		 *
		 * @return  boolean|WP_Error    True if no error occurred otherwise WP_Error
		 */
		public static function update_user_sync_job( $job ) {
			$can_custom_log = version_compare( Version::$current, '36.2' ) > 0;
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			$jobs      = Options_Service::get_global_list_var( 'user_sync_jobs', false );
			$job_index = -1;

			foreach ( $jobs as $i => $_job ) {

				if ( $_job['id'] === $job['id'] ) {
					$job_index = $i;
					break;
				}
			}

			if ( $job_index === -1 ) {
				$message = sprintf( '%s -> User synchronization stopped [Job with ID %s] not found.]', __METHOD__, $job['id'] );
				$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
				return new \WP_Error( 'NotFoundException', $message );
			}

			$jobs[ $job_index ] = $job;
			Options_Service::add_update_option( 'user_sync_jobs', $jobs );

			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> Updated WPO365 User Sync job with ID: %s', __METHOD__, $job['id'] ), 'sync' );

			return true;
		}

		/**
		 * Action handler that will check every minute whether there is a stale WPO365 User Synchronization Job (a job is
		 * considered stale if it was last updated 5 minutes ago and no WP Cron Job to execute the next batch has been created).
		 *
		 * @since 28.x
		 *
		 * @param mixed $job_id
		 * @return void
		 */
		public static function user_sync_monitor( $job_id ) {
			$can_custom_log = version_compare( Version::$current, '36.2' ) > 0;
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			// Get latest version of $job by its id
			$jobs = Options_Service::get_global_list_var( 'user_sync_jobs', false );
			$job  = false;

			foreach ( $jobs as $_job ) {

				if ( $_job['id'] === $job_id ) {
					$job = $_job;
					break;
				}
			}

			// Bail out if job was not found
			if ( empty( $job ) ) {
				$message = sprintf( '%s -> Cannot monitor WPO365 User Sync job with ID %s [job empty]', __METHOD__, $job_id );
				$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
				Log_Service::write_log( 'WARN', $message );
				return;
			}

			// Bail out if job.last is not an array
			if ( ! is_array( $job['last'] ) ) {
				$message = sprintf( '%s -> Cannot monitor WPO365 User Sync job with ID %s [job last missing]', __METHOD__, $job_id );
				$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
				Log_Service::write_log( 'WARN', $message );
				return;
			}

			// Unschedule monitor once job.last.stopped is true
			if ( isset( $job['last']['stopped'] ) && $job['last']['stopped'] === true ) {
				$result = wp_clear_scheduled_hook( 'wpo_sync_v2_monitor', array( $job_id ), true );
				$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> Unscheduling WPO365 User Synchronization monitor because the job with ID %s has already been stopped.', __METHOD__, $job_id ), 'sync' );
				return;
			}

			/**
			 * The monitor will run every minute but only after the WPO365 User Sync Job has not been
			 * updated since 300 seconds will it check if the event is still scheduled.
			 */

			if ( time() - $job['last']['date'] >= 300 ) {
				$cron_jobs = _get_cron_array();

				foreach ( $cron_jobs as $timestamp => $_array_of_jobs ) {

					foreach ( $_array_of_jobs as $_hook => $_jobs ) {

						if ( $_hook === 'wpo_sync_v2_users_next' ) {

							foreach ( $_jobs as $_id => $_job ) {

								if ( $job['id'] === $_job['args'][0] ) {
									$next_job_found = true;
									$message        = sprintf( '%s -> Event "wpo_sync_v2_users_next" for WPO365 User Sync Job with ID %s scheduled [nothing to do]', __METHOD__, $job['id'] );
									Log_Service::write_log( 'DEBUG', $message );
									$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
								}
							}
						}
					}
				}

				if ( empty( $next_job_found ) ) {
					Log_Service::write_log( 'ERROR', sprintf( '%s -> Event "wpo_sync_v2_users_next" for WPO365 User Sync Job with ID %s not scheduled [event will be rescheduled]', __METHOD__, $job['id'] ) );
					$random_arg = uniqid();
					$result     = wp_schedule_single_event( time(), 'wpo_sync_v2_users_next', array( $job['id'], null, $random_arg ) );
					$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> Result of rescheduling WPO365 User Synchronization job: %d', __METHOD__, $result ), 'sync' );
				}
			}
		}

		/**
		 * Adds a new WP Cron Job that will check every minute to see if there is a stale
		 * WPO365 User Synchronization Job.
		 *
		 * @since 28.x
		 *
		 * @param mixed $job
		 * @return void
		 */
		public static function schedule_user_sync_monitor( $job ) {
			$can_custom_log = version_compare( Version::$current, '36.2' ) > 0;
			$can_custom_log && Log_Service::write_to_custom_log( sprintf( '##### -> %s', __METHOD__ ), 'sync' );

			if ( empty( $job ) || empty( $job['id'] ) ) {
				$message = sprintf( '%s -> Cannot add WPO365 User Sync Monitor [job is empty or id is missing]', __METHOD__ );
				Log_Service::write_log( 'WARN', $message );
				$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
				return;
			}

			if ( $job['trigger'] === 'externalCron' ) {
				$message = sprintf( '%s -> Will not add WPO365 User Sync Monitor [external cron has been configured]', __METHOD__ );
				Log_Service::write_log( 'DEBUG', $message );
				$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
				return;
			}

			if ( wp_next_scheduled( 'wpo_sync_v2_monitor', array( $job['id'] ) ) === false ) {
				Log_Service::write_log( 'DEBUG', sprintf( '%s -> Adding WPO365 User Sync Monitor', __METHOD__ ) );
				$result = wp_schedule_event( time(), 'wpo_every_minute', 'wpo_sync_v2_monitor', array( $job['id'] ) );
				$can_custom_log && Log_Service::write_to_custom_log( sprintf( '%s -> Result of WPO365 User Synchronization monitor: %d', __METHOD__, $result ), 'sync' );
				return;
			}

			$message = sprintf( '%s -> Cannot add WPO365 User Sync Monitor [job already added]', __METHOD__ );
			$can_custom_log && Log_Service::write_to_custom_log( $message, 'sync' );
			Log_Service::write_log( 'DEBUG', $message );
		}

		/**
		 *
		 * @since 28.x  Restores for each user sync job its "last" cache from the database.
		 *
		 * @param mixed $options
		 * @param mixed $jobs
		 * @return void
		 */
		public static function restore_jobs_last( &$options, $jobs ) {

			if ( is_array( $options['user_sync_jobs'] ) ) {

				foreach ( $jobs as $job ) {

					if ( empty( $job['last'] ) ) {
						continue;
					}

					$job_count = count( $options['user_sync_jobs'] );

					for ( $i = 0; $i < $job_count; $i++ ) {

						if ( $options['user_sync_jobs'][ $i ]['id'] === $job['id'] ) {
							$options['user_sync_jobs'][ $i ]['last'] = $job['last'];
						}
					}
				}
			}
		}
	}
}
