<?php

namespace Wpo\Services;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

use Error;
use Wpo\Core\Permissions_Helpers;
use Wpo\Core\User;
use Wpo\Core\Wpmu_Helpers;
use Wpo\Services\Log_Service;
use Wpo\Services\Options_Service;

if ( ! class_exists( '\Wpo\Services\User_Create_Update_Service' ) ) {

	class User_Create_Update_Service {


		/**
		 * Updates a WordPress user.
		 *
		 * @since   11.0
		 *
		 * @param   User $wpo_usr        WordPress ID of the user that will be updated.
		 * @param   bool $is_deamon      If true then actions that may sign out the user are ignored.
		 * @param   bool $exit_on_error  If true the user my be signed out if an action fails.
		 *
		 * @return  int     The WordPress ID of the user.
		 */
		public static function create_user( &$wpo_usr, $is_deamon = false, $exit_on_error = true ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			$user_login = ! empty( $wpo_usr->preferred_username )
				? $wpo_usr->preferred_username
				: $wpo_usr->upn;

			if ( Options_Service::get_global_boolean_var( 'use_short_login_name' ) || Options_Service::get_global_string_var( 'user_name_preference' ) === 'short' ) {
				$user_login = \stristr( $user_login, '@', true );
			}

			if ( ! empty( $wpo_usr->custom_username ) ) {
				$user_login = $wpo_usr->custom_username;
			}

			if ( ! $is_deamon ) { // Do not apply when synchronizing users

				if ( is_multisite() ) {
					$blog_name = get_bloginfo( 'name' );

					/**
					 * 1. WPMU Shared Mode
					 * 2. Do NOT always add all users to all (sub) sites
					 */

					if ( ! Options_Service::mu_use_subsite_options() && ! Options_Service::get_global_boolean_var( 'mu_add_user_to_all_sites' ) ) {

						if ( is_main_site() ) { // Main Site

							if ( ! Options_Service::get_global_boolean_var( 'create_and_add_users' ) ) {
								Log_Service::write_log( 'WARN', sprintf( '%s -> User %s does not have privileges for site "%s" and is therefore denied access.', __METHOD__, $user_login, $blog_name ) );
								wp_die( sprintf( __( 'You attempted to access "%s", but you do not currently have privileges on this site. If you believe you should be able to access the site, please contact your network administrator.' ), $blog_name ), 403 ); // phpcs:ignore
							}
						} elseif ( Options_Service::get_global_boolean_var( 'skip_add_user_to_subsite' ) ) { // Sub Site
							Log_Service::write_log( 'WARN', sprintf( '%s -> User %s does not have privileges for site "%s" and is therefore denied access.', __METHOD__, $user_login, $blog_name ) );
							wp_die( sprintf( __( 'You attempted to access "%s", but you do not currently have privileges on this site. If you believe you should be able to access the site, please contact your network administrator.' ), $blog_name ), 403 ); // phpcs:ignore
						}
					}

					// WPMU Dedicated Mode
					if ( Options_Service::mu_use_subsite_options() && ! Options_Service::get_global_boolean_var( 'create_and_add_users' ) ) {
						Log_Service::write_log( 'WARN', sprintf( '%s -> User %s does not have privileges for site "%s" and is therefore denied access.', __METHOD__, $user_login, $blog_name ) );
						wp_die( sprintf( __( 'You attempted to access "%s", but you do not currently have privileges on this site. If you believe you should be able to access the site, please contact your network administrator.' ), $blog_name ), 403 ); // phpcs:ignore
					}
				} elseif ( ! Options_Service::get_global_boolean_var( 'create_and_add_users' ) ) {
					Log_Service::write_log( 'ERROR', sprintf( '%s -> User not found and settings prevented creating a new user on-demand for user %s', __METHOD__, $user_login ) );
					Authentication_Service::goodbye( Error_Service::USER_NOT_FOUND, false );
				}
			}

			/**
			 * @since   23.0    Added possibility to hook up (custom) actions to pre-defined events for various WPO365 workloads.
			 */

			do_action(
				'wpo365/user/creating',
				$wpo_usr->preferred_username,
				$wpo_usr->email,
				$wpo_usr->groups
			);

			$usr_default_role = is_main_site()
				? Options_Service::get_global_string_var( 'new_usr_default_role' )
				: Options_Service::get_global_string_var( 'mu_new_usr_default_role' );

			$password_length = Options_Service::get_global_numeric_var( 'password_length' );

			if ( empty( $password_length ) || $password_length < 16 ) {
				$password_length = 16;
			}

			if ( method_exists( '\Wpo\Core\Permissions_Helpers', 'generate_password' ) ) {
				$password = Permissions_Helpers::generate_password( $password_length );
			} else {
				$password = wp_generate_password( $password_length, true, false );
			}

			/**
			 * @since 33.2  Allow developers to filter the user_login.
			 */

			$user_login = apply_filters( 'wpo365/user/user_login', $user_login );

			$userdata = array(
				'user_login'   => $user_login,
				'user_pass'    => $password,
				'display_name' => $wpo_usr->full_name,
				'user_email'   => $wpo_usr->email,
				'first_name'   => $wpo_usr->first_name,
				'last_name'    => $wpo_usr->last_name,
				'role'         => $usr_default_role,
			);

			/**
			 * @since 9.4
			 *
			 * Optionally removing any user_register hooks as these more often than
			 * not interfer and cause unexpected behavior.
			 */

			$user_regiser_hooks = null;

			if ( Options_Service::get_global_boolean_var( 'skip_user_register_action' ) && isset( $GLOBALS['wp_filter'] ) && isset( $GLOBALS['wp_filter']['user_register'] ) ) {
				Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Temporarily removing all filters for the user_register action to avoid interference' );
				$user_regiser_hooks = $GLOBALS['wp_filter']['user_register'];
				unset( $GLOBALS['wp_filter']['user_register'] );
			}

			$existing_registering = remove_filter( 'wp_pre_insert_user_data', '\Wpo\Services\Wp_To_Aad_Create_Update_Service::handle_user_registering', PHP_INT_MAX );
			$existing_registered  = remove_action( 'user_register', '\Wpo\Services\Wp_To_Aad_Create_Update_Service::handle_user_registered', PHP_INT_MAX );

			$wp_usr_id = wp_insert_user( $userdata );

			if ( $existing_registering ) {
				add_filter( 'wp_pre_insert_user_data', '\Wpo\Services\Wp_To_Aad_Create_Update_Service::handle_user_registering', PHP_INT_MAX, 4 );
			}

			if ( $existing_registered ) {
				add_action( 'user_register', '\Wpo\Services\Wp_To_Aad_Create_Update_Service::handle_user_registered', PHP_INT_MAX, 1 );
			}

			if ( ! empty( $GLOBALS['wp_filter'] ) && ! empty( $user_regiser_hooks ) ) {
				$GLOBALS['wp_filter']['user_register'] = $user_regiser_hooks; // phpcs:ignore
			}

			if ( is_wp_error( $wp_usr_id ) ) {
				Log_Service::write_log( 'ERROR', __METHOD__ . ' -> Could not create wp user. See next line for error information.' );
				Log_Service::write_log( 'ERROR', $wp_usr_id );

				if ( $exit_on_error ) {
					Authentication_Service::goodbye( Error_Service::CHECK_LOG, false );
				}

				return 0;
			}

			if ( ! empty( $wpo_usr ) ) {
				User_Service::save_user_principal_name( $wpo_usr->upn, $wp_usr_id );
				User_Service::save_user_tenant_id( $wpo_usr->tid, $wp_usr_id );
				User_Service::save_user_object_id( $wpo_usr->oid, $wp_usr_id );
			}

			/**
			 * @since 15.0
			 */

			do_action( 'wpo365/user/created', $wp_usr_id );

			$wpo_usr->created = true;
			Log_Service::write_log( 'DEBUG', __METHOD__ . ' -> Created new user with ID ' . $wp_usr_id );

			// WPMU -> Add user to all or current blog(s)
			self::wpmu_add_user_to_blog( $wp_usr_id );

			if ( \class_exists( '\Wpo\Services\User_Role_Service' ) && \method_exists( '\Wpo\Services\User_Role_Service', 'update_user_roles' ) ) {
				\Wpo\Services\User_Role_Service::update_user_roles( $wp_usr_id, $wpo_usr );
			}

			add_filter( 'allow_password_reset', '\Wpo\Services\User_Create_Service::temporarily_allow_password_reset', PHP_INT_MAX, 1 );
			wp_new_user_notification( $wp_usr_id, null, 'both' );
			remove_filter( 'allow_password_reset', '\Wpo\Services\User_Create_Service::temporarily_allow_password_reset', PHP_INT_MAX );

			return $wp_usr_id;
		}

		/**
		 * Updates a WordPress user.
		 *
		 * @param   int  $wp_usr_id     WordPress ID of the user that will be updated.
		 * @param   User $wpo_usr        Internal user representation (Graph and ID token data).
		 * @param   bool $is_deamon      If true then actions that may sign out the user are ignored.
		 *
		 * @return  void
		 */
		public static function update_user( $wp_usr_id, $wpo_usr, $is_deamon = false ) {
			Log_Service::write_log( 'DEBUG', '##### -> ' . __METHOD__ );

			if ( ! empty( $wpo_usr ) && empty( $wpo_usr->created ) ) {
				User_Service::save_user_principal_name( $wpo_usr->upn, $wp_usr_id );
				User_Service::save_user_tenant_id( $wpo_usr->tid, $wp_usr_id );
				User_Service::save_user_object_id( $wpo_usr->oid, $wp_usr_id );
			}

			if ( ! $wpo_usr->created ) {

				if ( ! $is_deamon ) {
					self::wpmu_add_user_to_blog( $wp_usr_id );
				}

				if ( \class_exists( '\Wpo\Services\User_Role_Service' ) && \method_exists( '\Wpo\Services\User_Role_Service', 'update_user_roles' ) ) {
					\Wpo\Services\User_Role_Service::update_user_roles( $wp_usr_id, $wpo_usr );
				}
			}

			// Update Avatar
			if ( Options_Service::get_global_boolean_var( 'use_avatar' ) && class_exists( '\Wpo\Services\Avatar_Service' ) ) {
				$default_avatar = get_avatar( $wp_usr_id );
			}

			// Update custom fields
			if ( class_exists( '\Wpo\Services\User_Custom_Fields_Service' ) ) {
				\Wpo\Services\User_Custom_Fields_Service::update_custom_fields( $wp_usr_id, $wpo_usr );
			}

			// Update default WordPress user fields
			self::update_wp_user( $wp_usr_id, $wpo_usr );

			do_action( 'wpo365/user/updated', $wp_usr_id );
		}

		/**
		 * Creates a new WPMU network subsite for a user, e.g. as a personal blog.
		 *
		 * @since 34.x
		 *
		 * @param int $wp_usr_id
		 * @return false|int
		 */
		public static function wpmu_add_new_user_site( $wp_usr_id ) {

			if ( ! is_multisite()
				|| ! Options_Service::get_global_boolean_var( 'mu_add_new_user_site', false )
				|| Options_Service::mu_use_subsite_options()
				|| $wp_usr_id === 0 ) {
					return false;
			}

			$new_user_site_name = strval( $wp_usr_id );

			/**
			 * @since 34.x
			 *
			 * Filters the string that will be used as the new user site's path or subdomain.
			 */
			$new_user_site_name = apply_filters( 'wpo365/wpmu/user_site/name', $new_user_site_name );

			$current_site = get_current_site();
			$wpmu_type    = defined( 'SUBDOMAIN_INSTALL' ) && SUBDOMAIN_INSTALL ? 'domain' : 'path';
			$domain       = $wpmu_type === 'domain' ? $new_user_site_name : $current_site->domain;
			$path         = $wpmu_type === 'path' ? $new_user_site_name : '';
			$title        = sprintf( 'User Blog %s', $new_user_site_name );

			/**
			 * @since 34.x
			 *
			 * Filters the string that will be used as the new user site's title.
			 */
			$title = apply_filters( 'wpo365/wpmu/user_site/title', $title );

			$options = array();

			/**
			 * @since 34.x
			 *
			 * Filters the options for the new user site.
			 */
			$options = apply_filters( 'wpo365/wpmu/user_site/options', $options );

			$blog_id = wpmu_create_blog( $domain, $path, $title, $wp_usr_id, $options, get_current_network_id() );

			if ( is_wp_error( $blog_id ) ) {
				Log_Service::write_log(
					'ERROR',
					sprintf(
						'%s -> Error occurred whilst creating a new user-site [Error: %s]',
						__METHOD__,
						$blog_id->get_error_message()
					)
				);

				return false;
			}

			Log_Service::write_log(
				'DEBUG',
				sprintf(
					'%s -> Created a new user site with domain %s for user with ID %d',
					__METHOD__,
					$new_user_site_name,
					$wp_usr_id
				)
			);

			update_user_meta( $wp_usr_id, 'primary_blog', $blog_id );

			// Check if user has been added to main site and remove user if needed

			if ( ! Options_Service::get_global_boolean_var( 'create_and_add_users' ) ) {
				$main_site_id = get_main_site_id();

				if ( is_user_member_of_blog( $wp_usr_id, $main_site_id ) ) {
					Log_Service::write_log( 'DEBUG', sprintf( '%s -> Removed new user with ID %d from the main site after creating a personal blog site', __METHOD__, $wp_usr_id ) );
					remove_user_from_blog( $wp_usr_id, $main_site_id );
				}
			}

			return $blog_id;
		}

		/**
		 * Helper to add all users to a new subsite (triggered by "wp_initialize_site" action).
		 *
		 * @since 28.x
		 *
		 * @param mixed $site
		 * @return void
		 */
		public static function wpmu_add_users_to_blog( $site, $args ) {
			if ( ! Options_Service::get_global_boolean_var( 'mu_add_user_to_all_sites' ) ) {
				return;
			}

			$args = array(
				'fields' => array( 'ID' ),
			);

			$users = get_users( $args );

			foreach ( $users as $user ) {
				Wpmu_Helpers::wpmu_add_user_to_blog( $user->ID, $site->blog_id, $site->id );
			}
		}

		/**
		 * Helper to add users to all subsites if configured otherwise only to the current site (if allowed).
		 *
		 * @since 28.x
		 *
		 * @param mixed $wp_usr_id
		 * @return void
		 */
		private static function wpmu_add_user_to_blog( $wp_usr_id ) {
			if ( ! method_exists( '\Wpo\Core\Wpmu_Helpers', 'wpmu_add_user_to_blog' ) ) {
				Log_Service::write_log(
					'ERROR',
					sprintf(
						'%s -> Cannot add user to subsite [Error: Please update WPO365 | LOGIN to a version >= 28.x]',
						__METHOD__
					)
				);
				return;
			}

			if ( Options_Service::get_global_boolean_var( 'mu_add_user_to_all_sites' ) ) {
				$sites = get_sites();

				foreach ( $sites as $site ) {
					Wpmu_Helpers::wpmu_add_user_to_blog( $wp_usr_id, $site->blog_id, $site->id );
				}
			} else {
				Wpmu_Helpers::wpmu_add_user_to_blog( $wp_usr_id );
			}
		}

		/**
		 * @since 11.0
		 */
		private static function update_wp_user( $wp_usr_id, $wpo_usr ) {
			$wp_usr = get_user_by( 'ID', $wp_usr_id );

			if ( $wp_usr_id === 0 || $wp_usr === false ) {
				return;
			}

			// Update "core" WP_User fields
			$wp_user_data = array( 'ID' => $wp_usr_id );

			if ( ! empty( $wpo_usr->email ) && ( empty( $wp_usr->user_email ) || strcasecmp( $wpo_usr->email, $wp_usr->user_email ) !== 0 ) ) {
				$wp_user_data['user_email'] = $wpo_usr->email;
			}

			if ( ! empty( $wpo_usr->first_name ) ) {
				$wp_user_data['first_name'] = $wpo_usr->first_name;
			}

			if ( ! empty( $wpo_usr->last_name ) ) {
				$wp_user_data['last_name'] = $wpo_usr->last_name;
			}

			if ( ! empty( $wpo_usr->full_name ) ) {
				$wp_user_data['display_name'] = $wpo_usr->full_name;
			}

			$existing_registering = remove_filter( 'wp_pre_insert_user_data', '\Wpo\Services\Wp_To_Aad_Create_Update_Service::handle_user_registering', PHP_INT_MAX );
			$existing_registered  = remove_action( 'user_register', '\Wpo\Services\Wp_To_Aad_Create_Update_Service::handle_user_registered', PHP_INT_MAX );
			wp_update_user( $wp_user_data );

			if ( $existing_registering ) {
				add_filter( 'wp_pre_insert_user_data', '\Wpo\Services\Wp_To_Aad_Create_Update_Service::handle_user_registering', PHP_INT_MAX, 4 );
			}

			if ( $existing_registered ) {
				add_action( 'user_register', '\Wpo\Services\Wp_To_Aad_Create_Update_Service::handle_user_registered', PHP_INT_MAX, 1 );
			}
		}
	}
}
