<?php

namespace Wpo\Services;

use Error;
use OneLogin_Saml2_Error;
use WP_Post;
use Wpo\Core\Url_Helpers;
use Wpo\Core\WordPress_Helpers;
use Wpo\Services\Authentication_Service;
use Wpo\Services\Options_Service;

// Prevent public access to this script
defined( 'ABSPATH' ) || die();

if ( ! class_exists( '\Wpo\Services\Audiences_Service' ) ) {

	class Audiences_Service {


		/**
		 * Registers the wpo365_audiences post_meta for each (custom) post type.
		 *
		 * @since   16.0
		 *
		 * @return  void
		 */
		public static function aud_register_post_meta() {
			$post_types          = get_post_types();
			$excluded_post_types = Options_Service::get_global_list_var( 'audiences_excluded_post_types', false );

			foreach ( $post_types as $post_type ) {

				if ( \in_array( $post_type, $excluded_post_types, true ) ) {
					continue;
				}

				register_post_meta(
					$post_type,
					'wpo365_audiences',
					array(
						'single'        => false,
						'type'          => 'string',
						'show_in_rest'  => true,
						'auth_callback' => function () {
							return current_user_can( 'delete_posts' );
						},
					)
				);

				register_post_meta(
					$post_type,
					'wpo365_private',
					array(
						'single'        => true,
						'type'          => 'boolean',
						'show_in_rest'  => true,
						'auth_callback' => function () {
							return current_user_can( 'delete_posts' );
						},
					)
				);
			}
		}

		/**
		 * Registers a hook observer for each (supported) post / page type that will delete
		 * the audience post_meta if an Audience block is not found.
		 *
		 * @since   16.0
		 *
		 * @return  void
		 */
		public static function post_updated( $post_id, $post_after, $post_before ) {
			// Placeholder for backward compatibility
		}

		/**
		 * Updates the user's audience assignments (if any) based on the user's
		 * Azure AD group membership(s) and the configured audiences.
		 *
		 * @since   16.0
		 *
		 * @param   string $wp_usr_id  The ID of the user.
		 * @param   User   $wpo_usr    The internal (Microsft Graph) based user representation.
		 *
		 * @return  void
		 */
		public static function aad_group_x_audience( $wp_usr_id, $wpo_usr ) {
			if ( count( $wpo_usr->groups ) === 0 ) {
				return;
			}

			$audiences      = Options_Service::get_global_list_var( 'audiences' );
			$user_audiences = array();

			foreach ( $audiences as $audience ) {

				// Just to be sure that the configuration object valid
				if ( empty( $audience['values'] ) || empty( $audience['key'] ) ) {
					continue;
				}

				// Iterate over all the Azure AD group IDs added to this audience
				foreach ( $audience['values'] as $value ) {

					// User is member of a group that is mapped to an audience
					if ( in_array( $value, $user_audiences, true ) === false && array_key_exists( $value, $wpo_usr->groups ) === true ) {
						$user_audiences[] = $audience['key'];
					}
				}
			}

			update_user_meta( $wp_usr_id, 'wpo365_audiences', $user_audiences );
		}

		/**
		 * Stub for backward compatibility.
		 *
		 * @deprecated Since version 24.0
		 *
		 * @param mixed $posts
		 * @param mixed $query
		 * @return mixed
		 */
		public static function the_posts( $posts, $query ) { // phpcs:ignore
			return $posts;
		}

		/**
		 * Hooked into
		 *
		 * @since   16.0
		 *
		 * @see     ...
		 */
		public static function posts_where( $where, $query ) { // phpcs:ignore
			// Check if we need to skip audiences for the current user's role
			$wp_usr       = \wp_get_current_user();
			$wp_usr_roles = empty( $wp_usr ) ? array() : $wp_usr->roles;

			$audiences_excluded_roles_raw  = Options_Service::get_global_list_var( 'audiences_excluded_roles', false );
			$audiences_excluded_post_types = array();

			array_map(
				function ( $item ) use ( $wp_usr_roles, &$audiences_excluded_post_types ) {
					$splitted = explode( '|', $item );

					if ( in_array( $splitted[0], $wp_usr_roles, true ) ) {
						// Backward compatibility: any if no | was found
						$audiences_excluded_post_type    = count( $splitted ) > 1 ? $splitted[1] : 'any';
						$audiences_excluded_post_types[] = $audiences_excluded_post_type;
					}
				},
				$audiences_excluded_roles_raw
			);

			// Bail out early if user has a role that is excluded from all audience-based restrictions
			if ( in_array( 'any', $audiences_excluded_post_types, true ) ) {
				return $where;
			}

			if ( empty( $audiences_excluded_post_types ) ) {
				$audiences_excluded_post_types_str = "'_placeholder_post_type_'";
			} else {
				$_audiences_excluded_post_types    = array_map(
					function ( $item ) {
						return "'$item'";
					},
					$audiences_excluded_post_types
				);
				$audiences_excluded_post_types_str = implode( $_audiences_excluded_post_types );
			}

			global $wpdb;

			$request_service = Request_Service::get_instance();
			$request         = $request_service->get_request( $GLOBALS['WPO_CONFIG']['request_id'] );
			$audience_cache  = $request->get_item( 'audience_cache' );

			if ( empty( $audience_cache ) ) {
				$audience_cache = array(
					'aud_audiences_where'     => null,
					'aud_private_pages_where' => null,
				);
				$db_posts       = $wpdb->prefix . 'posts';
			}

			// Handle audiences

			if ( $audience_cache['aud_audiences_where'] === null ) {
				$audience_cache['aud_audiences_where'] = '';
				$audiences                             = Options_Service::get_global_list_var( 'audiences', false );
				$audiences_x_post_types                = Options_Service::get_global_list_var( 'audiences_x_post_types', false );
				$user_audiences                        = \get_user_meta( $wp_usr->ID, 'wpo365_audiences', true );
				$db_postmeta                           = $wpdb->prefix . 'postmeta';
				$db_posts                              = $wpdb->prefix . 'posts';

				if ( empty( $user_audiences ) ) {
					// 1. User is not in any audience -> Remove all posts with audiences
					$audience_cache['aud_audiences_where'] .= " AND $db_posts.ID NOT IN (SELECT post_id FROM $db_postmeta AS meta JOIN $db_posts as posts ON meta.post_id = posts.ID WHERE meta_key = 'wpo365_audiences' AND meta_value IS NOT NULL AND meta_value != '' AND posts.post_type NOT IN ($audiences_excluded_post_types_str)) ";

					// 2. User is not in any audience -> Remove all posts of types that have audiences defined for them
					$post_type_clause = '';
					$post_type_andor  = '';

					foreach ( $audiences_x_post_types as $audiences_x_post_type ) {

						// For the user in question the post type has been excluded based on their role
						if ( in_array( $audiences_x_post_type['value'], $audiences_excluded_post_types, true ) ) {
							continue;
						}

						$post_type_clause .= " $post_type_andor post_type = '" . $audiences_x_post_type['value'] . "' ";
						$post_type_andor   = 'OR';
					}

					if ( ! empty( $post_type_clause ) ) {
						$audience_cache['aud_audiences_where'] .= " AND $db_posts.ID NOT IN (SELECT ID FROM $db_posts WHERE $post_type_clause) ";
					}
				} else {
					$meta_value_clause = '';
					$meta_value_andor  = '';

					foreach ( $user_audiences as $user_audience ) {
						$meta_value_clause .= " $meta_value_andor meta_value = '" . $user_audience . "' ";
						$meta_value_andor   = 'OR';
					}

					$audience_cache['aud_audiences_where'] .= " AND $db_posts.ID NOT IN (SELECT DISTINCT post_id FROM $db_postmeta AS meta JOIN $db_posts as posts ON meta.post_id = posts.ID WHERE meta_key = 'wpo365_audiences' AND meta_value IS NOT NULL AND meta_value != '' AND posts.post_type NOT IN ($audiences_excluded_post_types_str) AND post_id NOT IN (SELECT DISTINCT post_id FROM $db_postmeta WHERE meta_key = 'wpo365_audiences' AND ($meta_value_clause))) ";

					$restricted_post_types = array();

					foreach ( $audiences_x_post_types as $audiences_x_post_type_a ) {

						// 1. The post type has been excluded for the current user
						if ( in_array( $audiences_x_post_type_a['value'], $audiences_excluded_post_types, true ) ) {
							continue;
						}

						// 2. This rule would remove the post type for the user
						if ( ! in_array( $audiences_x_post_type_a['key'], $user_audiences, true ) ) {
							$allowed_post_type = false;

							// 2. But we need to be sure that there is not another mapping for this post type to an audience that the user is a member of
							foreach ( $audiences_x_post_types as $audiences_x_post_type_b ) {

								if ( $audiences_x_post_type_b['value'] === $audiences_x_post_type_a['value'] && in_array( $audiences_x_post_type_b['key'], $user_audiences, true ) ) {
									$allowed_post_type = true;
								}
							}

							if ( $allowed_post_type ) {
								continue;
							}

							if ( ! in_array( $audiences_x_post_type_a['value'], $restricted_post_types, true ) ) {
								$restricted_post_types[] = $audiences_x_post_type_a['value'];
							}
						}
					}

					$post_type_clause = '';
					$post_type_andor  = '';

					foreach ( $restricted_post_types as $restricted_post_type ) {
						$post_type_clause .= " $post_type_andor post_type = '" . $restricted_post_type . "' ";
						$post_type_andor   = 'OR';
					}

					if ( ! empty( $post_type_clause ) ) {
						$audience_cache['aud_audiences_where'] .= " AND $db_posts.ID NOT IN (SELECT ID FROM $db_posts WHERE $post_type_clause) ";
					}
				}
			}

			// Handle pages marked private

			if ( method_exists( '\Wpo\Services\Auth_Only_Service', 'just_validate_auth_cookie' ) ) {
				$is_auth_only = \Wpo\Services\Auth_Only_Service::just_validate_auth_cookie();
			}

			if ( $wp_usr->ID === 0 && empty( $is_auth_only ) ) {

				if ( $audience_cache['aud_private_pages_where'] === null ) {
					$audience_cache['aud_private_pages_where'] = '';
					$auth_scenario                             = Options_Service::get_global_string_var( 'auth_scenario', false );

					if ( $auth_scenario === 'internet' || $auth_scenario === 'internetAuthOnly' ) {
						$audience_cache['aud_private_pages_where'] .= " AND ( $db_posts.ID NOT IN (SELECT post_id FROM $db_postmeta WHERE meta_key = 'wpo365_private' AND meta_value IS NOT NULL AND meta_value = 1 )) ";
						$private_post_types                         = Options_Service::get_global_list_var( 'audiences_private_post_types' );

						if ( ! empty( $private_post_types ) ) {
							$post_type_clause = '';
							$post_type_andor  = '';

							foreach ( $private_post_types as $private_post_type ) {

								$post_type_clause .= " $post_type_andor post_type = '" . $private_post_type . "'";
								$post_type_andor   = 'OR';
							}

							$audience_cache['aud_private_pages_where'] .= " AND ( $db_posts.ID NOT IN (SELECT ID FROM $db_posts WHERE $post_type_clause )) ";
						}
					}
				}
			}

			if ( $audience_cache['aud_audiences_where'] !== null ) {
				$where .= $audience_cache['aud_audiences_where'];
			}

			if ( $audience_cache['aud_private_pages_where'] !== null ) {
				$where .= $audience_cache['aud_private_pages_where'];
			}

			$request->set_item( 'audience_cache', $audience_cache );
			return $where;
		}

		/**
		 * Hooked into
		 *
		 * @since   16.0
		 *
		 * @see     ...
		 */
		public static function get_pages( $pages, $parsed_args ) { // phpcs:ignore
			$wp_usr = \wp_get_current_user();
			$result = array();

			foreach ( $pages as $page ) {

				if ( self::user_can_read( $page->ID, $wp_usr->ID ) ) {
					$result[] = $page;
				}
			}

			return $result;
		}

		/**
		 * Hooked into
		 *
		 * @since   16.0
		 *
		 * @see     ...
		 */
		public static function wp_count_posts( $counts, $type, $perm ) { // phpcs:ignore
			$excluded_post_types = Options_Service::get_global_list_var( 'audiences_excluded_post_types', false );

			if ( \in_array( $type, $excluded_post_types, true ) ) {
				return $counts;
			}

			foreach ( $counts as $post_status => $count ) {
				$query_args = array(
					'fields'           => 'ids',
					'post_type'        => $type,
					'post_status'      => $post_status,
					'numberposts'      => -1,
					'suppress_filters' => 0,
					'orderby'          => 'none',
					'no_found_rows'    => true,
					'nopaging'         => true,
				);

				$posts = get_posts( $query_args );
				$count = count( $posts );
				unset( $posts );
				$counts->$post_status = $count;
			}

			return $counts;
		}

		/**
		 * Hooked into get_adjacent_post_where filter.
		 *
		 * @since   16.0
		 *
		 * @see     https://developer.wordpress.org/reference/hooks/get_adjacent_post_where/
		 */
		public static function get_previous_post_where( $where, $in_same_term, $excluded_terms, $taxonomy, $post ) {
			return self::get_next_post_where( $where, $in_same_term, $excluded_terms, $taxonomy, $post );
		}

		/**
		 * Hooked into get_adjacent_post_where filter.
		 *
		 * @since   16.0
		 *
		 * @see     https://developer.wordpress.org/reference/hooks/get_adjacent_post_where/
		 */
		public static function get_next_post_where( $where, $in_same_term, $excluded_terms, $taxonomy, $post ) {
			if ( ! empty( $post ) ) {

				$excluded_post_types = Options_Service::get_global_list_var( 'audiences_excluded_post_types', false );

				if ( in_array( $post->post_type, $excluded_post_types, true ) ) {
					return $where;
				}

				$post_ids = get_posts(
					array(
						'post_type'        => $post->post_type,
						'numberposts'      => -1,
						'suppress_filters' => false,
						'fields'           => 'ids',
					)
				);

				if ( is_array( $post_ids ) && count( $post_ids ) > 0 ) {
					$post_ids  = array_map( 'intval', $post_ids );
					$condition = ' p.ID IN (' . implode( ',', $post_ids ) . ') ';

					if ( ! empty( $where ) ) {
						$where .= ' AND ' . $condition;
					} else {
						$where = ' WHERE ' . $condition;
					}
				}
			}

			return $where;
		}

		/**
		 * Hooked into the rest_prepare_%post_type% filter.
		 *
		 * @since   16.0
		 *
		 * @see     https://developer.wordpress.org/reference/hooks/rest_prepare_this-post_type/
		 */
		public static function rest_prepare_post( $response, $post, $request ) { // phpcs:ignore
			$wp_usr = \wp_get_current_user();

			if ( self::skip_audiences( $wp_usr, $post->post_type ) ) {
				return $response;
			}

			$excluded_post_types = Options_Service::get_global_list_var( 'audiences_excluded_post_types', false );

			if ( in_array( $post->post_type, $excluded_post_types, true ) ) {
				return $response;
			}

			if ( isset( $post->ID ) && ! self::user_can_read( $post->ID, $wp_usr->ID ) ) {
				$response = array(
					'code'    => 'rest_post_invalid_id',
					'message' => __( 'Invalid post ID.' ),
					'data'    => array( 'status' => 404 ),
				);
			}

			return $response;
		}

		/**
		 * Helper to register a custom column to show a user's audiences on the default WordPress Users screen.
		 *
		 * @since   16.0
		 *
		 * @param   Array $columns Array of columns.
		 *
		 * @return  Arry    Array of colums with optionally the "Audiences" column addded.
		 */
		public static function register_users_audiences_column( $columns ) {
			$columns['wpo365_audiences'] = 'Audiences';
			return $columns;
		}

		/**
		 * Helper to render the custom "Audiences" column that is added to the default WordPress Users screen.
		 *
		 * @since   16.0
		 *
		 * @param   string $output         Rendered HTML.
		 * @param   string $column_name    Name of the column being rendered.
		 * @param   string $user_id        ID of the user the column's cell is being rendered for.
		 *
		 * @return  string  Rendered HTML.
		 */
		public static function render_users_audiences_column( $output, $column_name, $user_id ) {
			if ( $column_name === 'wpo365_audiences' ) {
				$audiences      = Options_Service::get_global_list_var( 'audiences', false );
				$user_audiences = \get_user_meta( $user_id, 'wpo365_audiences', true );

				// Not all users necessary have audiences defined for them
				if ( empty( $user_audiences ) ) {
					return $output;
				}

				$output = '<div>';

				foreach ( $user_audiences as $user_audience ) {

					foreach ( $audiences as $audience ) {

						if ( $audience['key'] === $user_audience ) {

							if ( $output !== '<div>' ) {
								$output .= '<br/>';
							}

							$output .= '<span>' . \esc_html( $audience['title'] ) . '</span>';
							break;
						}
					}
				}

				$output .= '</div>';
			}

			return $output;
		}

		/**
		 * Helper to register a custom column to show the configured audiences for a (custom) post or page on the default WordPress Posts and Pages screens.
		 *
		 * @since   28.x
		 *
		 * @param   Array $columns Array of columns.
		 *
		 * @return  Array    Array of colums with optionally the "Audiences" column addded.
		 */
		public static function register_posts_audiences_column( $columns, $post_type = 'page' ) { // phpcs:ignore
			$columns['wpo365_audiences'] = 'Audiences';
			$columns['wpo365_private']   = 'Requires login';
			return $columns;
		}

		/**
		 * Helper to render the custom "Audiences" column that is added to the default WordPress Posts and Pages screens.
		 *
		 * @since   28.0
		 *
		 * @param   string $column_name    Name of the column being rendered.
		 * @param   string $post_id        ID of the user the column's cell is being rendered for.
		 *
		 * @return  void
		 */
		public static function render_posts_audiences_column( $column_name, $post_id ) {
			if ( $column_name === 'wpo365_audiences' ) {
				$audiences        = Options_Service::get_global_list_var( 'audiences', false );
				$audiences_count  = count( $audiences );
				$tagged_audiences = \get_post_meta( $post_id, 'wpo365_audiences', false );
				$post_audiences   = array();

				if ( is_array( $tagged_audiences ) ) {

					foreach ( $tagged_audiences as $tagged_audience ) {

						for ( $i = 0; $i < $audiences_count; $i++ ) {

							if ( $audiences[ $i ]['key'] === $tagged_audience ) {
								$post_audiences[] = $audiences[ $i ]['title'];
								break;
							}
						}
					}
				}

				$post_type             = get_post_type( $post_id );
				$audience_x_post_types = Options_Service::get_global_list_var( 'audiences_x_post_types', false );

				foreach ( $audience_x_post_types as $audience_x_post_type ) {

					if ( $audience_x_post_type['value'] === $post_type ) {

						for ( $i = 0; $i < $audiences_count; $i++ ) {

							if ( $audiences[ $i ]['key'] === $audience_x_post_type['key'] ) {
								$post_audiences[] = $audiences[ $i ]['title'];
								break;
							}
						}
					}
				}

				$output = '<ul>';

				foreach ( $post_audiences as $post_audience ) {
					$output .= "<li>$post_audience</li>";
				}

				$output .= '</ul>';
				echo wp_kses( $output, WordPress_Helpers::get_allowed_html() );
			}

			if ( $column_name === 'wpo365_private' ) {
				$private_post_types = Options_Service::get_global_list_var( 'audiences_private_post_types', false );
				$is_private         = \get_post_meta( $post_id, 'wpo365_private', true );
				$post_type          = get_post_type( $post_id );

				$output  = '<div>';
				$output .= in_array( $post_type, $private_post_types, true ) === true || $is_private ?
					'<input name="admin_bar_front" type="checkbox" id="admin_bar_front" value="1" checked="checked" disabled>' :
					'';
				$output .= '</div>';
				echo wp_kses( $output, WordPress_Helpers::get_allowed_html() );
			}
		}

		/**
		 * Hooks into the add_meta_boxes action to render the custom Audiences meta box.
		 *
		 * @since   19.0
		 *
		 * @param   mixed $post_type
		 * @param   mixed $post
		 * @return  void
		 */
		public static function audiences_add_meta_box( $post_type, $post ) {
			// Do nothing if the current user cannot edit the post
			if ( ! is_admin() || ! current_user_can( 'edit_post', $post->ID ) ) {
				return;
			}

			$excluded_post_types = Options_Service::get_global_list_var( 'audiences_excluded_post_types', false );

			// Do nothing if $post_type is excluded from audiences
			if ( \in_array( $post_type, $excluded_post_types, true ) ) {
				return;
			}

			add_meta_box( 'audiences_meta_box', __( 'WPO365 Audiences', 'wpo365_login' ), '\Wpo\Services\Audiences_Service::audiences_render_meta_box', $post_type, 'side', 'low' );
		}

		/**
		 * Renders the custom (HTML for the) meta box for Audiences.
		 *
		 * @since   19.0
		 *
		 * @return  void
		 */
		public static function audiences_render_meta_box( $post ) {
			wp_nonce_field( basename( __FILE__ ), 'audiences_meta_box_nonce' );
			$audiences = Options_Service::get_global_list_var( 'audiences', false );
			array_multisort( array_column( $audiences, 'title' ), SORT_ASC, $audiences );
			$current_audiences = get_post_meta( $post->ID, 'wpo365_audiences', false );
			$currently_private = ! empty( get_post_meta( $post->ID, 'wpo365_private', true ) );

			if ( empty( $current_audiences ) ) {
				$current_audiences = array();
			}

			$auth_scenario = Options_Service::get_global_string_var( 'auth_scenario', false );

			if ( WordPress_Helpers::stripos( $auth_scenario, 'internet' ) !== false || $currently_private ) {
				echo '<p>You can make this content exclusively available for users that logged into your website (e.g. with Microsoft) by checking the box below.</p>';
				printf(
					'<input id="wpo365Private" type="checkbox" onclick="javascript:if(document.getElementById(\'wpo365Private\').checked){document.getElementById(\'wpo365Audiences\').style.display = \'none\'}else{document.getElementById(\'wpo365Audiences\').style.display = \'initial\'}" name="wpo365_private" %s />%s<br />',
					$currently_private === true ? 'checked' : '',
					'Make private'
				);
				echo '<p>&nbsp;</p>';
			}

			printf(
				'<div id="wpo365Audiences" style="display: %s">',
				$currently_private ? 'none' : 'initial'
			);

			printf(
				'<p>%s can make this content exclusively available for users that are a member of one of the Audiences checked below.</p>',
				WordPress_Helpers::stripos( $auth_scenario, 'internet' ) !== false ? 'Alternatively, you' : 'You'
			);

			foreach ( $audiences as $audience ) {
				printf(
					'<input type="checkbox" name="wpo365_audiences[]" value="%s" %s />%s<br />',
					wp_kses( $audience['key'], WordPress_Helpers::get_allowed_html() ),
					in_array( $audience['key'], $current_audiences, true ) ? 'checked' : '',
					wp_kses( $audience['title'], WordPress_Helpers::get_allowed_html() )
				);
			}

			echo '</div><div></div>';
		}

		/**
		 * Helper to save the audiences related post meta.
		 *
		 * @since   19.0
		 *
		 * @param   mixed $post_id
		 * @param   mixed $post
		 * @param   bool  $update
		 * @return  void
		 */
		public static function audiences_save_post( $post_id, $post, $update ) { // phpcs:ignore
			// Do nothing if the current user cannot edit the post
			if ( ! is_admin() || ! current_user_can( 'edit_post', $post->ID ) ) {
				return;
			}

			// Verify meta box nonce
			if ( ! isset( $_POST['audiences_meta_box_nonce'] ) || ! wp_verify_nonce( $_POST['audiences_meta_box_nonce'], basename( __FILE__ ) ) ) { // phpcs:ignore
				return;
			}

			// Always delete the post meta first
			delete_post_meta( $post_id, 'wpo365_audiences' );
			delete_post_meta( $post_id, 'wpo365_private' );

			if ( isset( $_POST['wpo365_audiences'] ) ) {
				$configured_audiences = $_POST['wpo365_audiences']; // phpcs:ignore
				$configured_audiences = array_map( 'sanitize_text_field', $configured_audiences );

				foreach ( $configured_audiences as $configured_audience ) {
					add_post_meta( $post_id, 'wpo365_audiences', $configured_audience, false );
				}
			}

			if ( isset( $_POST['wpo365_private'] ) && WordPress_Helpers::stripos( sanitize_key( $_POST['wpo365_private'] ), 'on' ) !== false ) {
				add_post_meta( $post_id, 'wpo365_private', true, true );
			}
		}

		/**
		 * Hooks into the map_meta_cap function to prevent a user to directly edit a post or page
		 * if he / she is not entitled to do so in the context of audiences.
		 *
		 * @since   19.4
		 *
		 * @param   mixed $caps
		 * @param   mixed $cap
		 * @param   mixed $user_id
		 * @param   mixed $args
		 * @return  mixed
		 */
		public static function map_meta_cap( $caps, $cap, $user_id, $args ) {
			$current_user = wp_get_current_user();

			if ( ! empty( $GLOBALS['post'] ) && $GLOBALS['post'] instanceof WP_Post && ! empty( $args[0] ) ) {

				if ( WordPress_Helpers::stripos( $cap, 'edit_' ) === 0 || WordPress_Helpers::stripos( $cap, 'delete_' ) === 0 ) {
					$post_id = null;

					if ( is_numeric( $args[0] ) ) {
						$post_id = $args[0];
					} elseif ( $args[0] instanceof WP_Post ) {
						$post_id = $args[0]->ID;
					}

					$post_type = get_post_type( $post_id );

					if ( self::skip_audiences( $current_user, $post_type ) ) {
						return $caps;
					}

					if ( $post_id && $GLOBALS['post']->ID === $post_id ) {

						if ( ! self::user_can_read( $post_id, $user_id ) ) {
							$caps[] = 'do_not_allow';
						}
					}
				}
			}

			return $caps;
		}

		/**
		 * Hooks into the pre_handle_404 hook to redirect visitors that are not logged-in and that request a page that requires a
		 * logged-in user to the login page or to Microsoft.
		 *
		 * @param string $status_header
		 * @param int    $code
		 * @param string $description
		 * @param string $protocol
		 * @return string
		 */
		public static function handle_404( $status_header, $code, $description, $protocol ) {  // phpcs:ignore
			// Do not continue if no 404 has been issued yet.
			if ( $code !== 404 ) {
				return $status_header;
			}

			// Checking for private pages only makes sense if the user is not yet logged in.
			if ( is_user_logged_in() ) {
				return $status_header;
			}

			remove_filter( 'posts_where', '\Wpo\Services\Audiences_Service::posts_where', 10, 2 );
			$post_id = url_to_postid( $GLOBALS['WPO_CONFIG']['url_info']['current_url'] );
			add_filter( 'posts_where', '\Wpo\Services\Audiences_Service::posts_where', 10, 2 );

			if ( ! empty( $post_id ) ) {
				$post_type              = get_post_type( $post_id );
				$private_post_types     = Options_Service::get_global_list_var( 'audiences_private_post_types', false );
				$is_private             = \get_post_meta( $post_id, 'wpo365_private', true );
				$audiences_x_post_types = array_map(
					function ( $item ) {
						return $item['value'];
					},
					Options_Service::get_global_list_var( 'audiences_x_post_types', false )
				);
				$has_audiences          = ! empty( \get_post_meta( $post_id, 'wpo365_audiences', false ) ) || in_array( $post_type, $audiences_x_post_types, true );

				if ( filter_var( $is_private, FILTER_VALIDATE_BOOLEAN ) === true || in_array( $post_type, $private_post_types, true ) === true || $has_audiences ) {
					$private_page_response = Options_Service::get_global_string_var( 'audiences_private_page_response' );

					if ( $private_page_response === 'loginPage' ) {
						$redirect_url = Options_Service::get_aad_option( 'redirect_url' );
						$redirect_url = Options_Service::get_global_boolean_var( 'use_saml' )
							? Options_Service::get_aad_option( 'saml_sp_acs_url' )
							: $redirect_url;
						$redirect_url = apply_filters( 'wpo365/aad/redirect_uri', $redirect_url );
						$referer      = ( WordPress_Helpers::stripos( $redirect_url, 'https' ) !== false ? 'https' : 'http' ) . '://' . $GLOBALS['WPO_CONFIG']['url_info']['host'] . $GLOBALS['WPO_CONFIG']['url_info']['request_uri'];
						$login_url    = Url_Helpers::get_preferred_login_url();
						$login_url    = add_query_arg( 'login_errors', 'PRIVATE_PAGE', $login_url );
						$login_url    = add_query_arg( 'redirect_to', $referer, $login_url );
						Url_Helpers::force_redirect( $login_url );
					}

					if ( $private_page_response === 'microsoft' ) {
						Authentication_Service::redirect_to_microsoft();
					}
				}
			}

			return $status_header;
		}

		/**
		 * Helper to check if the current user can read a specific post. A user can read a specific content item when:
		 *
		 * 1. No audiences are defined and the page is not marked as private
		 * 2. No audiences are defined and the page is marked as private and the user is logged in
		 * 3. Audiences are defined and the user is logged in and is in one of the audiences added to the content
		 *
		 * @since   16.0
		 *
		 * @param   int     $post_id    The ID of the post to check.
		 * @param   int     $wp_usr_id  The ID of the user.
		 *
		 * @return  bool    True if the user may read the content according to WPO365 audience rules.
		 */
		private static function user_can_read( $post_id, $wp_usr_id ) {
			$current_user = wp_get_current_user();
			$post_type    = get_post_type( $post_id );

			if ( self::skip_audiences( $current_user, $post_type ) ) {
				return true;
			}

			$is_private     = \get_post_meta( $post_id, 'wpo365_private', true );
			$post_audiences = \get_post_meta( $post_id, 'wpo365_audiences', false );
			$user_audiences = \get_user_meta( $wp_usr_id, 'wpo365_audiences', true );

			// Handle audiences

			if ( ! empty( $post_audiences ) ) {

				// Post has audiences defined for it but user is not signed-in
				if ( $wp_usr_id === 0 ) {
					return false;
				}

				// User is signed in but is not in any audience
				if ( empty( $user_audiences ) ) {
					return false;
				}

				foreach ( $post_audiences as $post_audience ) {

					// User is signed in and in one of the defined audiences
					if ( \in_array( $post_audience, $user_audiences, true ) ) {
						return true;
					}
				}

				return false;
			}

			$audiences_x_post_types = Options_Service::get_global_list_var( 'audiences_x_post_types', false );

			if ( ! empty( $audiences_x_post_types ) ) {

				foreach ( $audiences_x_post_types as $kv_pair ) {

					if ( $kv_pair['value'] === $post_type ) {

						if ( ! in_array( $kv_pair['key'], $user_audiences, true ) ) {
							return false;
						}
					}
				}
			} elseif ( filter_var( $is_private, FILTER_VALIDATE_BOOLEAN ) === true ) { // Handle content is private

				// User is logged in thus may he / she see the post
				if ( $wp_usr_id > 0 ) {
					return true;
				}

				return false;
			}

			return true;
		}

		/**
		 * Helper to check if audiences need to be applied at all. Reasons not the apply audiences are:
		 *
		 * 1. The administrator disabled support for REST and the current request is a json request
		 * 2. When the current user's role has been excluded from audiences.
		 *
		 * @since 16.0
		 *
		 * @param   WP_User $wp_usr
		 *
		 * @return  bool    True if audiences can be skipped otherwise false.
		 */
		private static function skip_audiences( $wp_usr, $post_type = 'any' ) {
			// Check if current request is json request

			if ( defined( 'WPO365_REST_REQUEST' ) && WPO365_REST_REQUEST && Options_Service::get_global_boolean_var( 'enable_audiences_rest', false ) === false ) {
				return true;
			}

			// Check if we need to skip audiences for the current user's role

			$wp_usr_roles = empty( $wp_usr ) ? array() : $wp_usr->roles;

			$audiences_excluded_roles_raw  = Options_Service::get_global_list_var( 'audiences_excluded_roles', false );
			$audiences_excluded_post_types = array();

			array_map(
				function ( $item ) use ( $wp_usr_roles, &$audiences_excluded_post_types ) {
					$splitted = explode( '|', $item );

					if ( in_array( $splitted[0], $wp_usr_roles, true ) ) {
						// Backward compatibility: any if no | was found
						$audiences_excluded_post_type    = count( $splitted ) > 1 ? $splitted[1] : 'any';
						$audiences_excluded_post_types[] = $audiences_excluded_post_type;
					}
				},
				$audiences_excluded_roles_raw
			);

			foreach ( $audiences_excluded_post_types as $audiences_excluded_post_type ) {

				if ( $audiences_excluded_post_type === 'any' || strcasecmp( $audiences_excluded_post_type, $post_type ) === 0 ) {
					return true;
				}
			}

			return false;
		}
	}
}
