<?php

namespace WPSocialReviews\App\Http\Controllers\Platforms;

use WPSocialReviews\App\Http\Controllers\Controller;
use WPSocialReviews\Framework\Request\Request;

class AddonController extends Controller
{

    public function activePlugin(Request $request)
    {
        $platform = sanitize_text_field($request->get('platform', ''));
        if ($platform === 'tiktok') {
            $this->handleNinjaTikTokInstall();
        }
        if( $platform === 'fluent_forms' ) {
            $this->handleFluentFormInstall();
        }
    }

    public function handleFluentFormInstall()
    {
        if (!current_user_can('install_plugins')) {
            return $this->sendError([
                'message' => __('Sorry! you do not have permission to install plugin', 'wp-social-reviews')
            ]);
        }

        $this->installFluentForm();

        wp_send_json_success([
            'message' => __('Fluent Forms has been installed and activated', 'wp-social-reviews'),
            'is_installed' => defined('FLUENTFORM'),
        ]);
    }

    public function handleNinjaTikTokInstall()
    {
        if (!current_user_can('install_plugins')) {
            return $this->sendError([
                'message' => __('Sorry! you do not have permission to install plugin', 'wp-social-reviews')
            ]);
        }

        $this->installCustomTikTokFeed();

        wp_send_json_success([
            'message' => __('Custom Feed for TikTok Plugin activated successfully', 'wp-social-reviews'),
            'is_installed' => defined('CUSTOM_FEED_FOR_TIKTOK'),
        ]);
    }

    private function installCustomTikTokFeed()
    {
        $plugin_id = 'custom-feed-for-tiktok';
        $plugin = [
            'name'      => __('Custom Feed for TikTok', 'wp-social-reviews'),
            'repo-slug' => 'custom-feed-for-tiktok',
            'file'      => 'custom-feed-for-tiktok.php',
        ];
        $this->backgroundInstaller($plugin, $plugin_id);
    }

    private function installFluentForm()
    {
        $plugin_id = 'fluentform';
        $plugin = [
            'name'      => __('Fluent Forms', 'wp-social-reviews'),
            'repo-slug' => 'fluentform',
            'file'      => 'fluentform.php',
        ];
        $this->backgroundInstaller($plugin, $plugin_id);
    }

    private function backgroundInstaller($plugin_to_install, $plugin_id)
    {
        if (!empty($plugin_to_install['repo-slug'])) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
            require_once ABSPATH . 'wp-admin/includes/plugin.php';

            WP_Filesystem();

            $skin = new \Automatic_Upgrader_Skin();
            $upgrader = new \WP_Upgrader($skin);
            $installed_plugins = array_reduce(array_keys(\get_plugins()), array($this, 'associatePluginFile'), array());
            $plugin_slug = $plugin_to_install['repo-slug'];
            $plugin_file = isset($plugin_to_install['file']) ? $plugin_to_install['file'] : $plugin_slug . '.php';
            $installed = false;
            $activate = false;

            // See if the plugin is installed already.
            if (isset($installed_plugins[$plugin_file])) {
                $installed = true;
                $activate = !is_plugin_active($installed_plugins[$plugin_file]);
            }

            if (!$installed) {
                ob_start();

                try {
                    $plugin_information = plugins_api(
                        'plugin_information',
                        array(
                            'slug'   => $plugin_slug,
                            'fields' => array(
                                'short_description' => false,
                                'sections'          => false,
                                'requires'          => false,
                                'rating'            => false,
                                'ratings'           => false,
                                'downloaded'        => false,
                                'last_updated'      => false,
                                'added'             => false,
                                'tags'              => false,
                                'homepage'          => false,
                                'donate_link'       => false,
                                'author_profile'    => false,
                                'author'            => false,
                            ),
                        )
                    );

                    if (is_wp_error($plugin_information)) {
                        throw new \Exception($plugin_information->get_error_message());
                    }

                    $package = $plugin_information->download_link;
                    $download = $upgrader->download_package($package);

                    if (is_wp_error($download)) {
                        throw new \Exception($download->get_error_message());
                    }

                    $working_dir = $upgrader->unpack_package($download, true);

                    if (is_wp_error($working_dir)) {
                        throw new \Exception($working_dir->get_error_message());
                    }

                    $result = $upgrader->install_package(
                        array(
                            'source'                      => $working_dir,
                            'destination'                 => WP_PLUGIN_DIR,
                            'clear_destination'           => false,
                            'abort_if_destination_exists' => false,
                            'clear_working'               => true,
                            'hook_extra'                  => array(
                                'type'   => 'plugin',
                                'action' => 'install',
                            ),
                        )
                    );

                    if (is_wp_error($result)) {
                        throw new \Exception($result->get_error_message());
                    }

                    $activate = true;

                } catch (\Exception $e) {
                }

                ob_end_clean();
            }

            wp_clean_plugins_cache();

            if ($activate) {
                try {
                    $result = activate_plugin($installed ? $installed_plugins[$plugin_file] : $plugin_slug . '/' . $plugin_file);
                    if (is_wp_error($result)) {
                        throw new \Exception($result->get_error_message());
                    }

                } catch (\Exception $e) {
                }
            }
        }
    }

    private function associatePluginFile($plugins, $key)
    {
        $path = explode('/', $key);
        $filename = end($path);
        $plugins[$filename] = $key;
        return $plugins;
    }
}