<?php

namespace WPSocialReviewsPro\App\Services;

use WPSocialReviews\App\Services\Platforms\PlatformManager;
use WPSocialReviews\Framework\Support\Arr;
use WPSocialReviews\App\Models\Review;
use WPSocialReviews\App\Models\Post;
use League\Csv\Writer;
use League\Csv\Reader;
use WPSocialReviews\App\Services\Helper as GlobalHelper;
use WPSocialReviews\App\Services\PermissionManager;
use WPSocialReviews\App\Services\DataProtector;
use WPSocialReviews\App\Services\Platforms\Reviews\Helper;
use WPSocialReviews\App\Services\Platforms\Chats\SocialChat;
use WPSocialReviewsPro\App\Services\Helper as ProHelper;


class ImportExportHandler
{
    protected $protector;
    private $platformManager;

    public function __construct()
    {
        $this->protector = new DataProtector();
        $this->platformManager = new PlatformManager();
    }

    private function getTableHeaders()
    {
        $commonHeaders = ['platform_name', 'review_title', 'reviewer_name', 'reviewer_url', 'reviewer_img', 'reviewer_text', 'review_time', 'rating', 'created_at', 'updated_at'];
        $testimonialSpecificHeaders = ['category', 'author_company', 'author_position', 'author_website_logo', 'author_website_url'];

        return [
            'custom' => $commonHeaders,
            'testimonial' => array_merge($commonHeaders, $testimonialSpecificHeaders)
        ];
    }

    public function includeAutoloadFile()
    {
        require_once WPSOCIALREVIEWS_PRO_DIR.'app/Services/Libs/CSV/autoload.php';
    }

    private function isValidCsvFile($fileType)
    {
        $mimes = ['text/csv', 'application/csv', 'application/json', 'application/octet-stream'];
        return in_array($fileType, $mimes);
    }

    private function sanitizeData(&$data)
    {
        foreach ($data as $datumKey => $datum) {
            if( is_array($datum) && !empty($datum) ) {
                foreach ($datum as $key => $value) {
                    if (is_scalar($value)) { // Check if the value is scalar (string, int, float, etc.)
                        $data[$datumKey][$key] = ProHelper::sanitizeForCSV($value);
                    }
                }
            }
        }
    }

    private function writeCsv($data, $header, $fileName)
    {
        $writer = Writer::createFromFileObject(new \SplTempFileObject());
        $writer->setDelimiter(",");
        $writer->setNewline("\r\n");
        $writer->insertOne($header);
        $writer->insertAll($data);
        $writer->output(sanitize_file_name($fileName));
        die();
    }

    private function sendJSONResponse($data, $platformName)
    {
        $fileName = 'wpsn-' . $platformName . '-template-export-' . date('Y-m-d-H-i-s') . '.json';
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $fileName . '"');
        echo json_encode($data);
        die();
    }

    private function generateFileName($prefix)
    {
        return $prefix . '-' . date('Y-m-d-H-i-s') . '.csv';
    }

    public function updatePlatformOptions($platform, $data)
    {
        update_option('wpsr_' . $platform . '_verification_configs', $data['verification_configs']);
    }

    private function fetchCustomData($headers)
    {
        return Review::select($headers)
            ->where('platform_name', 'custom')
            ->get()
            ->toArray();
    }

    private function fetchTestimonialData($headers)
    {
        $queryColumns = array_merge($headers, ['category', 'fields']);
        $testimonials = Review::select($queryColumns)
            ->where('platform_name', 'testimonial')
            ->get()
            ->toArray();

        return array_map([$this, 'mapTestimonialData'], $testimonials);
    }

    private function mapTestimonialData($testimonial)
    {
        $fields = Arr::get($testimonial, 'fields');
        return [
            'platform_name' => Arr::get($testimonial, 'platform_name', ''),
            'review_title' => Arr::get($testimonial, 'review_title', ''),
            'reviewer_name' => Arr::get($testimonial, 'reviewer_name', ''),
            'reviewer_url' => Arr::get($testimonial, 'reviewer_url', ''),
            'reviewer_img' => Arr::get($testimonial, 'reviewer_img', ''),
            'reviewer_text' => Arr::get($testimonial, 'reviewer_text', ''),
            'review_time' => Arr::get($testimonial, 'review_time'),
            'rating' => Arr::get($testimonial, 'rating'),
            'created_at' => Arr::get($testimonial, 'created_at', null),
            'updated_at' => Arr::get($testimonial, 'updated_at', null),
            'category' => Arr::get($testimonial, 'category', ''),
            'author_company' => Arr::get($fields, 'author_company', ''),
            'author_position' => Arr::get($fields, 'author_position', ''),
            'author_website_logo' => Arr::get($fields, 'author_website_logo', ''),
            'author_website_url' => Arr::get($fields, 'author_website_url', ''),
        ];
    }

    private function addTestimonialColumns($itemTemp)
    {
        $extraColumns = [
            'author_company' => $itemTemp['author_company'] ?? '',
            'author_position' => $itemTemp['author_position'] ?? '',
            'author_website_logo' => $itemTemp['author_website_logo'] ?? '',
            'author_website_url' => $itemTemp['author_website_url'] ?? '',
        ];

        $itemTemp['fields'] = json_encode($extraColumns);
        unset($itemTemp['author_company'], $itemTemp['author_position'], $itemTemp['author_website_logo'], $itemTemp['author_website_url']);

        return $itemTemp;
    }

    private function mapItemToHeader($item, $csvHeader, $headerCount)
    {
        if ($headerCount == count($item)) {
            return array_combine($csvHeader, $item);
        }

        return array_combine(
            $csvHeader,
            array_merge(
                array_intersect_key($item, array_fill_keys(array_values($csvHeader), null)),
                array_fill_keys(array_diff(array_values($csvHeader), array_keys($item)), null)
            )
        );
    }

    public function exportData()
    {
        if(!PermissionManager::currentUserCan('wpsn_full_access')){
            return false;
        }

        $this->includeAutoloadFile();

        $type = isset($_GET['type']) ? sanitize_text_field($_GET['type']) : '';
        $templateID = isset($_GET['templateID']) ? sanitize_text_field($_GET['templateID']) : null;
        $platformName = isset($_GET['platformName']) ? sanitize_text_field($_GET['platformName']) : '';
        $tableHeaders = $this->getTableHeaders();
        
        $type = ($type == 'notifications') ? 'template' : $type;

        switch ($type) {
            case 'custom':
                $data = $this->fetchCustomData($tableHeaders['custom']);
                $fileName = $this->generateFileName('wpsn-export-reviews');
                break;
            case 'testimonial':
                $data = $this->fetchTestimonialData($tableHeaders['custom']);
                $fileName = $this->generateFileName('wpsn-export-testimonials');
                break;
            case 'template':
                $this->handleTemplateExport($templateID, $platformName);
                return;
            case 'chat-widget':
                $this->handleChatWidgetTemplateExport($templateID);
                return;
            default:
                return false;
        }
        if (empty($data)) {
            wp_redirect(admin_url('admin.php?data=empty&page=wpsocialninja.php') .'#/tools/export');
            exit;
        }

        $this->sanitizeData($data);
        $this->writeCsv($data, $tableHeaders[$type], $fileName);
    }

    private function handleChatWidgetTemplateExport($templateID) {
        if (!$templateID) {
            return false;
        }

        $metaData = (new SocialChat())->processMetadata($templateID);
        $postData = $this->formatPostData($templateID);

        $data = [
            'post_meta' => $metaData,
            'post_data' => $postData
        ];

        $this->sanitizeData($data);

        $this->sendJSONResponse($data, 'chat-widget');
    }

    private function handleTemplateExport($templateID, $platformNames)
    {
        if (!$templateID) {
            return false;
        }
        $post = new Post();
        $metaData = $post->getConfig($templateID);
        $styleData = $post->getConfig($templateID, '_wpsr_template_styles_config');

        $platforms = is_array($platformNames) ? $platformNames : explode(',', $platformNames);
        $exportData = [];

        foreach ($platforms as $platformName) {
            $platformName = trim($platformName);

            $connectedConfigs = $this->platformManager->getConnectedSourcesConfigs($platformName);
            $selectedAccounts = $this->platformManager->getSelectedFeedAccounts($platformName, $metaData);
            $filteredVerificationConfigs = $this->platformManager->getFeedVerificationConfigsBySourceId($platformName, $connectedConfigs, $selectedAccounts);

            $data = [
                'platform_name' => $platformName,
                'connected_sources_config' => $filteredVerificationConfigs,
                'verification_configs' => $this->getVerificationConfigsIfNeeded($platformName),
                'platform_global_settings' => get_option('wpsr_' . $platformName . '_global_settings', []),
                'post_meta' => $metaData,
                'post_data' => $this->formatPostData($templateID),
                'post_style_meta' => $styleData,
            ];

            if (!in_array($platformName, $this->platformManager->feedPlatforms())) {
                $data['business_info'] = get_option('wpsr_reviews_' . $platformName . '_business_info');
            }

            if ($platformName === 'google') {
                $data['location_lists'] = get_option('wpsr_reviews_google_locations_list');
            }

            $this->sanitizeData($data);
            $exportData[] = $data;
        }

        $this->sendJSONResponse($exportData, $platformNames);
    }

    public function importData()
    {
        if(!PermissionManager::currentUserCan('wpsn_full_access')){
            return false;
        }

        $this->includeAutoloadFile();
        $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : 'custom';
        $file = $_FILES['file'] ?? null;

        if (!$this->isValidCsvFile($file['type'])) {
            wp_send_json_error(
                ['message' => __('Please upload a valid CSV or JSON file.', 'wp-social-ninja-pro')],
                423
            );
        }

        $data = $this->handleFileUpload($file, $type);
        if ($type === 'template') {
            $this->processTemplateImport($data);
        } elseif ($type === 'chat-widget') {
            $this->processChatWidgetTemplateImport($data);
        } elseif ($type === 'notifications') {
            $this->processNotificatioImport($data);
        } else {
            $this->processCSVImport($data, $type);
        }
    }

    private function handleFileUpload($file, $type)
    {
        $tmpName = sanitize_text_field($file['tmp_name']);
        $fileContents = file_get_contents($tmpName);

        if ($type === 'template' || $type === 'chat-widget' || $type === 'notifications') {
            $jsonData = json_decode($fileContents, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                wp_send_json_error(['message' => __('Invalid JSON format.', 'wp-social-ninja-pro')], 423);
            }
            return $jsonData;
        }

        return Reader::createFromString($fileContents)->fetchAll();
    }

    private function processChatWidgetTemplateImport($data)
    {
        if (empty($data)) {
            wp_send_json_error(['message' => __('File is empty or invalid.', 'wp-social-ninja-pro')], 423);
        }

        $postDetails = maybe_unserialize($data['post_data']);

        $post = new Post();
        $postId = $post->createPost([
            'post_title' => $postDetails['post_title'],
            'post_content' => $postDetails['post_content'],
            'post_type' => $postDetails['post_type'],
            'post_status' => $postDetails['post_status'],
        ]);

        if (!$postId || is_wp_error($postId)) {
            wp_send_json_error(['message' => __('Failed to create post.', 'wp-social-ninja-pro')], 423);
        }

        $metaData = maybe_unserialize($data['post_meta']);
        if (!add_post_meta($postId, '_wpsr_template_config', $metaData)) {
            wp_send_json_error(['message' => __('Failed to add post meta.', 'wp-social-ninja-pro')], 423);
        }

        wp_send_json_success([
            'message' => __('Successfully uploaded data.', 'wp-social-ninja-pro'),
        ], 200);
    }

    private function processNotificatioImport($data)
    {
        if (empty($data)) {
            wp_send_json_error(['message' => __('File is empty or invalid.', 'wp-social-ninja-pro')], 423);
        }

        $platformDataList = isset($data[0]) ? $data : [$data];

        $mergedPostMeta = [];
        $mergedPlatforms = [];
        $postData = null;

        foreach ($platformDataList as $platformData) {
            $platform = Arr::get($platformData, 'platform_name');
            $postData = Arr::get($platformData, 'post_data');
            $platform = sanitize_text_field($platform ?? '');
            if (!$platform) {
                wp_send_json_error(['message' => __('Platform name is missing.', 'wp-social-ninja-pro')], 423);
            }

            if ($platform == 'fluent_forms') {
                $this->verifyFluentFormExists();
            } elseif ($platform != 'fluent_forms' && $platform != 'custom' && $platform != 'testimonial') {
                $this->validatePlatformData($platform, $platformData);
            }

            $postMeta = Arr::get($platformData, 'post_meta');
            if (isset($postMeta)) {
                $mergedPostMeta = array_merge($mergedPostMeta, $postMeta);
            }

            $mergedPlatforms[] = $platform;

            if (!empty($platformData['platform_global_settings'])) {
                $platformGlobalSettings = Arr::get($platformData, 'platform_global_settings');
                update_option('wpsr_' . $platform . '_global_settings', $platformGlobalSettings);
            }

            if (in_array($platform, $this->platformManager->feedPlatforms())) {
                $verificationConfigs = Arr::get($platformData, 'verification_configs');
                if (!empty($verificationConfigs)) {
                    $this->updatePlatformVerificationConfigs($platform, $verificationConfigs);
                }
            } else {
                $this->updateReviewsPlatformVerificationConfigs($platform, $platformData);
            }

        }

        $this->createPostForImport($mergedPlatforms, 'notifications', $platformData['post_meta'], $postData);

        wp_send_json_success([
            'message' => __('Successfully uploaded data.', 'wp-social-reviews')
        ], 200);
    }

    private function processTemplateImport($data)
    {
        if (empty($data)) {
            wp_send_json_error(['message' => __('File is empty or invalid.', 'wp-social-ninja-pro')], 423);
        }

        $mergedPlatforms = [];
        $postData = null;
        $platformDataList = isset($data[0]) ? $data : [$data];
        foreach ($platformDataList as $platformData) {
            $platform = sanitize_text_field($platformData['platform_name'] ?? '');

            $postData = Arr::get($platformData, 'post_data');
            $mergedPlatforms[] = $platform;
            if (!$platform) {
                wp_send_json_error(['message' => __('Platform name is missing.', 'wp-social-ninja-pro')], 423);
            }

            if ($platform == 'fluent_forms') {
                $this->verifyFluentFormExists();
            } elseif ($platform != 'fluent_forms' && $platform != 'custom' && $platform != 'testimonial') {
                $this->validatePlatformData($platform, $platformData);
            }

            update_option('wpsr_' . $platform . '_global_settings', $platformData['platform_global_settings']);
            if (in_array($platform, $this->platformManager->feedPlatforms())) {
                $this->updatePlatformVerificationConfigs($platform, $platformData);
            } else {
                $this->updateReviewsPlatformVerificationConfigs($platform, $platformData);
            }
        }

        $this->createPostForImport($mergedPlatforms, 'wp_social_reviews', $platformData['post_meta'], $postData);

        wp_send_json_success([
            'message' => __('Successfully uploaded data.', 'wp-social-reviews')
        ], 200);
    }

    private function validatePlatformData($platform, $platformData)
    {
        $errorMessages = [
            'youtube' => __('Verification config is missing, please connect the platform before exporting feeds.', 'wp-social-ninja-pro'),
            'tiktok' => __('Connected sources config is missing, please connect the platform before exporting feeds.', 'wp-social-ninja-pro'),
            'tiktok_not_installed' => __('TikTok is not installed. Please install and connect TikTok to proceed.', 'wp-social-ninja-pro'),
            'default' => __('Verification configs are missing, please connect the platform before exporting feeds.', 'wp-social-ninja-pro'),
        ];
    
        if ($platform === 'tiktok') {
            if (!GlobalHelper::isCustomFeedForTiktokInstalled() ) { //  `isTikTokInstalled()` checks if TikTok is installed
                wp_send_json_error(['message' => $errorMessages['tiktok_not_installed']], 423);
            }
            if (empty($platformData['connected_sources_config'])) {
                wp_send_json_error(['message' => $errorMessages['tiktok']], 423);
            }
        }
    
        // YouTube-specific validation
        if (empty($platformData['verification_configs']) && $platform === 'youtube') {
            wp_send_json_error(['message' => $errorMessages['youtube']], 423);
        }
    
        // Default validation for other platforms
        if ((empty($platformData['connected_sources_config']) || empty($platformData['verification_configs'])) 
            && !in_array($platform, ['tiktok', 'youtube'])) {
            wp_send_json_error(['message' => $errorMessages['default']], 423);
        }
    
    }

    private function getVerificationConfigsIfNeeded($platformName)
    {
        $platformsRequiringVerification = ['instagram', 'facebook_feed', 'youtube', 'twitter', 'google', 'airbnb', 'yelp', 'tripadvisor', 'amazon', 'aliexpress', 'booking.com', 'facebook', 'trustpilot'];
        if (in_array($platformName, $platformsRequiringVerification)) {
            if ($platformName === 'google') {
                return get_option('wpsr_reviews_google_verification_configs');
            }

            $platformsWithSettings = ['airbnb', 'yelp', 'tripadvisor', 'amazon', 'aliexpress', 'booking.com', 'facebook', 'trustpilot'];
    
            if (in_array($platformName, $platformsWithSettings, true)) {
                return get_option("wpsr_reviews_{$platformName}_settings");
            }
            return $this->platformManager->getFeedVerificationConfigs($platformName);
        }
        return [];
    }

    private function updateReviewsPlatformVerificationConfigs($platform, $platformData)
    {
        if ($platform === 'google') {
            update_option('wpsr_reviews_google_verification_configs', $platformData['verification_configs']);
            update_option('wpsr_reviews_' . $platform . '_locations_list', $platformData['location_lists']);
            $apiSettings = $platformData['connected_sources_config'] ?? [];
            update_option('wpsr_reviews_google_settings', $apiSettings, 'no');
        } else if (isset($platformData['verification_configs']) && $platform === 'facebook') {
            $this->processVerificationConfigs(
                $platformData['verification_configs'],
                $this->getVerificationType($platform),
                $platform,
                $platformData
            );
            update_option('wpsr_reviews_' . $platform . '_settings', $platformData['verification_configs']);
        } else {
            update_option('wpsr_reviews_' . $platform . '_settings', $platformData['verification_configs']);
        }
        update_option('wpsr_reviews_' . $platform . '_business_info', $platformData['business_info']);
    }
    private function updatePlatformVerificationConfigs($platform, $data)
    {
        if (isset($data['connected_sources_config']) || isset($data['verification_configs'])) {
            $configType = isset($data['connected_sources_config']) &&
                          in_array($platform, ['facebook_feed', 'instagram', 'tiktok']) 
                          ? 'connected_sources_config' 
                          : 'verification_configs';
        
            $this->processVerificationConfigs($data[$configType], $this->getVerificationType($platform), $platform);
        }

        switch ($platform) {
            case 'facebook_feed':
                $this->updatePlatformOptions($platform, $data);
                update_option('wpsr_facebook_feed_connected_sources_config', ['sources' => $data['connected_sources_config']]);
                break;
            case 'instagram':
                $verification_configs['verification_configs'] = [
                    'connected_accounts' => $data['connected_sources_config']
                ];
                $this->updatePlatformOptions($platform, $verification_configs);
                break;
            case 'youtube':
                $this->updatePlatformOptions($platform, $data);
                break;
            case 'twitter':
                $this->updatePlatformOptions($platform, $data);
                break;
            case 'tiktok':
                update_option('wpsr_tiktok_connected_sources_config', ['sources' => $data['connected_sources_config']]);
                break;
        }
    }

    private function getVerificationType($platform)
    {
        $verificationTypes = [
            'facebook_feed' => 'EA',
            'instagram' => 'IG',
            'tiktok' => 'act',
            'youtube' => 'YT',
            'facebook' => 'EA'
        ];

        return $verificationTypes[$platform] ?? '';
    }

    public function processVerificationConfigs( &$verificationConfigs, $tokenPrefix, $platform, &$platformData = null)
    {
        foreach ($verificationConfigs as &$verificationConfig) {
            if (isset($verificationConfig['access_token']) || isset($verificationConfig['api_key'])) {
                $access_token = $verificationConfig['access_token'] ?? $verificationConfig['api_key'];
                $parsedToken = $this->protector->decrypt($access_token) ?: $access_token;

                if ($parsedToken && !str_contains($parsedToken, $tokenPrefix)) {
                    if ($platform == 'facebook') {
                       $placeId = $verificationConfig['place_id'];
                        if (!isset($platformData['business_info'][$placeId])) {
                            $platformData['business_info'][$placeId] = [];
                        }
                       $platformData['business_info'][$placeId]['encryption_error'] = true;
                    } else {
                        $verificationConfig['encryption_error'] = true;
                    }
                }
            }
        }
    }

    private function processCSVImport($data, $type)
    {
        $tableHeaders = $this->getTableHeaders();

        // Sanitize CSV header
        $csvHeader = array_shift($data);
        $csvHeader = array_map('esc_attr', $csvHeader);
        array_splice($csvHeader, 1, 0, "platform_name");

        // Validate headers against allowed columns for the type
        foreach ($csvHeader as $column) {
            if (!in_array($column, $tableHeaders[$type])) {
                wp_send_json_error([
                    'message' => __('Unknown column: ' . $column . '. Invalid characters in column name!', 'wp-social-ninja-pro')
                ], 423);
            }
        }

        // Format the data according to the sanitized header
        $formattedData = array_map(function ($el) use ($type) {
            array_splice($el, 1, 0, $type); // Insert platform_name into the data row
            return $el;
        }, $data);

        $dataRecords = $this->prepareDataRecords($formattedData, $csvHeader, $type);

        foreach (array_chunk($dataRecords, 3000) as $chunk) {
            $this->batchInsert($chunk, $type);
        }
    }

    private function prepareDataRecords($reader, $csvHeader, $type)
    {
        $dataRecords = [];
        $headerCount = count($csvHeader);

        foreach ($reader as $item) {
            $itemTemp = $this->mapItemToHeader($item, $csvHeader, $headerCount);
            $itemTemp['review_time'] = date('Y-m-d H:i:s', strtotime($itemTemp['review_time']));
            $itemTemp['created_at'] = date('Y-m-d H:i:s');
            $itemTemp['updated_at'] = date('Y-m-d H:i:s');

            // Add the extra testimonial columns to the fields column
            if ($type == 'testimonial') {
                $itemTemp = $this->addTestimonialColumns($itemTemp);
            }

            $dataRecords[] = $itemTemp;
        }

        return $dataRecords;
    }

    public function batchInsert($rows, $type)
    {
        global $wpdb;

        // Extract column list from first row of data
        $columns = array_keys($rows[0]);

        $table  = $wpdb->prefix.'wpsr_reviews';

        asort($columns);
        $columnList = '`' . implode('`, `', $columns) . '`';
        // Start building SQL, initialise data and placeholder arrays
        $sql = "INSERT INTO `$table` ($columnList) VALUES\n";
        $placeholders = array();
        $data = array();

        // Build placeholders for each row, and add values to data array
        foreach ($rows as $row) {
            ksort($row);
            $rowPlaceholders = array();
            foreach ($row as $key => $value) {
                $data[] = json_decode(json_encode(sanitize_text_field($value), JSON_UNESCAPED_UNICODE), true);
                
                if($key === 'source_id') {
                    $rowPlaceholders[] = '%s';
                } else {
                    $rowPlaceholders[] = is_numeric($value) ? '%d' : '%s';
                }
            }
            $placeholders[] = '(' . implode(', ', $rowPlaceholders) . ')';
        }

        // Stitch all rows together
        $sql .= implode(",\n", $placeholders);
        // Run the query.  Returns number of affected rows.

        $res = $wpdb->query($wpdb->prepare($sql, $data));
        if(!$res) {
            wp_send_json_error([
                'message' => __('Data is not uploaded!!',  'wp-social-ninja-pro')
            ], 423);
        }

        $businessInfo = Review::getInternalBusinessInfo($type);
        update_option('wpsr_reviews_' . $type . '_business_info', $businessInfo);

        wp_send_json_success([
            'message' => __('Successfully uploaded data.',  'wp-social-ninja-pro')
        ], 200);
    }

    /**
     * Create a post for the imported data
     *
     * @param array $platforms
     * @param string $type (template, notification, chat-widget)
     * @param array $postMeta
     * @return int|false
     */
    private function createPostForImport($platforms, $type, $postMeta, $postInformation = null)
    {
        $post = new Post();
        $implodedPlatforms = implode(', ', $platforms);

        $postData = [
            'post_title' => $postInformation['post_title'],
            'post_content' => $postInformation['post_content'],
            'post_type' => $postInformation['post_type'],
        ];

        $postId = $post->createPost($postData);

        if ($postId && !empty($postMeta)) {
            $post->updatePostMeta($postId, $postMeta, $implodedPlatforms);

            switch ($type) {
                case 'template':
                case 'notifications':
                    update_post_meta($postId, '_wpsr_template_styles_config', json_encode($postMeta['post_style_meta'] ?? []));
                    break;
                case 'chat-widget':
                    update_post_meta($postId, '_wpsr_template_config', $postMeta);
                    break;
            }
        }

        return $postId;
    }

    private function formatPostData($postId)
    {
        $post = get_post($postId);
        return [
            'ID' => $post->ID,
            'post_title' => $post->post_title,
            'post_content' => $post->post_content,
            'post_type' => $post->post_type,
            'post_status' => $post->post_status,
        ];
    }

    private function verifyFluentFormExists()
    {
        $hasFluentForm = defined('FLUENTFORM_VERSION');
        if (!$hasFluentForm) {
            wp_send_json_error(['message' => __('Fluent Forms is not installed. Please install Fluent Forms to proceed.', 'wp-social-ninja-pro')], 423);
        }
    }
}