<?php
/**
 * Stripe Checkout: Session
 *
 * Pro-only functionality adjustments for Stripe Checkout Sessions.
 *
 * @package SimplePay\Pro\Payments\Stripe_Checkout\Plan
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8.0
 */

namespace SimplePay\Pro\Payments\Stripe_Checkout\Session;

use SimplePay\Core\API;
use SimplePay\Pro\Payment_Methods;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


/**
 * Maps Stripe Checkout Session data to the Customer.
 *
 * Collected Billing Address is only assigned to the Payment Method, and collected
 * Shipping Address is only available directly on the Checkout Session object.
 *
 * This data may not be immediately available in confirmations or emails due
 * to the order (or lack of order) webhooks fire. Populating the data still benefits
 * Stripe's Customer Billing Portal.
 *
 * @link https://github.com/wpsimplepay/wp-simple-pay-pro/issues/973
 *
 * @since 4.1.0
 *
 * @param \SimplePay\Vendor\Stripe\Event         $event Stripe webhook event.
 * @param null|\SimplePay\Vendor\Stripe\Customer $customer Stripe Customer.
 */
function map_object_data_to_customer( $event, $customer ) {
	if ( null === $customer ) {
		return;
	}

	$object = $event->data->object;

	$form_id = isset( $object->metadata->simpay_form_id )
		? $object->metadata->simpay_form_id
		: '';

	if ( empty( $form_id ) ) {
		return;
	}

	$form = simpay_get_form( $form_id );

	if ( false === $form ) {
		return;
	}

	$payment_methods = API\PaymentMethods\all(
		array(
			'type'     => 'card',
			'customer' => $customer->id,
		),
		$form->get_api_request_args()
	);

	$payment_method = current( $payment_methods->data );

	if ( false === $payment_method ) {
		return;
	}

	// Metadata.
	$metadata = $customer->metadata->toArray();

	// Ensure generated metadata key is not copied over.
	if ( isset( $metadata['simpay_is_generated_customer'] ) ) {
		$metadata['simpay_is_generated_customer'] = '';
	}

	// Address.
	$billing_address = isset(
		$payment_method->billing_details,
		$payment_method->billing_details->address
	)
		? $payment_method->billing_details->address->toArray()
		: array();

	// Name.
	$billing_name = isset(
		$payment_method->billing_details,
		$payment_method->billing_details->name
	)
		? $payment_method->billing_details->name
		: '';

	// Attempt to find the Shipping Address.
	if ( isset( $event->data->object->shipping ) ) {
		$shipping_name = isset( $event->data->object->shipping->name )
			? $event->data->object->shipping->name
			: '';

		$shipping_address = isset( $event->data->object->shipping->address )
			? $event->data->object->shipping->address->toArray()
			: '';

		$shipping_args = array();

		if ( ! empty( $shipping_name ) ) {
			$shipping_args['name'] = $shipping_name;
		}

		if ( ! empty( $shipping_address ) ) {
			$shipping_args['address'] = $shipping_address;
		}
	} else {
		$shipping_args = array();
	}

	$customer_update_args = array(
		'metadata' => $metadata,
	);

	if ( ! empty( $billing_address ) ) {
		$customer_update_args['address'] = $billing_address;
	}

	if ( ! empty( $billing_name ) ) {
		$customer_update_args['name'] = $billing_name;
	}

	if ( ! empty( $shipping_args ) ) {
		$customer_update_args['shipping'] = $shipping_args;
	}

	try {
		API\Customers\update(
			$customer->id,
			$customer_update_args,
			$form->get_api_request_args()
		);
	} catch ( \Exception $e ) {
		// Updating this is not required, and a failure should not cause other
		// actions to rerun.
	}
}
add_filter(
	'simpay_webhook_checkout_session_completed',
	__NAMESPACE__ . '\\map_object_data_to_customer',
	20,
	2
);
