<?php
/**
 * Simple Pay: Tax collection options
 *
 * @package SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8.0
 */

namespace SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form_Payment_Options;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SimplePay\Core\Settings;
use SimplePay\Core\Utils;

/**
 * Tax Collection Fields for Payment Forms.
 *
 * @since 4.11.0
 */
class Tax_Collection {

	/**
	 * Tax_Collection constructor.
	 */
	public function __construct() {
		add_action(
			'simpay_form_settings_meta_payment_options_panel',
			array( $this, 'unstable_add_tax' ),
			5
		);

		remove_action(
			'simpay_form_settings_meta_payment_options_panel',
			'SimplePay\Core\Post_Types\Simple_Pay\Edit_Form\__unstable_add_tax_upsell',
			9
		);
	}


	/**
	 * Outputs markup for the "Tax Rates" setting.
	 *
	 * @since 4.5.0
	 * @access private
	 *
	 * @param int $post_id Payment form ID.
	 * @return void
	 */
	public function unstable_add_tax( $post_id ) {
		$license = simpay_get_license();

		// Tax status defaults to "fixed-global" if not previously set.
		// Set to "automatic" for new payment forms.
		$tax_status = get_post_meta( $post_id, '_tax_status', true );

		if ( 'auto-draft' !== get_post_status() && empty( $tax_status ) ) {
			$tax_status = 'fixed-global';
		} elseif ( 'auto-draft' === get_post_status() ) {
			$tax_status = simpay_get_payment_form_setting(
				$post_id,
				'tax_status',
				'none',
				__unstable_simpay_get_payment_form_template_from_url()
			);
		}

		// Tax code.
		$tax_code = get_post_meta( $post_id, '_tax_code', true );

		// Tax behavior.
		$tax_behavior = get_post_meta( $post_id, '_tax_behavior', true );

		if ( empty( $tax_behavior ) ) {
			$tax_behavior = 'unspecified';
		}

		$automatic_tax_supported = in_array(
			strtolower( simpay_get_setting( 'account_country', 'us' ) ),
			array(
				'at',
				'au',
				'be',
				'bg',
				'ca',
				'cz',
				'de',
				'dk',
				'ee',
				'es',
				'fi',
				'fr',
				'gb',
				'gr',
				'hk',
				'hr',
				'ie',
				'is',
				'it',
				'jp',
				'lt',
				'lu',
				'lv',
				'mt',
				'nl',
				'no',
				'nz',
				'pl',
				'pt',
				'ro',
				'se',
				'sg',
				'si',
				'sk',
				'us',
				'za',
			),
			true
		);

		$upgrade_automatic_title = esc_html__(
			'Unlock Automatically Calculated Tax Amounts',
			'simple-pay'
		);

		$upgrade_automatic_description = esc_html__(
			'We\'re sorry, automatically calculating and collecting taxes is not available in your plan. Please upgrade to the <strong>Professional</strong> plan or higher to unlock this and other awesome features.',
			'simple-pay'
		);

		$upgrade_automatic_url = simpay_pro_upgrade_url(
			'form-tax-settings',
			'Automatic Rate Taxes'
		);

		$upgrade_automatic_purchased_url = simpay_docs_link(
			'Automatic Rate Taxes (already purchased)',
			'upgrading-wp-simple-pay-pro-license',
			'form-tax-settings',
			true
		);
		?>
	<table>
		<tbody class="simpay-panel-section">
			<tr class="simpay-panel-field">
				<th>
					<label for="_tax_status">
						<?php esc_html_e( 'Tax Collection', 'simple-pay' ); ?>
					</label>
				</th>
				<td>
					<div style="display: flex; align-items: center;">
						<select id="_tax_status" name="_tax_status">
							<option
								value="none"
								<?php selected( 'none', $tax_status ); ?>
							>
								<?php esc_html_e( 'None', 'simple-pay' ); ?>
							</option>
							<option
								value="fixed-global"
								<?php selected( 'fixed-global', $tax_status ); ?>
							>
								<?php esc_html_e( 'Fixed tax rates', 'simple-pay' ); ?>
							</option>
							<option
								value="automatic"
								<?php selected( 'automatic', $tax_status ); ?>
								<?php disabled( false, $automatic_tax_supported ); ?>
								data-available="<?php echo $license->is_pro( 'professional' ) ? 'yes' : 'no'; ?>"
								data-upgrade-title="<?php echo esc_attr( $upgrade_automatic_title ); ?>"
								data-upgrade-description="<?php echo esc_attr( $upgrade_automatic_description ); ?>"
								data-upgrade-url="<?php echo esc_url( $upgrade_automatic_url ); ?>"
								data-upgrade-purchased-url="<?php echo esc_url( $upgrade_automatic_purchased_url ); ?>"
								data-prev-value="<?php echo esc_attr( $tax_status ); ?>"
							>
								<?php
								esc_html_e(
									'Automatically calculated by location',
									'simple-pay'
								);
								?>
							</option>
						</select>
					</div>

					<p
						class="description simpay-show-if"
						data-if="_tax_status"
						data-is="none"
					>
							<?php
							esc_html_e(
								'No tax will be collected.',
								'simple-pay'
							);
							?>
					</p>

					<p
						class="description simpay-show-if"
						data-if="_tax_status"
						data-is="fixed-global"
					>
							<?php
							$tax_rates_url = Settings\get_url(
								array(
									'section'    => 'general',
									'subsection' => 'taxes',
								)
							);

							echo wp_kses(
								sprintf(
								/* translators: Opening anchor tag, do not translate. %2$s Closing anchor tag, do not translate. */
									__(
										'Apply all enabled %1$sglobal tax rates%2$s to payments.',
										'simple-pay'
									),
									'<a href="' . esc_url( $tax_rates_url ) . '" target="_blank">',
									'</a>'
								),
								array(
									'a' => array(
										'href'   => true,
										'target' => true,
									),
								)
							);
							?>
					</p>

						<?php if ( true === $automatic_tax_supported ) : ?>
					<p
						class="description simpay-show-if"
						data-if="_tax_status"
						data-is="automatic"
					>
							<?php
							echo wp_kses(
								sprintf(
								/* translators: Opening anchor tag, do not translate. %2$s Closing anchor tag, do not translate. */
									__(
										'Automatically determine tax based on your %1$sStripe tax registrations%2$s and users\'s location.',
										'simple-pay'
									),
									'<a href="https://dashboard.stripe.com/test/settings/tax" target="_blank" rel="noopener noreferrer" class="simpay-external-link">',
									Utils\get_external_link_markup() . '</a>'
								),
								array(
									'a'    => array(
										'href'   => true,
										'target' => true,
										'rel'    => true,
										'class'  => true,
									),
									'span' => array(
										'class' => true,
									),
								)
							);
							?>
					</p>
					<?php else : ?>
					<p class="description">
						<?php
						esc_html_e(
							'Automatic tax is not available for your Stripe account\'s country.',
							'simple-pay'
						);
						?>
					</p>
					<?php endif; ?>

					<div
						class="simpay-show-if"
						data-if="_tax_status"
						data-is="automatic"
						style="margin-top: 12px;"
					>
						<label for="_tax_code" style="display: block; margin-bottom: 4px;">
							<strong>
								<?php
								esc_html_e(
									'Tax Category',
									'simple-pay'
								);
								?>
							</strong>
						</label>
						<select id="_tax_code" name="_tax_code" style="max-width: 300px;">
								<?php
								$stripe_tax_codes = simpay_get_stripe_tax_codes();

								foreach ( $stripe_tax_codes as $stripe_tax_code ) :
									?>
							<option
								value="<?php echo esc_attr( $stripe_tax_code->id ); ?>"
									<?php selected( $stripe_tax_code->id, $tax_code ); ?>
							>
									<?php echo esc_html( $stripe_tax_code->name ); ?>
							</option>
								<?php endforeach; ?>
						</select>

						<p class="description">
								<?php
								echo wp_kses(
									sprintf(
									/* translators: Opening anchor tag, do not translate. %2$s Closing anchor tag, do not translate. */
										__(
											'Selecting the appropriate %1$stax category%2$s ensures the lowest applicable tax rate is applied.',
											'simple-pay'
										),
										'<a href="https://stripe.com/docs/tax/tax-categories" target="_blank" rel="noopener noreferrer" class="simpay-external-link">',
										Utils\get_external_link_markup() . '</a>'
									),
									array(
										'a'    => array(
											'href'   => true,
											'target' => true,
											'rel'    => true,
											'class'  => true,
										),
										'span' => array(
											'class' => true,
										),
									)
								);
								?>
						</p>
					</div>

					<div
						class="simpay-show-if"
						data-if="_tax_status"
						data-is="automatic"
						style="margin-top: 12px;"
					>
						<label
							for="_tax_behavior"
							style="display: block; margin-bottom: 4px;"
						>
							<strong>
								<?php
								esc_html_e(
									'Tax Behavior',
									'simple-pay'
								);
								?>
							</strong>
						</label>

						<div style="display: flex; align-items: center;">
							<select
								id="_tax_behavior"
								name="_tax_behavior"
									<?php if ( 'unspecified' !== $tax_behavior ) : ?>
									disabled
								<?php endif; ?>
							>
								<option
									value="exclusive"
									<?php selected( 'exclusive', $tax_behavior ); ?>
								>
									<?php
									esc_html_e(
										'Exclusive',
										'simple-pay'
									);
									?>
								</option>
								<option
									value="inclusive"
										<?php selected( 'inclusive', $tax_behavior ); ?>
								>
									<?php
									esc_html_e(
										'Inclusive',
										'simple-pay'
									);
									?>
								</option>
							</select>
						</div>

						<p class="description">
								<?php
								if ( 'unspecified' !== $tax_behavior ) :
									esc_html_e(
										'Tax behavior for automatic taxes cannot be adjusted after being set to ensure accuracy in accounting. Create a new payment form if you need to change the tax calculation behavior.',
										'simple-pay'
									);
								else :
									esc_html_e(
										'Setting tax behavior to "Exclusive" adds tax onto the subtotal amount specified on the price options. When set to inclusive, the amount your buyer pays never changes, even if the tax rate varies.',
										'simple-pay'
									);
								endif;
								?>
						</p>
					</div>

					<p style="margin: 0.5em 0;"></p>
				</td>
			</tr>
		</tbody>
	</table>

		<?php
	}
}
