<?php
/**
 * Settings: Emails > Payment Processing Confirmation
 *
 * @package SimplePay
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 4.10.0
 */

namespace SimplePay\Pro\Settings\Emails\PaymentProcessingConfirmation;

use SimplePay\Core\Emails;
use Simplepay\Core\Settings;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers "Email > Payment Processing Confirmation" subsection.
 *
 * @since 4.10.0
 *
 * @param \SimplePay\Core\Settings\Subsections_Collection $subsections Subsections collection.
 * @return void
 */
function register_subsection( $subsections ) {
	$email = new Emails\Email\PaymentProcessingConfirmationEmail();

	$subsections->add(
		new Settings\Subsection(
			array(
				'id'       => $email->get_id(),
				'section'  => 'emails',
				'label'    => esc_html( $email->get_label() ),
				'priority' => 30,
			)
		)
	);
}
add_action(
	'simpay_register_settings_subsections',
	__NAMESPACE__ . '\\register_subsection'
);

/**
 * Registers "Emails > Payment Processing Confirmation" settings.
 *
 * @since 4.10.0
 *
 * @param \SimplePay\Core\Settings\Settings_Collection $settings Settings collection.
 * @return void
 */
function register_settings( $settings ) {
	$email = new Emails\Email\PaymentProcessingConfirmationEmail();
	$id    = $email->get_id();

	// Enable/Disable.
	$settings->add(
		new Settings\Setting_Checkbox(
			array(
				'id'          => sprintf( 'email_%s', $id ),
				'section'     => 'emails',
				'subsection'  => $id,
				'label'       => $email->get_label(),
				'input_label' => $email->get_description(),
				'value'       => $email->is_enabled() ? 'yes' : 'no',
				'priority'    => 10,
				'schema'      => array(
					'type'    => 'string',
					'enum'    => array( 'yes', 'no' ),
					'default' => 'yes',
				),
				'toggles'     => array(
					'value'    => 'yes',
					'settings' => array(
						sprintf( 'email_%s_to', $id ),
						sprintf( 'email_%s_subject', $id ),
						sprintf( 'email_%s_body', $id ),

					),
				),
				'description' => '',
			)
		)
	);

	// To.
	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_to', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Send To',
					'setting label',
					'simple-pay'
				),
				'value'      => esc_html__( 'Customer email address', 'simple-pay' ),
				'classes'    => array(
					'regular-text',
					'readonly',
				),
				'priority'   => 20,
				'schema'     => array(
					'type' => 'string',
				),
				'readonly'   => true,
			)
		)
	);

	// Subject.
	$subject = simpay_get_setting(
		sprintf( 'email_%s_subject', $id ),
		esc_html(
			sprintf(
				/* translators: %s Site name */
				__( 'Payment Processing for %s', 'simple-pay' ),
				get_bloginfo( 'name' )
			)
		)
	);

	$settings->add(
		new Settings\Setting_Input(
			array(
				'id'         => sprintf( 'email_%s_subject', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Subject',
					'setting label',
					'simple-pay'
				),
				'value'      => $subject,
				'classes'    => array(
					'regular-text',
				),
				'priority'   => 30,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);

	// Bodies.

	$settings->add(
		new Settings\Setting(
			array(
				'id'         => sprintf( 'email_%s_body', $id ),
				'section'    => 'emails',
				'subsection' => $id,
				'label'      => esc_html_x(
					'Message',
					'setting label',
					'simple-pay'
				),
				'output'     => function () use ( $id ) {
					wp_editor(
						get_body_setting_or_default(),
						sprintf( 'email_%s_body', $id ),
						array(
							'textarea_name' => sprintf(
								'simpay_settings[email_%s_body]',
								$id
							),
							'textarea_rows' => 10,
						)
					);
				},
				'priority'   => 40,
				'schema'     => array(
					'type' => 'string',
				),
			)
		)
	);
}
add_action(
	'simpay_register_settings',
	__NAMESPACE__ . '\\register_settings'
);

/**
 * Returns the email body setting for the given type, or the default.
 *
 * @since 4.10.0
 *
 * @return string Email message body.
 */
function get_body_setting_or_default() {
	$email = new Emails\Email\PaymentProcessingConfirmationEmail();

	/** @var string $body */
	$body = simpay_get_setting(
		sprintf( 'email_%s_body', $email->get_id() ),
		sprintf(
				/* translators: %1$s: Customer email, %2$s: Form title */
			__(
				'Dear %1$s,<br /><br />We are currently processing your payment for %2$s and will notify you as soon as it\'s completed. Your payment details will be sent in a separate email once the payment has cleared.',
				'simple-pay'
			),
			'{customer:email}',
			'{form-title}'
		)
	);

	return $body;
}
