<?php
/**
 * Simple Pay: Price fields options
 *
 * @package SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 4.11.0
 */

namespace SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form_Payment_Options;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SimplePay\Core\PaymentForm\PriceOption;

/**
 * Price Fields class.
 *
 * @since 4.11.0
 */
class Price_Fields {
	use Price_Option\Utils;
	use Price_Option\Label;
	use Price_Option\Amount;
	use Price_Option\Recurring;
	use Price_Option\Custom_Amount;
	use Price_Option\Quantity;

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action(
			'simpay_form_settings_meta_payment_options_panel',
			array( $this, 'unstable_price_options' ),
			5
		);
		// Remove Lite output.
		remove_action(
			'simpay_form_settings_meta_payment_options_panel',
			'SimplePay\Core\Post_Types\Simple_Pay\Edit_Form\__unstable_add_price_options'
		);
	}

	/**
	 * Outputs markup for the price option list.
	 *
	 * @since 4.11.0
	 *
	 * @param int $post_id Current Post ID (Payment Form ID).
	 */
	public function unstable_price_options( $post_id ) {
		$form = simpay_get_form( $post_id );

		if ( false === $form ) {
			return;
		}

		$add_price_nonce = wp_create_nonce( 'simpay_add_price_nonce' );
		$add_plan_nonce  = wp_create_nonce( 'simpay_add_plan_nonce' );

		$prices = simpay_get_payment_form_prices( $form );

		// Prefill the price options from a template or default fallback.
		//
		// Special handling vs using simpay_get_payment_form_setting() because we need
		// full access to the form to create a PriceOption instance.
		if ( empty( $prices ) ) {
			$template = __unstable_simpay_get_payment_form_template_from_url();

			// Generate from a template.
			if ( ! empty( $template ) ) {
				foreach ( $template['data']['prices'] as $price ) {
					$price = new PriceOption(
						$price,
						$form,
						wp_generate_uuid4()
					);

					$price->__unstable_unsaved = true;

					$prices[ wp_generate_uuid4() ] = $price;
				}

				// Single price option fallback.
			} else {
				$currency = strtolower(
					simpay_get_setting( 'currency', 'USD' )
				);

				$default_amount = simpay_is_zero_decimal( $currency )
				? 100
				: 1000;

				$prices = array(
					wp_generate_uuid4() => new PriceOption(
						array(
							'unit_amount' => $default_amount,
							'currency'    => $currency,
						),
						$form,
						wp_generate_uuid4()
					),
				);
			}
		}
		?>

	<table>
		<tbody class="simpay-panel-section">
			<tr class="simpay-panel-field">
				<th>
					<?php esc_html_e( 'Price Options', 'simple-pay' ); ?>
				</th>
				<td style="border-bottom: 0;">
					<div
						style="
							display: flex;
							align-items: center;
							justify-content: space-between;
							margin: 0 0 12px;
						"
					>
						<button
							id="simpay-add-price"
							class="button button-secondary"
							data-nonce="<?php echo esc_attr( $add_price_nonce ); ?>"
							data-form-id="<?php echo esc_attr( $form->id ); ?>"
						>
							<?php esc_html_e( 'Add Price', 'simple-pay' ); ?>
						</button>

						<button
							id="simpay-prices-advanced-toggle"
							class="button button-link"
							style="
								display: flex;
								text-decoration: none;
								align-items: center;
								color: #666;
							"
						>
							<?php esc_html_e( 'Advanced', 'simple-pay' ); ?>
							<span
								class="dashicons dashicons-arrow-down-alt2"
								style="
									width: 14px;
									height: 14px;
									font-size: 14px;
									margin-left: 4px;
								"
							></span>
						</button>
					</div>

					<div
						id="simpay-prices-advanced"
						style="display: none; margin-bottom: 12px;"
					>
						<input
							type="text"
							value=""
							style="margin-right: 5px; width: 150px;"
							placeholder="plan_123"
							id="simpay-prices-advanced-plan-id"
						/>
						<button
							id="simpay-prices-advanced-add"
							class="button button-secondary"
							data-nonce="<?php echo esc_attr( $add_plan_nonce ); ?>"
							data-form-id="<?php echo esc_attr( $post_id ); ?>"
						>
							<?php esc_html_e( 'Add existing Plan', 'simple-pay' ); ?>
						</button>
					</div>

					<div
						id="simpay-prices-wrap"
						class="panel simpay-metaboxes-wrapper"
					>
						<div
							id="simpay-prices"
							class="simpay-prices simpay-metaboxes ui-sortable"
						>
							<?php
							/** @var \SimplePay\Core\PaymentForm\PriceOption[] $price Price option.  */
							foreach ( $prices as $instance_id => $price ) :
								$this->unstable_price_option( $price, $instance_id, $prices );
								endforeach;
							?>
						</div>
					</div>

						<?php
						/**
						 * Allows extra output after the price option list.
						 *
						 * @since 4.4.0
						 */
						do_action( 'unstable_simpay_form_settings_pro_after_price_options' )
						?>
				</td>
			</tr>
		</tbody>
	</table>

		<?php
	}



	/**
	 * Displays a single price option.
	 *
	 * @since 4.11.0
	 *
	 * @param \SimplePay\Core\PaymentForm\PriceOption        $price Price option.
	 * @param string                                         $instance_id Price option instance ID.
	 *                                                                    Shared by both payment
	 *                                                                    modes.
	 * @param array<\SimplePay\Core\PaymentForm\PriceOption> $prices All price options.
	 */
	public function unstable_price_option( $price, $instance_id, $prices ) {
		$amount_type = null !== $price->recurring && false === $price->can_recur
		? 'recurring'
		: 'one-time';
		$label       = $price->get_display_label();

		$recurring_settings_display = 'recurring' === $amount_type ? 'table' : 'none';

		$one_time_settings_display = (
		'recurring' === $amount_type &&
		! isset( $price->recurring['id'] ) &&
		false === $price->can_recur
		)
			? 'none'
			: 'table';

		$has_one_price = 1 === count( $prices );
		?>

	<div
		id="price-<?php echo esc_attr( $instance_id ); ?>"
		class="postbox <?php echo esc_attr( $has_one_price ? '' : 'closed' ); ?> simpay-field-metabox simpay-metabox simpay-price"
		<?php if ( false === $has_one_price ) : ?>
		aria-expanded="false"
		<?php endif; ?>
	>
		<input
			type="hidden"
			name="<?php echo esc_attr( $this->unstable_get_input_name( 'id', $instance_id ) ); ?>"
			value="<?php echo esc_attr( $price->id ); ?>"
		/>

		<input
			type="hidden"
			name="<?php echo esc_attr( $this->unstable_get_input_name( 'amount_type', $instance_id ) ); ?>"
			value="<?php echo esc_attr( $amount_type ); ?>"
			class="simpay-price-amount-type"
		/>

		<button type="button" class="simpay-handlediv simpay-price-label-expand">
			<span class="screen-reader-text">
				<?php
				echo esc_html(
					sprintf(
						/* translators: Price option label. */
						__( 'Toggle price option: %s', 'simple-pay' ),
						$label
					)
				);
				?>
			</span>
			<span class="toggle-indicator" aria-hidden="true"></span>
		</button>

		<h2 class="simpay-hndle ui-sortable-handle">
			<span class="custom-field-dashicon dashicons dashicons-menu-alt2" style="cursor: move;"></span>

			<strong class="simpay-price-label-display">
					<?php echo esc_html( $label ); ?>
			</strong>

			<strong class="simpay-price-label-default" style="display: none;">
				<?php esc_html_e( 'Default Price', 'simple-pay' ); ?>
			</strong>
		</h2>

		<div class="simpay-field-data simpay-metabox-content inside">
			<table>
				<?php
				$this->unstable_price_option_label( $price, $instance_id );
				$this->unstable_price_option_amount( $price, $instance_id );
				?>
			</table>

			<table
				class="simpay-price-recurring-amount-toggle">
				<?php
				$this->unstable_price_option_recurring_amount_toggle( $price, $instance_id );
				?>
			</table>
			<div
				id="recurring-amount-<?php echo esc_attr( $instance_id ); ?>"
				style="display: none;"
			>
				<?php
				$this->__unstable_price_option_recurring_options( $price, $instance_id );
				?>
			</div>

			<table>
				<?php
				$this->unstable_price_option_custom_amount_toggle( $price, $instance_id );
				?>
			</table>
			<div
				id="custom-amount-<?php echo esc_attr( $instance_id ); ?>"
				style="display: none;"
			>
				<?php
				$this->unstable_price_option_custom_amount( $price, $instance_id );
				?>
			</div>

			<table class="simpay-quantity-toggle">
				<?php
				$this->unstable_quantity_toggle( $price, $instance_id );
				?>
			</table>
			<div
				id="item-quantity-<?php echo esc_attr( $instance_id ); ?>"
				style="display: none;"
			>
				<?php $this->unstable_quantity_options( $price, $instance_id ); ?>
			</div>

			<div
				class="simpay-metabox-content-actions"
				style="display: flex; align-items: center;"
			>
				<button class="button-link button-link-delete simpay-price-remove" style="padding: 8px 0;">
					<?php esc_html_e( 'Remove Price', 'simple-pay' ); ?>
				</button>
				<div style="display: flex; align-items: center; gap: 10px;">
				<!-- Required Option -->
				<label
					class="simpay-price-required-check"
					for="<?php echo esc_attr( $this->unstable_get_input_id( 'required', $instance_id ) ); ?>"
					style="display: flex; align-items: center; padding: 8px 0; margin-left: auto"
				>
					<input
						type="checkbox"
						name="<?php echo esc_attr( $this->unstable_get_input_name( 'required', $instance_id ) ); ?>"
						id="<?php echo esc_attr( $this->unstable_get_input_id( 'required', $instance_id ) ); ?>"
						class="simpay-price-required"
						style="margin: 0 4px 0 0;"
						<?php if ( true === $price->required ) : ?>
						checked
						<?php endif; ?>
					/>
					<?php esc_html_e( 'Required', 'simple-pay' ); ?>
				</label>

				<!-- Default Price -->
				<label
					class="simpay-price-default-check"
					for="<?php echo esc_attr( $this->unstable_get_input_id( 'default', $instance_id ) ); ?>"
					style="display: flex; align-items: center; padding: 8px 0; margin-left: auto"
				>
					<input
						type="checkbox"
						name="<?php echo esc_attr( $this->unstable_get_input_name( 'default', $instance_id ) ); ?>"
						id="<?php echo esc_attr( $this->unstable_get_input_id( 'default', $instance_id ) ); ?>"
						class="simpay-price-default"
						style="margin: 0 4px 0 0;"
						value=""
						<?php if ( true === $price->default ) : ?>
						checked
						<?php endif; ?>
					/>
					<?php esc_html_e( 'Default Price', 'simple-pay' ); ?>
				</label>
				</div>
			</div>

				<?php
				if (
				$price->is_defined_amount() &&
				$price->__unstable_stripe_object->livemode !==
					$price->form->is_livemode()
				) :
					?>
			<p style="margin: 0; padding: 9px 18px; font-size: 12px; color: #d63638;">
					<?php
					esc_html_e(
						'Price not available in the current payment mode. Please remove and add again.',
						'simple-pay'
					);
					?>
			</p>
				<?php endif; ?>

		</div>
	</div>

		<?php
	}



	/**
	 * Outputs markup for an "Amount" control with a fixed currency symbol.
	 *
	 * @since 4.11.0
	 * @access private
	 *
	 * @param \SimplePay\Core\PaymentForm\PriceOption $price Price option.
	 * @param string                                  $instance_id Unique instance ID.
	 * @param int                                     $unit_amount Unit amount to display.
	 * @param string                                  $amount_input_name Amount input name.
	 */
	private function unstable_price_option_amount_control_fixed_currency(
		$price,
		$instance_id,
		$unit_amount,
		$amount_input_name,
		$amount_custom_placeholder = null
	) {
		$currency_position = simpay_get_currency_position();
		$is_zero_decimal   = simpay_is_zero_decimal( $price->currency );

		$currency_position_left = in_array(
			$currency_position,
			array( 'left', 'left_space' ),
			true
		);

		$currency_position_right = ! $currency_position_left;

		$amount_placeholder = null !== $amount_custom_placeholder ? $amount_custom_placeholder : simpay_format_currency(
			simpay_get_currency_minimum( $price->currency ),
			$price->currency,
			false
		);

		$amount_id   = $this->unstable_get_input_id( $amount_input_name, $instance_id );
		$amount_name = $this->unstable_get_input_name( $amount_input_name, $instance_id );

		if ( 0 === $unit_amount ) {
			$amount = '';
		} else {
			$amount = simpay_format_currency( $unit_amount, $price->currency, false );
		}

		?>

	<div class="simpay-currency-field">
		<?php if ( $currency_position_left ) : ?>
			<div
				class="simpay-price-currency-symbol simpay-currency-symbol simpay-currency-symbol-left"
				style="border-top-right-radius: 0; border-bottom-right-radius: 0;"
			>
				<?php
				echo esc_html(
					simpay_get_currency_symbol( $price->currency )
				);
				?>
			</div>
		<?php endif; ?>

		<input
			type="text"
			name="<?php echo esc_attr( $amount_name ); ?>"
			id="<?php echo esc_attr( $amount_id ); ?>"
			class="simpay-field simpay-field-tiny simpay-field-amount simpay-price-amount"
			value="<?php echo esc_attr( $amount ); ?>"
			placeholder="<?php echo esc_attr( $amount_placeholder ); ?>"
		/>

			<?php if ( $currency_position_right ) : ?>
			<div
				class="simpay-price-currency-symbol simpay-currency-symbol simpay-currency-symbol-right"
				style="border-top-left-radius: 0; border-bottom-left-radius: 0;"
			>
				<?php
				echo esc_html(
					simpay_get_currency_symbol( $price->currency )
				);
				?>
			</div>
		<?php endif ?>
	</div>

		<?php
	}

	/**
	 * Outputs markup for an "Amount Type" control.
	 *
	 * @since 4.11.0
	 * @access private
	 *
	 * @param \SimplePay\Core\PaymentForm\PriceOption $price Price option.
	 * @param string                                  $instance_id Unique instance ID.
	 */
	private function unstable_price_option_amount_type_control( $price, $instance_id ) {
		$one_time_active_class = (
		null === $price->recurring ||
		isset( $price->recurring['id'] ) ||
		true === $price->can_recur
		)
			? 'button-primary'
			: '';

		$recurring_active_class = (
			null !== $price->recurring &&
			! isset( $price->recurring['id'] ) &&
			false === $price->can_recur
		)
			? 'button-primary'
			: '';

		$license = simpay_get_license();

		$upgrade_title = __(
			'Unlock Subscription Functionality',
			'simple-pay'
		);

		$upgrade_description = __(
			'We\'re sorry, recurring payments through subscriptions are not available on your plan. Please upgrade to the <strong>Plus</strong> plan or higher to unlock this and other awesome features.',
			'simple-pay'
		);

		$upgrade_url = simpay_pro_upgrade_url(
			'form-price-option-settings',
			'Subscription amount type'
		);

		$upgrade_purchased_url = simpay_docs_link(
			'Subscription amount type (already purchased)',
			$license->is_lite()
			? 'upgrading-wp-simple-pay-lite-to-pro'
			: 'activate-wp-simple-pay-pro-license',
			'form-price-option-settings',
			true
		);
		?>

	<fieldset>
		<legend class="screen-reader-text">
			<?php esc_html( 'Price Type', 'simple-pay' ); ?>
		</legend>

		<div class="button-group simpay-price-amount-type">
			<button
				class="button <?php echo esc_attr( $one_time_active_class ); ?>"
				aria-title="<?php esc_attr_e( 'One time', 'simple-pay' ); ?>"
				data-amount-type="one-time"
			>
				<?php esc_html_e( 'One time', 'simple-pay' ); ?>
			</button>
			<button
				class="button <?php echo esc_attr( $recurring_active_class ); ?>"
				aria-title="<?php esc_attr_e( 'Subscription', 'simple-pay' ); ?>"
				data-amount-type="recurring"
				data-available="<?php echo esc_attr( simpay_subscriptions_enabled() ? 'yes' : 'no' ); ?>"
				data-upgrade-title="<?php echo esc_attr( $upgrade_title ); ?>"
				data-upgrade-description="<?php echo esc_attr( $upgrade_description ); ?>"
				data-upgrade-url="<?php echo esc_url( $upgrade_url ); ?>"
				data-upgrade-purchased-url="<?php echo esc_url( $upgrade_purchased_url ); ?>"
			>
				<?php esc_html_e( 'Subscription', 'simple-pay' ); ?>
			</button>
		</div>
	</fieldset>

		<?php
	}

	/**
	 * Outputs <options> markup for a list of available Stripe currencies.
	 *
	 * @since 4.11.0
	 * @access private
	 *
	 * @param false|string $selected Currently selected option.
	 */
	private function unstable_currency_select_options( $selected = false ) {
		$currencies = simpay_get_currencies();
		$options    = array();

		foreach ( $currencies as $code => $symbol ) {
			$options[] = sprintf(
				'<option value="%1$s" %4$s data-symbol="%3$s">%2$s (%3$s)</option>',
				esc_attr( strtolower( $code ) ),
				esc_html( $code ),
				esc_html( $symbol ),
				selected( $selected, strtolower( $code ), false )
			);
		}

		$options = implode( '', $options );

		echo $options;
	}
}
