<?php
/**
 * Payments: Shared
 *
 * @package SimplePay\Pro\Payments
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.6.0
 */

namespace SimplePay\Pro\Payments;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Stores General field values in object metadata.
 *
 * @since 3.6.0
 *
 * @param array                         $object_args Object arguments.
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 */
function add_general_metadata( $object_args, $form, $form_data, $form_values ) {
	// Can't do anything without custom fields.
	if ( ! isset( $form_values['simpay_field'] ) || empty( $form_values['simpay_field'] ) ) {
		return $object_args;
	}

	foreach ( $form_values['simpay_field'] as $name => $value ) {
		// Skip empty.
		if ( '' === $value ) {
			continue;
		}

		$name  = simpay_truncate_metadata( 'title', $name );
		$value = simpay_truncate_metadata( 'description', $value );

		$name  = sanitize_text_field( stripslashes( $name ) );
		$value = sanitize_text_field( stripslashes( $value ) );

		$object_args['metadata'][ $name ] = $value;
	}

	return $object_args;
}
// Only update if UPE is not enabled. Otherwise it is handled in the updated `wpsp/__internal__payment` endpoint.
if ( ! simpay_is_upe() ) {
	add_filter( 'simpay_get_paymentintent_args_from_payment_form_request', __NAMESPACE__ . '\\add_general_metadata', 10, 4 );
	add_filter( 'simpay_get_subscription_args_from_payment_form_request', __NAMESPACE__ . '\\add_general_metadata', 10, 4 );
	add_filter( 'simpay_get_charge_args_from_payment_form_request', __NAMESPACE__ . '\\add_general_metadata', 10, 4 );
}

/**
 * Stores coupon information in PaymentIntent metadata.
 *
 * @since 3.6.0
 *
 * @param array                         $object_args PaymentIntent arguments.
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @return array
 */
function add_coupon_metadata( $object_args, $form, $form_data, $form_values ) {
	$coupon = isset( $form_values['simpay_coupon'] ) ? esc_html( $form_values['simpay_coupon'] ) : false;

	// Ensure "ghost" coupon metadata is not applied. The coupon field uses the same
	// format as other custom fields. If a user inputs a coupon does not apply it
	// then it will appear as metadata. Ensure it does not.
	if (
		isset( $object_args['metadata'] ) &&
		isset( $object_args['metadata']['coupon'] )
	) {
		unset( $object_args['metadata']['coupon'] );
	}

	if ( false === $coupon ) {
		return $object_args;
	}

	$coupon_details = isset( $form_values['simpay_coupon_details'] ) ? esc_html( $form_values['simpay_coupon_details'] ) : false;

	if ( $coupon_details ) {
		$value = simpay_truncate_metadata( 'description', $coupon_details );
	} else {
		$value = simpay_truncate_metadata( 'description', $coupon );
	}

	$object_args['metadata']['coupon_code'] = $value;

	return $object_args;
}
// Only update if UPE is not enabled. Otherwise it is handled in the updated `wpsp/__internal__payment` endpoint.
if ( ! simpay_is_upe() ) {
	add_filter( 'simpay_get_paymentintent_args_from_payment_form_request', __NAMESPACE__ . '\\add_coupon_metadata', 10, 4 );
	add_filter( 'simpay_get_subscription_args_from_payment_form_request', __NAMESPACE__ . '\\add_coupon_metadata', 10, 4 );
	add_filter( 'simpay_get_charge_args_from_payment_form_request', __NAMESPACE__ . '\\add_coupon_metadata', 10, 4 );
}

/**
 * Stores tax information in PaymentIntent metadata.
 *
 * @since 3.6.0
 *
 * @param array                         $object_args PaymentIntent arguments.
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @return array
 */
function add_tax_metadata( $object_args, $form, $form_data, $form_values ) {
	$tax_status = get_post_meta( $form->id, '_tax_status', true );

	if ( ! (
		empty( $tax_status ) ||
		'fixed-global' === $tax_status
	) ) {
		return $object_args;
	}

	$taxes = simpay_get_setting( 'taxes', 'no' );

	if ( 'yes' !== $taxes ) {
		return $object_args;
	}

	// Retrieve price option.
	$price = simpay_payment_form_prices_get_price_by_id(
		$form,
		$form_data['price']['id']
	);

	if ( false === $price ) {
		return $object_args;
	}

	// Legacy tax percentage.
	if ( has_filter( 'simpay_form_' . $form->id . '_tax_percent' ) ) {
		$tax_percent = floatval(
			simpay_get_filtered(
				'tax_percent',
				0,
				$form->id
			)
		);

		// Tax rates.
	} else {
		$tax_percent = simpay_get_payment_form_tax_percentage(
			$form,
			'exclusive'
		);

		$tax_percent_inclusive = simpay_get_payment_form_tax_percentage(
			$form,
			'inclusive'
		);

		if ( 0 !== $tax_percent_inclusive ) {
			$object_args['metadata']['simpay_tax_percent_inclusive'] =
				$tax_percent_inclusive;
		}
	}

	if ( 0 !== $tax_percent ) {
		$object_args['metadata']['simpay_tax_percent'] = $tax_percent;
	}

	// Reference the tax amount from the tracked DOM value.
	// For the time being it's easier to use the legacy tracked value
	// for the legacy metadata.
	$tax_amount = isset( $form_values['simpay_tax_amount'] )
		? $form_values['simpay_tax_amount']
		: 0;

	if ( 0 !== $tax_amount ) {
		$object_args['metadata']['simpay_tax_amount'] = html_entity_decode(
			simpay_format_currency(
				$tax_amount,
				$price->currency
			)
		);

		$object_args['metadata']['simpay_tax_unit_amount'] = $tax_amount;
	}

	return $object_args;
}
// Only update if UPE is not enabled. Otherwise it is handled in the updated `wpsp/__internal__payment` endpoint.
if ( ! simpay_is_upe() ) {
	add_filter(
		'simpay_get_paymentintent_args_from_payment_form_request',
		__NAMESPACE__ . '\\add_tax_metadata',
		10,
		4
	);
	add_filter(
		'simpay_get_subscription_args_from_payment_form_request',
		__NAMESPACE__ . '\\add_tax_metadata',
		10,
		4
	);
	add_filter(
		'simpay_get_charge_args_from_payment_form_request',
		__NAMESPACE__ . '\\add_tax_metadata',
		10,
		4
	);
}
