<?php
/**
 * Form Field: Price Select
 *
 * @package SimplePay\Pro\Forms\Fields
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 4.1.0
 */

namespace SimplePay\Pro\Forms\Fields;

use SimplePay\Core\Abstracts\Custom_Field;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Price_Select class.
 *
 * @since 4.1.0
 */
class Price_Select extends Custom_Field {

	/**
	 * Prints HTML for price on frontend.
	 *
	 * @since 4.1.0
	 *
	 * @param array $settings Field settings.
	 * @return string
	 */
	public static function print_html( $settings ) {
		$display_type                 = self::get_display_type();
		$is_allow_multiple_line_items = simpay_get_saved_meta(
			self::$form->id,
			'_allow_purchasing_multiple_line_items',
			false
		);

		// if multiple line items are allowed, we need to use basic style.
		if ( 'yes' === $is_allow_multiple_line_items ) {
			$display_type = 'multiline';
		}
		switch ( $display_type ) {
			case 'multiline':
				return self::get_price_selector_radio();
			case 'dropdown':
				return self::get_price_selector_select();
			case 'radio':
				return self::get_price_selector_radio();
			case 'list':
				return self::get_price_selector_list();
			case 'buttons':
				return self::get_price_selector_buttons();
		}
	}

	/**
	 * Returns the markup for the dropdown/select price selector.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	private static function get_price_selector_select() {
		$prices = simpay_get_payment_form_prices( self::$form );
		$id     = self::get_id_attr();

		// Backwards compatibility fallback.
		$label_fallback = simpay_get_saved_meta(
			self::$form->id,
			'_plan_select_form_field_label',
			''
		);

		$display_type  = self::get_display_type();
		$display_style = 1 === count( $prices ) ? 'none' : 'block';

		$wrapper_classes = 'simpay-plan-wrapper simpay-field-wrap simpay-dropdown-wrap';

		ob_start();
		?>

		<div
			class="simpay-form-control simpay-plan-select-container"
			data-display-type="<?php echo esc_attr( $display_type ); ?>"
			style="display: <?php echo esc_attr( $display_style ); ?>"
		>
			<?php
			// phpcs:ignore WordPress.Security.EscapeOutput.
			echo self::get_label( 'label', $label_fallback );
			?>
			<div class="<?php echo esc_attr( $wrapper_classes ); ?>">
				<select id="<?php echo esc_attr( $id ); ?>" name="simpay_price">
				<?php
				/* @var $prices \SimplePay\Core\PaymentForm\PriceOption[] */
				foreach ( $prices as $price ) :
					?>
					<option
						<?php
						// phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
						echo self::get_price_option_atts( $price, 'select', '' );
						?>
					>
						<?php echo esc_html( $price->get_display_label() ); ?>
					</option>
				<?php endforeach; ?>
				</select>
			</div>
		</div>

		<?php
		return ob_get_clean();
	}

	/**
	 * Returns the markup for the radio price selector.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	private static function get_price_selector_radio() {
		$prices = simpay_get_payment_form_prices( self::$form );
		$id     = self::get_id_attr();
		// Backwards compatibility fallback.
		$label_fallback = simpay_get_saved_meta(
			self::$form->id,
			'_plan_select_form_field_label',
			''
		);

		$label = isset( self::$settings['label'] )
			? self::$settings['label']
			: $label_fallback;

		$label_classes = array();

		if ( empty( $label ) ) {
			$label           = self::$settings['type'];
			$label_classes[] = 'screen-reader-text';
		}

		$wrapper_classes = 'simpay-plan-wrapper simpay-field-wrap simpay-dropdown-wrap';

		$display_type = self::get_display_type();

		$is_allow_multiple_line_items = self::$form->allows_multiple_line_items();

		$multiple_with_price_settings = array_filter(
			$prices,
			function ( $price ) {
				return $price->quantity_toggle || $price->can_recur || $price->unit_amount_min;
			}
		);

		$is_single_price_option = 1 === count( $prices );

		$display_style = (
			(
				$is_allow_multiple_line_items &&
				$is_single_price_option &&
				0 === count( $multiple_with_price_settings )
			) ||
			( $is_single_price_option && ! $is_allow_multiple_line_items )
		)
			? 'none'
			: 'block';

		ob_start();
		?>

		<fieldset
			class="simpay-form-control simpay-plan-select-container"
			data-display-type="<?php echo esc_attr( $display_type ); ?>"
			style="display: <?php echo esc_attr( $display_style ); ?>"
		>
			<div class="simpay-<?php echo esc_attr( self::$settings['type'] ); ?>-label simpay-label-wrap">
				<legend class="<?php echo esc_attr( implode( ' ', $label_classes ) ); ?>">
					<?php echo esc_html( $label ); ?>
				</legend>
			</div>

			<div class="<?php echo esc_attr( $wrapper_classes ); ?>">
				<ul class="simpay-multi-plan-radio-group">
				<?php
				/* @var $prices \SimplePay\Core\PaymentForm\PriceOption[] */
				foreach ( $prices as $price ) :
					$id = self::get_unique_price_id( $price );
					?>
					<li class="simpay-price-selection-label">
						<?php if ( $is_allow_multiple_line_items ) : ?>
							<label for="<?php echo esc_attr( $id ); ?>">
								<input
									<?php if ( $is_single_price_option ) : ?>
									disabled="disabled"
									<?php endif ?>
									type="<?php echo esc_attr( self::get_input_type() ); ?>"
									<?php
									// phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
									echo self::get_price_option_atts( $price, 'radio', $id );
									?>
								/>

								<?php echo esc_html( $price->get_display_label() ); ?>
							</label>

							<?php
							if (
								$price->unit_amount_min ||
								$price->quantity_toggle ||
								$price->can_recur
							) :
								?>
							<div class="simpay-additional-fields">
								<?php if ( $price->unit_amount_min ) : ?>
									<?php
									$custom_amount_settings = array(
										'default_price'   => $price,
										'prefill_default' => 'yes',
										'form'            => self::$form,
										'multiple'        => true,
									);

									$custom_amount_allowed_html = array(
										'input' => array(
											'type'        => array(),
											'name'        => array(),
											'value'       => array(),
											'placeholder' => array(),
											'class'       => array(),
											'data-prefill-default' => array(),
											'data-placeholder' => array(),
										),
										'label' => array(
											'for'   => array(),
											'class' => array(),
										),
										'span'  => array(
											'class' => array(),
											'id'    => array(),
										),
										'div'   => array(
											'class' => array(),
											'id'    => array(),
										),
										'style' => array(),

									);

									$custom_amount = new Custom_Amount();

									echo wp_kses(
										$custom_amount->print_html(
											$custom_amount_settings
										),
										$custom_amount_allowed_html
									);
									?>
								<?php endif; ?>

								<?php
								if ( $price->quantity_toggle ) :
									$max_attr = isset( $price->quantity_maximum )
										? min(
											$price->quantity_maximum,
											self::$form->is_managing_inventory()
												? $price->get_remaining_stock()
												: $price->quantity_maximum
										)
										: ''
									?>
									<div
										class="simpay-quantity-field"
										style="display: flex; align-items: center; gap: 8px;"
									>
										<span>&times;</span>

										<input
											value="<?php echo esc_attr( isset( $price->quantity_minimum ) ? $price->quantity_minimum : 1 ); ?>"
											min="<?php echo esc_attr( isset( $price->quantity_minimum ) ? $price->quantity_minimum : 1 ); ?>"
											max="<?php echo esc_attr( $max_attr ); ?>"
											class="simpay-field simpay-quantity-field-input small-text"
											type="number"
										/>
									</div>
								<?php endif; ?>

								<?php
								if ( $price->can_recur ) :
									$recurring_amount_toggle_settings = array(
										'label'    => isset( $price->recurring_amount_toggle_label )
											? $price->recurring_amount_toggle_label
											: esc_html__(
												'Make this a recurring amount',
												'simple-pay'
											),
										'selected' => $price->can_recur_selected_by_default ? 'yes' : 'no',
									);

									$recurring_amount_toggle_allowed_html = array(
										'input' => array(
											'type'    => array(),
											'name'    => array(),
											'id'      => array(),
											'class'   => array(),
											'checked' => array(),
										),
										'label' => array(
											'for'   => array(),
											'class' => array(),
										),
										'div'   => array(
											'class' => array(),
										),
										'style' => array(),
									);

									$recurring_amount_toggle = new Recurring_Amount_Toggle();

									echo wp_kses(
										$recurring_amount_toggle->print_html(
											$recurring_amount_toggle_settings
										),
										$recurring_amount_toggle_allowed_html
									);
								endif;
								?>
							</div>
							<?php endif; ?>
						<?php else : ?>
							<label for="<?php echo esc_attr( $id ); ?>">
							<input
								type="radio"
								<?php
								// phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
								echo self::get_price_option_atts( $price, 'radio', $id );
								?>
							/>
							<?php echo esc_html( $price->get_display_label() ); ?>
						</label>
						<?php endif; ?>
					</li>
				<?php endforeach; ?>
				</ul>
			</div>
		</fieldset>

		<?php if ( $is_allow_multiple_line_items ) : ?>
			<div
				class="simpay-errors simpay-custom-amount-error"
				aria-live="assertive"
				aria-relevant="additions text"
				aria-atomic="true"
			></div>
			<?php
		endif;

		return ob_get_clean();
	}

	/**
	 * Returns the markup for the list price selector.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	private static function get_price_selector_list() {
		$prices = simpay_get_payment_form_prices( self::$form );
		$id     = self::get_id_attr();

		// Backwards compatibility fallback.
		$label_fallback = simpay_get_saved_meta(
			self::$form->id,
			'_plan_select_form_field_label',
			''
		);

		$label = isset( self::$settings['label'] )
			? self::$settings['label']
			: $label_fallback;

		$label_classes = array();

		if ( empty( $label ) ) {
			$label           = self::$settings['type'];
			$label_classes[] = 'screen-reader-text';
		}

		$wrapper_classes = 'simpay-plan-wrapper simpay-field-wrap';

		$display_type = self::get_display_type();

		$display_style = 1 === count( $prices ) ? 'none' : 'block';

		ob_start();
		?>

		<fieldset
			class="simpay-form-control simpay-plan-select-container"
			data-display-type="<?php echo esc_attr( $display_type ); ?>"
			style="display: <?php echo esc_attr( $display_style ); ?>"
		>
			<div class="simpay-<?php echo esc_attr( self::$settings['type'] ); ?>-label simpay-label-wrap">
				<legend class="<?php echo esc_attr( implode( ' ', $label_classes ) ); ?>">
					<?php echo esc_html( $label ); ?>
				</legend>
			</div>

			<div class="<?php echo esc_attr( $wrapper_classes ); ?>">
				<ul class="simpay-multi-plan-list-group">
				<?php
				/* @var $prices \SimplePay\Core\PaymentForm\PriceOption[] */
				foreach ( $prices as $price ) :
					$custom_label = $price->label;
					$amount       = $price->get_generated_label();

					$label = ! empty( $custom_label ) ? $custom_label : $amount;

					$id = self::get_unique_price_id( $price );
					?>
					<li>
						<input
							type="<?php echo esc_attr( self::get_input_type() ); ?>"
							<?php
							// phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
							echo self::get_price_option_atts( $price, 'radio', $id );
							?>
						/>
						<label for="<?php echo esc_attr( $id ); ?>">
							<?php echo esc_html( $label ); ?>
							<?php if ( ! empty( $custom_label ) ) : ?>
							<small><?php echo esc_html( $amount ); ?></small>
							<?php endif; ?>

							<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg>
						</label>
					</li>
				<?php endforeach; ?>
				</ul>
			</div>
		</fieldset>

		<?php
		return ob_get_clean();
	}

	/**
	 * Returns the markup for the buttons price selector.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	private static function get_price_selector_buttons() {
		$prices = simpay_get_payment_form_prices( self::$form );
		$id     = self::get_id_attr();

		// Backwards compatibility fallback.
		$label_fallback = simpay_get_saved_meta(
			self::$form->id,
			'_plan_select_form_field_label',
			''
		);

		$label = isset( self::$settings['label'] )
			? self::$settings['label']
			: $label_fallback;

		$label_classes = array();

		if ( empty( $label ) ) {
			$label           = self::$settings['type'];
			$label_classes[] = 'screen-reader-text';
		}

		$wrapper_classes = 'simpay-plan-wrapper simpay-field-wrap';

		$display_type = self::get_display_type();

		$display_style = 1 === count( $prices ) ? 'none' : 'block';

		ob_start();
		?>

		<fieldset
			class="simpay-form-control simpay-plan-select-container"
			data-display-type="<?php echo esc_attr( $display_type ); ?>"
			style="display: <?php echo esc_attr( $display_style ); ?>"
		>
			<div class="simpay-<?php echo esc_attr( self::$settings['type'] ); ?>-label simpay-label-wrap">
				<legend class="<?php echo esc_attr( implode( ' ', $label_classes ) ); ?>">
					<?php echo esc_html( $label ); ?>
				</legend>
			</div>

			<div class="<?php echo esc_attr( $wrapper_classes ); ?>">
				<ul class="simpay-multi-plan-buttons-group">
				<?php
				/* @var $prices \SimplePay\Core\PaymentForm\PriceOption[] */
				foreach ( $prices as $price ) :
					$label = $price->get_simplified_label();

					$is_custom = ! simpay_payment_form_prices_is_defined_price(
						$price->id
					);

					$classname = 'simpay-multi-plan-buttons-group__' . (
						$is_custom ? 'custom' : 'defined'
					);

					$id = self::get_unique_price_id( $price );
					?>
					<li class="<?php echo esc_attr( $classname ); ?>">
						<input
							type="<?php echo esc_attr( self::get_input_type() ); ?>"
							<?php
							// phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
							echo self::get_price_option_atts( $price, 'radio', $id );
							?>
						/>
						<label for="<?php echo esc_attr( $id ); ?>">
							<?php echo esc_html( $label ); ?>
						</label>
					</li>
				<?php endforeach; ?>
				</ul>
			</div>
		</fieldset>

		<?php
		return ob_get_clean();
	}

	/**
	 * Returns markup for price option attributes.
	 *
	 * @since 4.1.0
	 *
	 * @param \SimplePay\Core\PaymentForm\PriceOption $price Price option.
	 * @param string                                  $display_type Price option display type.
	 * @param string                                  $id HTML element ID (unique).
	 * @return string
	 */
	private static function get_price_option_atts( $price, $display_type, $id ) {
		$selected_func = 'radio' === $display_type ? 'checked' : 'selected';
		$is_multiple   = self::$form->allows_multiple_line_items();

		return sprintf(
			'
			id="%1$s"
			class="simpay-multi-sub simpay-price-option-%8$s"
			data-price=\'%2$s\'
			%3$s
			%4$s
			%5$s
			%6$s
			%7$s
			',
			esc_attr( $id ),
			wp_json_encode(
				array_merge(
					$price->to_array(),
					array(
						'generated_label'     => $price->get_generated_label(),
						'simplified_label'    => $price->get_simplified_label(),
						'currency_min_amount' => simpay_get_currency_minimum(
							$price->currency
						),
					)
				),
				JSON_HEX_QUOT | JSON_HEX_APOS
			),
			in_array( $display_type, array( 'radio', 'list' ), true )
				? 'name="simpay_price"'
				: '',
			$selected_func( true, ( $price->default && $price->is_in_stock() ) || ( $is_multiple && $price->required ), false ),
			disabled( false, ( $price->is_in_stock() && ! ( $is_multiple && $price->required ) ), false ),
			// Backwards compatibility attributes.
			self::get_price_option_atts_price_id( $price ),
			self::get_price_option_atts_recurring( $price ),
			esc_attr( $price->id )
		);
	}

	/**
	 * Returns markup for price option attributes when a Price/Plan
	 * has been defined.
	 *
	 * These attributes are no longer used by the plugin but remain in the DOM
	 * for backwards compatibility.
	 *
	 * @since 4.1.0
	 *
	 * @param \SimplePay\Core\PaymentForm\PriceOption $price Price option.
	 * @return string
	 */
	private static function get_price_option_atts_price_id( $price ) {
		if ( null === $price->id ) {
			return 'value=""';
		}

		$currency_amount = simpay_convert_amount_to_dollars(
			$price->unit_amount
		);

		return sprintf(
			'
			value="%1$s"
			data-plan-id="%1$s"
			data-plan-amount="%2$s"',
			esc_attr( $price->id ),
			esc_attr( $currency_amount )
		);
	}

	/**
	 * Returns markup for price option attributes when a Price/Plan
	 * is recurring.
	 *
	 * These attributes are no longer used by the plugin but remain in the DOM
	 * for backwards compatibility.
	 *
	 * @since 4.1.0
	 *
	 * @param \SimplePay\Core\PaymentForm\PriceOption $price Price option.
	 * @return string
	 */
	private static function get_price_option_atts_recurring( $price ) {
		if ( true === $price->can_recur || null === $price->recurring ) {
			return '';
		}

		$currency_setup_fee_amount = '';

		if ( null !== $price->line_items && isset( $price->line_items[0] ) ) {
			$currency_setup_fee_amount = simpay_convert_amount_to_dollars(
				$price->line_items[0]['unit_amount']
			);
		}

		$invoice_limit = 0;

		if ( null !== $price->recurring && isset( $price->recurring['invoice_limit'] ) ) {
			$invoice_limit = absint( $price->recurring['invoice_limit'] );
		}

		$trial_period_days = null !== $price->recurring
			&& isset( $price->recurring['trial_period_days'] );

		return sprintf(
			'
			data-plan-setup-fee="%1$s"
			data-plan-interval="%2$s"
			data-plan-interval-count="%3$s"
			data-plan-max-charges="%4$s"
			data-plan-trial="%5$s"
			',
			esc_attr( $currency_setup_fee_amount ),
			esc_attr( $price->recurring['interval'] ),
			esc_attr( $price->recurring['interval_count'] ),
			esc_attr( $invoice_limit ),
			esc_attr( $trial_period_days )
		);
	}

	/**
	 * Retrieves the field's display type.
	 *
	 * Falls back to the legacy "User select" display style.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	private static function get_display_type() {
		$legacy_display_type = simpay_get_saved_meta(
			self::$form->id,
			'_multi_plan_display',
			'radio'
		);

		$display_type = ! empty( self::$settings['display_type'] )
			? self::$settings['display_type']
			: $legacy_display_type;

		return $display_type;
	}

	/**
	 * Retrieves an unique ID for HTML to avoid duplicate elements.
	 *
	 * Based off the price option ID and a random number.
	 *
	 * @since 4.2.0
	 *
	 * @param \SimplePay\Core\PaymentForm\PriceOption $price Price option.
	 * @return string
	 */
	private static function get_unique_price_id( $price ) {
		return $price->id . '-instance-' . rand( 0, 1000 );
	}

	/**
	 * Retrieves the input type for the price selector.
	 *
	 * @since 4.11.0
	 *
	 * @return string
	 */
	private static function get_input_type() {
		$is_allow_multiple_line_items = simpay_get_saved_meta( self::$form->id, '_allow_purchasing_multiple_line_items' );

		if ( 'yes' === $is_allow_multiple_line_items ) {
			return 'checkbox';
		}

		return 'radio';
	}
}
