<?php
/**
 * Forms field: Date
 *
 * @package SimplePay\Pro\Forms\Fields
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.0.0
 */

namespace SimplePay\Pro\Forms\Fields;

use SimplePay\Core\Abstracts\Custom_Field;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Date class.
 *
 * @since 3.0.0
 */
class Date extends Custom_Field {

    /**
	 * Prints HTML for field on frontend.
	 *
	 * @since 3.0.0
	 *
	 * @param array $settings Field settings.
	 * @return string
	 */
	public static function print_html( $settings ) {
		$id          = self::get_id_attr();
		$default     = self::get_default_value();
		$placeholder = isset( $settings['placeholder'] ) ? $settings['placeholder'] : '';
		$required    = isset( $settings['required'] );

		$meta_name = isset( $settings['metadata'] ) && ! empty( $settings['metadata'] )
			? $settings['metadata']
			: $id;
		$name      = 'simpay_field[' . esc_attr( $meta_name ) . ']';

        $date_format = self::get_date_format( self::$form );

        $elements_locale   = simpay_get_setting( 'stripe_elements_locale', '' );
        $flatpickr_locales = self::get_flatpickr_locales();

		if ( in_array( $elements_locale, $flatpickr_locales, true ) ) {
			wp_enqueue_script(
				sprintf( 'simpay-date-flatpickr-%s', $elements_locale ),
				sprintf( 'https://npmcdn.com/flatpickr/dist/l10n/%s.js', $elements_locale ),
				array(
						'simpay-date-flatpickr',
				),
				'4.6.13',
				true
			);
		}

		wp_enqueue_style(
			'simpay-date-flatpickr',
			SIMPLE_PAY_INC_URL . 'pro/assets/css/vendor/flatpickr/flatpickr.min.css',
			array(),
			'4.6.13'
		);

		wp_enqueue_script(
			'simpay-date-flatpickr',
			SIMPLE_PAY_INC_URL . 'pro/assets/js/vendor/flatpickr/flatpickr.min.js',
			array(),
			'4.6.13',
			true
		);

		ob_start();
		?>

		<div class="simpay-form-control simpay-date-container">
			<div class="simpay-date-label simpay-label-wrap">
				<?php echo self::get_label(); // WPCS: XSS okay. ?>
				<div class="simpay-date-wrap simpay-field-wrap">
					<input
						type="text"
						name="<?php echo esc_attr( $name ); ?>"
						class="simpay-date-input"
						id="<?php echo esc_attr( $id ); ?>"
						placeholder="<?php echo esc_attr( $placeholder ); ?>"
						value="<?php echo esc_attr( $default ); ?>"
						<?php if ( true === $required ) : ?>
							required
						<?php endif; ?>
					/>

					<script type="text/javascript">
						document.addEventListener( 'DOMContentLoaded', function() {
							flatpickr( '#<?php echo esc_attr( $id ); ?>', {
								dateFormat: '<?php echo $date_format; ?>',
								<?php if ( in_array( $elements_locale, $flatpickr_locales, true ) ) : ?>
								locale: '<?php echo $elements_locale; ?>',
								<?php endif; ?>
							} );
						} );
					</script>
				</div>
			</div>
		</div>

		<?php
		return ob_get_clean();
	}

	/**
	 * Returns the date picker date format for Flatpickr.
	 *
	 * Provides a compatibility layer to attempt to convert jQuery UI Datepicker
	 * date formats (old) to Flatpickr date formats (new).
	 *
	 * @see https://github.com/awesomemotive/wp-simple-pay-pro/issues/2655
	 *
	 * @since 4.7.7
	 *
	 * @param \SimplePay\Pro\Forms\Pro_Form $form Payment form.
	 * @return string
	 */
	private static function get_date_format( $form ) {
		$date_format = $form->date_format;

		if (empty($date_format)) {
			$date_format = get_option('date_format');
		}

		$conversion_map = array(
			'yy' => 'Y',
			'mm' => 'm',
			'dd' => 'd',
		);

		$old_keys = array_keys( $conversion_map );
		$legacy   = false;

		// If the date format does not contain any of the old keys, return the date format as-is.
		foreach ( $old_keys as $token ) {
			if ( strpos( $date_format, $token ) !== false ) {
				$legacy = true;
				break;
			}
		}

		if ( ! $legacy ) {
			return $date_format;
		}

		// Convert the date format to the new format.
		return strtr( $date_format, $conversion_map );
	}

	/**
	 * Provides list of available flatpickr locales.
	 *
	 * @since 4.7.7
	 *
	 * @return string[]
	 */
	private static function get_flatpickr_locales() {
		return array(
			'ar-dz',
			'ar',
			'at',
			'az',
			'be',
			'bg',
			'bn',
			'bs',
			'cat',
			'ckb',
			'cs',
			'cy',
			'da',
			'de',
			'eo',
			'es',
			'et',
			'fa',
			'fi',
			'fo',
			'fr',
			'ga',
			'gr',
			'he',
			'hi',
			'hr',
			'hu',
			'hy',
			'id',
			'is',
			'it',
			'ja',
			'ka',
			'km',
			'ko',
			'kz',
			'lt',
			'lv',
			'mk',
			'mn',
			'ms',
			'my',
			'nl',
			'nn',
			'no',
			'pa',
			'pl',
			'pt',
			'ro',
			'ru',
			'si',
			'sk',
			'sl',
			'sq',
			'sr-cyr',
			'sr',
			'sv',
			'th',
			'tr',
			'uk',
			'uz_latn',
			'uz',
			'vn',
			'zh_tw',
			'zh'
		);
	}
}
