<?php
/**
 * Forms field: Address
 *
 * @package SimplePay\Pro\Forms\Fields
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.0.0
 */

namespace SimplePay\Pro\Forms\Fields;

use SimplePay\Core\Abstracts\Custom_Field;
use SimplePay\Core\i18n;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Address class.
 *
 * @since 3.0.0
 */
class Address extends Custom_Field {

	/**
	 * Prints HTML for field on frontend.
	 *
	 * @since 3.0.0
	 *
	 * @param array $settings Field settings.
	 * @return string
	 */
	public static function print_html( $settings ) {
		return self::get_upe_address_field( $settings );
	}


	/**
	 * Returns the markup for the "Address" custom field when using the UPE.
	 *
	 * @since 4.7.0
	 *
	 * @param array<string, string> $settings The field settings.
	 */
	private static function get_upe_address_field( $settings ) {
		// Billing container label.
		$billing_address_label = isset( $settings['billing-container-label'] )
			? $settings['billing-container-label']
			: esc_html__( 'Billing Address', 'simple-pay' );

		// Shipping toggle.
		$collect_shipping = isset( $settings['collect-shipping'] ) && 'yes' === $settings['collect-shipping'];

		// Shipping container label.
		$shipping_address_label = isset( $settings['shipping-container-label'] )
			? $settings['shipping-container-label']
			: esc_html__( 'Shipping Address', 'simple-pay' );

		ob_start();
		?>

		<?php if ( false === $collect_shipping ) : ?>

		<fieldset class="simpay-form-control simpay-address-container simpay-address-container-upe simpay-billing-address-container">
			<?php if ( ! empty( $billing_address_label ) ) : ?>
			<legend class="simpay-address-billing-container-label simpay-label-wrap">
				<?php echo esc_html( $billing_address_label ); ?>
			</legend>
			<?php endif; ?>

			<?php echo self::get_upe_address_field_input(); ?>
		</fieldset>

		<?php else : ?>

		<fieldset class="simpay-form-control simpay-address-container simpay-address-container-upe simpay-shipping-address-container">
			<legend class="simpay-address-shipping-container-label simpay-label-wrap">
				<?php echo esc_html( $shipping_address_label ); ?>
			</legend>

			<?php echo self::get_upe_address_field_input(); ?>
		</fieldset>
		<?php endif; ?>

		<div class="simpay-address-error simpay-errors" aria-live="assertive" aria-relevant="additions text" aria-atomic="true"></div>

		<?php
		return ob_get_clean();
	}

	/**
	 * Returns the markup for the "Address" custom field when using the UPE.
	 *
	 * @since 4.6.7
	 *
	 * @return string
	 */
	private static function get_upe_address_field_input() {
		ob_start();

		/**
		 * Filters the list of countries that can be selected in the Address Element.
		 *
		 * @since 4.6.7
		 *
		 * @param array<string, string> $allowed_countries List of countries.
		 */
		$allowed_countries = apply_filters(
			'simpay_custom_field_address_allowed_countries',
			array()
		);

		$line1       = self::get_default_value( 'default_line1' );
		$line2       = self::get_default_value( 'default_line2' );
		$city        = self::get_default_value( 'default_city' );
		$state       = self::get_default_value( 'default_state' );
		$postal_code = self::get_default_value( 'default_postal_code' );
		$country     = self::get_default_value( 'default_country' );
		?>
			<div
				class="simpay-address-element"
				data-allowed-countries='<?php echo wp_json_encode( $allowed_countries ); ?>'
				data-default-line1="<?php echo esc_attr( $line1 ); ?>"
				data-default-line2="<?php echo esc_attr( $line2 ); ?>"
				data-default-city="<?php echo esc_attr( $city ); ?>"
				data-default-state="<?php echo esc_attr( $state ); ?>"
				data-default-postal-code="<?php echo esc_attr( $postal_code ); ?>"
				data-default-country="<?php echo esc_attr( $country ); ?>"
			></div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Returns a standard address field.
	 *
	 * @since 3.9.0
	 *
	 * @param string $field             Field type.
	 * @param string $field_id          Field ID.
	 * @param string $field_name        Field name.
	 * @param string $field_default     Field default.
	 * @param string $field_placeholder Field placeholder.
	 */
	private static function get_field(
		$field,
		$field_id,
		$field_name,
		$field_default,
		$field_placeholder
	) {
		$required = isset( self::$settings['required'] );
		?>

		<div class="simpay-form-control simpay-address-<?php echo esc_attr( $field ); ?>-container">
			<?php echo self::get_label( 'label-' . $field ); // WPCS: XSS okay. ?>
			<div class="simpay-<?php echo esc_attr( $field ); ?>-wrap simpay-field-wrap">
				<input
					type="text"
					name="<?php echo esc_attr( $field_name ); ?>"
					id="<?php echo esc_attr( $field_id ); ?>"
					class="simpay-address-<?php echo esc_attr( $field ); ?>"
					value="<?php echo esc_attr( $field_default ); ?>"
					placeholder="<?php echo esc_attr( $field_placeholder ); ?>"
					maxlength="500"
					<?php if ( true === $required ) : ?>
						required
					<?php endif; ?>
				/>
			</div>
		</div>

		<?php
	}

	/**
	 * Returns a country address field.
	 *
	 * @since 3.9.0
	 *
	 * @param string $field             Field type.
	 * @param string $field_id          Field ID.
	 * @param string $field_name        Field name.
	 * @param string $field_default     Field default.
	 * @param string $field_placeholder Field placeholder.
	 */
	private static function get_country_field(
		$field,
		$field_id,
		$field_name,
		$field_default,
		$field_placeholder
	) {
		$required = isset( self::$settings['required'] );

		$tax_status       = get_post_meta( self::$form->id, '_tax_status', true );
		$countries        = i18n\get_countries();
		$selected_country = self::get_default_value( 'default-country', 'US' );
		?>

		<div class="simpay-form-control simpay-address-<?php echo esc_attr( $field ); ?>-container">
			<?php echo self::get_label( 'label-' . $field ); // WPCS: XSS okay. ?>
			<div class="simpay-<?php echo esc_attr( $field ); ?>-wrap simpay-field-wrap">
				<select
					name="<?php echo esc_attr( $field_name ); ?>"
					id="<?php echo esc_attr( $field_id ); ?>"
					class="simpay-address-<?php echo esc_attr( $field ); ?>"
					<?php if ( true === $required ) : ?>
						required
					<?php endif; ?>
				>
					<option value="">
						<?php esc_html_e( 'Select a country&hellip;', 'simple-pay' ); ?>
					</option>
					<?php foreach ( $countries as $country_code => $country ) : ?>
					<option
						value="<?php echo esc_attr( $country_code ); ?>"
						<?php
						if ( 'automatic' !== $tax_status ) :
							selected( $country_code, $selected_country );
						endif;
						?>
					>
						<?php echo esc_html( $country ); ?>
					</option>
					<?php endforeach; ?>
				</select>
			</div>
		</div>

		<?php
	}
}
