<?php
/**
 * Forms field: Telephone
 *
 * @package SimplePay\Pro\Forms\Fields
 * @copyright Copyright (c) 2022, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.5.0
 */

namespace SimplePay\Pro\Forms\Fields;

use SimplePay\Core\Abstracts\Custom_Field;
use SimplePay\Core\i18n;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Telephone class.
 *
 * @since 3.5.0
 */
class Telephone extends Custom_Field {

	/**
	 * Prints HTML for field on frontend.
	 *
	 * @since 3.5.0
	 *
	 * @param array $settings Field settings.
	 * @return string
	 */
	public static function print_html( $settings ) {
		$id          = self::get_id_attr();
		$default     = self::get_default_value();
		$placeholder = isset( $settings['placeholder'] ) ? $settings['placeholder'] : '';
		$required    = isset( $settings['required'] );
		$is_intl     = isset( $settings['smart'] );

		//
		// Enqueue the scripts and styles inline. This is done for two reasons:
		// 1. Our existing asset loader loads assets on all pages, and these assets are not global.
		// 2. Our existing asset build system attempts to chunk/rename the images, but we don't want that.
		//
		// The remaining functionality/asset usage is handled amongst the existing assets.
		//
		if ( true === $is_intl ) {
			wp_enqueue_style(
				'simpay-intl-tel-input',
				SIMPLE_PAY_INC_URL . 'pro/assets/css/vendor/intlTelInput/intlTelInput.css',
				array(),
				'18.1.4'
			);

			wp_add_inline_style(
				'simpay-intl-tel-input',
				sprintf(
					'.iti__flag { background-image: url( "%1$s" ); } @media (-webkit-min-device-pixel-ratio: 2), (min-resolution: 192dpi) { .iti__flag { background-image: url( "%2$s" ); } }',
					SIMPLE_PAY_INC_URL . 'pro/assets/img/vendor/flags.png',
					SIMPLE_PAY_INC_URL . 'pro/assets/img/vendor/flags@2x.png'
				)
			);

			wp_enqueue_script(
				'simpay-intl-tel-input',
				SIMPLE_PAY_INC_URL . 'pro/assets/js/vendor/intlTelInput/intlTelInput.min.js',
				array(),
				'18.1.4',
				true
			);

			$countries = array_keys(
				array_change_key_case(
					i18n\get_countries(),
					CASE_LOWER
				)
			);

			/**
			 * Allows the country list to be filtered
			 *
			 * @since 4.7.4
			 * @param array<string> $countries List of country codes.
			 * @return array<string>
			 */
			$countries = apply_filters( 'simpay_phone_intl_tel_countries', $countries );

			$initial_country = isset( $settings['country'] ) ?
				strtolower( $settings['country'] )
				: 'us';

			if ( ! in_array( $initial_country, $countries, true ) ) {
				$initial_country = current( $countries );
			}

			wp_localize_script(
				'simpay-intl-tel-input',
				'simpayIntlTelInput',
				array(
					'utilsScript'   => SIMPLE_PAY_INC_URL . 'pro/assets/js/vendor/intlTelInput/utils.js',
					'onlyCountries' => $countries,
					'initialCountry' => $initial_country,
				)
			);
		}

		ob_start();
		?>

		<div class="simpay-form-control simpay-telephone-container">
			<?php echo self::get_label(); // WPCS: XSS okay. ?>
			<div class="simpay-telephone-wrap simpay-telephone-field simpay-field-wrap">
				<input
					type="tel"
					<?php
						// The IntlTelInput library creates a hidden field with the "unformatted" value.
						// It gets created with the same simpay_telephone name so no server side changes are needed.
						if ( true === $is_intl ) :
					?>
						name="simpay_telephone_intl"
					<?php else : ?>
						name="simpay_telephone"
					<?php endif; ?>
					id="<?php echo esc_attr( $id ); ?>"
					class="simpay-telephone"
					value="<?php echo esc_attr( $default ); ?>"
					placeholder="<?php echo esc_attr( $placeholder ); ?>"
					<?php if ( true === $required ) : ?>
						required
					<?php endif; ?>
				/>
			</div>
		</div>

		<?php if ( $is_intl ) : ?>
			<div class="simpay-phone-error simpay-errors" aria-live="assertive" aria-relevant="additions text" aria-atomic="true"></div>
			<?php
		endif;

		return ob_get_clean();
	}

}
