<?php

class MeowPro_WR2X_Rest
{
  private $core;
	private $namespace = 'wp-retina-2x/v1';

	public function __construct( $core ) {
    $this->core = $core;
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );
	}

	function rest_api_init() {
		if ( !current_user_can( 'administrator' ) ) {
			return;
		} 

		register_rest_route( $this->namespace, '/replace', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_replace' )
		) );
		register_rest_route( $this->namespace, '/upload_retina', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_upload_retina' )
		) );
		register_rest_route( $this->namespace, '/upload_retina_fullsize', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_upload_retina_fullsize' )
		) );
		register_rest_route( $this->namespace, '/delete_retina_fullsize', array(
			'methods' => 'POST',
			'permission_callback' => '__return_true',
			'callback' => array( $this, 'rest_delete_retina_fullsize' )
		) );
  }

	function rest_delete_retina_fullsize( $request ) {
		if ( !current_user_can( 'upload_files' ) ) {
			$this->core->log( "You do not have permission to upload files." );
			return __( "You do not have permission to upload files.", 'wp-retina-2x' );
		}

		// Check errors
		$params = $request->get_json_params();
		$mediaId = isset( $params['mediaId'] ) ? (int)$params['mediaId'] : null;
		if ( empty( $mediaId ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "The Media ID is required." ] );
		}

		// Delete Retina
		$this->core->delete_retina_fullsize( $mediaId );
		$info = $this->core->get_media_status_one( $mediaId );
		return new WP_REST_Response( [ 'success' => true, 'data' => $info  ], 200 );
	}

	function rest_upload_retina( $request ) {
		require_once ABSPATH . 'wp-admin/includes/image.php';
		$files = $request->get_file_params();
		$file = $files['file'];
		$tmpfname = $file['tmp_name'];
		$filename = $file['name'];

		// Check errors
		if ( empty( $tmpfname ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "A file is required." ] );
		}
		$error = $this->check_upload( $tmpfname, $filename );
		if ( !empty( $error ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $error ], 200 );
		}

		try {
			$ftype = wp_check_filetype( $filename );
			$image = wp_get_image_editor( $tmpfname, array ( 'mime_type' => $ftype['type'] ) );
			$size = $image->get_size();

			// Halve the size of the uploaded image
			if ( $size['width'] >= $size['height'] ) $image->resize( round($size['width'] * .5), null );
			else $image->resize( null, round($size['height'] * .5) );
			$image->set_quality( get_option('wr2x_quality', 90) );
			$halved = $image->save( $tmpfname . 'H', $ftype['type'] );
			if ( !$halved ) throw new Exception( "Failed to halve the uploaded image" );
			if ( is_wp_error($halved) ) throw new Exception( $halved->get_error_message() );

			// Upload the halved image
			$content = file_get_contents( $halved['path'] );
			if ( $content === false ) throw new Exception( "Couldn't read the uploaded file: {$halved['file']}" );
			$uploaded = wp_upload_bits( $filename, null, $content );
			if ( isset($uploaded['error']) && $uploaded['error'] ) throw new Exception( $uploaded['error'] );

			// Register the file as a new attachment
			$attachTo = 0; // TODO Support specifying which post the media attach to
			$attachment = array (
				'post_mime_type' => $ftype['type'],
				'post_parent' => $attachTo,
				'post_title' => preg_replace( '/\.[^.]+$/', '', $filename ),
				'post_content' => '',
				'post_status' => 'inherit'
			);
			$attachmentId = wp_insert_attachment( $attachment, $uploaded['file'], $attachTo );
			if ( !$attachmentId ) throw new Exception( "Couldn't add an attachment file: {$uploaded['file']}" );
			if ( is_wp_error($attachmentId) ) throw new Exception( $attachmentId->get_error_message() );
			$meta = wp_generate_attachment_metadata( $attachmentId, $uploaded['file'] );
			wp_update_attachment_metadata( $attachmentId, $meta );
			return $this->rest_upload_retina_fullsize( $request, $attachmentId );
		} 
		catch ( Exception $e ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $e->getMessage() ] );
		}
	}

	function check_upload( $tmpfname, $filename ) {
		if ( !current_user_can( 'upload_files' ) ) {
			$this->core->log( "You do not have permission to upload files." );
			unlink( $tmpfname );
			return __( "You do not have permission to upload files.", 'wp-retina-2x' );
		}
		$file_info = getimagesize( $tmpfname );
		if ( empty( $file_info ) ) {
			$this->core->log( "The file is not an image or the upload went wrong." );
			unlink( $tmpfname );
			return __( "The file is not an image or the upload went wrong.", 'wp-retina-2x' );
		}
		$filedata = wp_check_filetype_and_ext( $tmpfname, $filename );
		if ( $filedata["ext"] == "" ) {
			$this->core->log( "You cannot use this file (wrong extension? wrong type?)." );
			unlink( $tmpfname );
			return __( "You cannot use this file (wrong extension? wrong type?).", 'wp-retina-2x' );
		}
		return null;
	}

	function rest_replace( $request ) {
		require_once ABSPATH . 'wp-admin/includes/image.php';
		$params = $request->get_body_params();
		$mediaId = $params['mediaId'];
		$files = $request->get_file_params();
		$file = $files['file'];
		$tmpfname = $file['tmp_name'];

		// Check errors
		if ( empty( $mediaId ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "The Media ID is required." ] );
		}
		if ( empty( $tmpfname ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "A file is required." ] );
		}
		$error = $this->check_upload( $tmpfname, $file['name'] );
		if ( !empty( $error ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $error ], 200 );
		}

		$meta = wp_get_attachment_metadata( $mediaId );
		$current_file = get_attached_file( $mediaId );
		do_action( 'wr2x_before_replace', $mediaId, $tmpfname );
		$this->core->delete_attachment( $mediaId, false );
		$pathinfo = pathinfo( $current_file );
		$basepath = $pathinfo['dirname'];

		// Let's clean everything first
		if ( wp_attachment_is_image( $mediaId ) ) {
			$sizes = $this->core->get_image_sizes();
			foreach ( $sizes as $name => $attr ) {
				if ( isset( $meta['sizes'][$name] ) && isset( $meta['sizes'][$name]['file'] ) && 
					file_exists( trailingslashit( $basepath ) . $meta['sizes'][$name]['file'] ) ) {
					$normal_file = trailingslashit( $basepath ) . $meta['sizes'][$name]['file'];
					$pathinfo = pathinfo( $normal_file );
					$retina_file = trailingslashit( $pathinfo['dirname'] ) . $pathinfo['filename'] . $this->core->retina_extension() . $pathinfo['extension'];

					// Test if the file exists and if it is actually a file (and not a dir)
					// Some old WordPress Media Library are sometimes broken and link to directories
					if ( file_exists( $normal_file ) && is_file( $normal_file ) )
						unlink( $normal_file );
					if ( file_exists( $retina_file ) && is_file( $retina_file ) )
						unlink( $retina_file );
				}
			}
		}
		if ( file_exists( $current_file ) )
			unlink( $current_file );

		// Insert the new file and delete the temporary one
		rename( $tmpfname, $current_file );
		chmod( $current_file, 0644 );

		// Generate the images
		wp_update_attachment_metadata( $mediaId, wp_generate_attachment_metadata( $mediaId, $current_file ) );
		$meta = wp_get_attachment_metadata( $mediaId );
		$this->core->generate_images( $meta );

		// Increase the version number
		$this->core->increase_media_version( $mediaId );

		// Finalize
		do_action( 'wr2x_replace', $mediaId );
		$info = $this->core->get_media_status_one( $mediaId );
		return new WP_REST_Response( [ 'success' => true, 'data' => $info  ], 200 );
	}

	function rest_upload_retina_fullsize( $request, $mediaId = null ) {
		require_once ABSPATH . 'wp-admin/includes/image.php';
		$params = $request->get_body_params();
		$mediaId = $mediaId ? $mediaId : $params['mediaId'];
		$files = $request->get_file_params();
		$file = $files['file'];
		$tmpfname = $file['tmp_name'];

		if ( empty( $mediaId ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "The Media ID is required." ] );
		}
		if ( empty( $tmpfname ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "A file is required." ] );
		}
		$error = $this->check_upload( $tmpfname, $file['name'] );
		if ( !empty( $error ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $error ], 200 );
		}

		$meta = wp_get_attachment_metadata( $mediaId );
		$current_file = get_attached_file( $mediaId );
		$pathinfo = pathinfo( $current_file );
		$retinafile = trailingslashit( $pathinfo['dirname'] ) . $pathinfo['filename'] . $this->core->retina_extension() . $pathinfo['extension'];
		do_action( 'wr2x_before_upload_retina', $mediaId, $retinafile );

		// Let's clean everything first
		if ( file_exists( $retinafile ) )
			unlink( $retinafile );

		// Insert the new file and delete the temporary one
		list( $width, $height ) = getimagesize( $tmpfname );
		if ( !$this->core->are_dimensions_ok( $width, $height, $meta['width'] * 2, $meta['height'] * 2 ) ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => "This image has a resolution of ${width}×${height} but your Full Size image requires a retina image of at least " . ( $meta['width'] * 2 ) . "x" . ( $meta['height'] * 2 ) . "." ], 200 );
		}
		$this->core->resize( $tmpfname, $meta['width'] * 2, $meta['height'] * 2, null, $retinafile );
		chmod( $retinafile, 0644 );
		unlink( $tmpfname );
		
		// Finished
		do_action( 'wr2x_upload_retina', $mediaId, $retinafile );
		$info = $this->core->get_media_status_one( $mediaId );
		return new WP_REST_Response( [ 'success' => true, 'data' => $info  ], 200 );
	}

}

?>