<?php

namespace OM4\WooCommerceZapier\SystemStatus;

use OM4\WooCommerceZapier\Auth\KeyDataStore;
use OM4\WooCommerceZapier\Installer;
use OM4\WooCommerceZapier\Plugin;
use OM4\WooCommerceZapier\Settings;
use OM4\WooCommerceZapier\TaskHistory\Task\Task;
use OM4\WooCommerceZapier\TaskHistory\Task\TaskDataStore;
use OM4\WooCommerceZapier\Webhook\DataStore as WebhookDataStore;
use OM4\WooCommerceZapier\Webhook\ZapierWebhook;
use RecursiveArrayIterator;
use RecursiveIteratorIterator;

defined( 'ABSPATH' ) || exit;

/**
 * Adds additional WC Zapier related information to the bottom of the
 * WooCommerce System Status screen and report.
 *
 * @since 2.0.0
 */
class UI {

	/**
	 * Settings instance.
	 *
	 * @var Settings
	 */
	protected $settings;

	/**
	 * TaskDataStore instance.
	 *
	 * @var TaskDataStore
	 */
	protected $task_data_store;

	/**
	 * SessionAuthenticate instance.
	 *
	 * @var KeyDataStore
	 */
	protected $key_data_store;

	/**
	 * WebhookDataStore instance.
	 *
	 * @var WebhookDataStore
	 */
	protected $webhook_data_store;

	/**
	 * Installer instance.
	 *
	 * @var Installer
	 */
	protected $installer;

	/**
	 * UI constructor.
	 *
	 * @param Settings         $settings Settings instance.
	 * @param TaskDataStore    $task_data_store TaskDataStore instance.
	 * @param KeyDataStore     $key_data_store KeyDataStore instance.
	 * @param WebhookDataStore $webhook_data_store WebhookDataStore instance.
	 * @param Installer        $installer Installer instance.
	 */
	public function __construct(
		Settings $settings,
		TaskDataStore $task_data_store,
		KeyDataStore $key_data_store,
		WebhookDataStore $webhook_data_store,
		Installer $installer
	) {
		$this->settings           = $settings;
		$this->task_data_store    = $task_data_store;
		$this->key_data_store     = $key_data_store;
		$this->webhook_data_store = $webhook_data_store;
		$this->installer          = $installer;
	}

	/**
	 * Instructs the functionality to initialise itself.
	 *
	 * @return void
	 */
	public function initialise() {
		add_action(
			'woocommerce_system_status_report',
			array( $this, 'woocommerce_system_status_report' ),
			100000
		);
	}

	/**
	 * Output the Zapier Integration System Status information.
	 * Executed during the `woocommerce_system_status_report` hook.
	 *
	 * @return void
	 */
	public function woocommerce_system_status_report() {
		$section_title = __( 'Zapier Integration', 'woocommerce-zapier' );
		/**
		 * Override the rows of data that is displayed on the status screen.
		 *
		 * @internal
		 * @since 2.0.0
		 *
		 * @param array $rows The rows of information to be displayed.
		 */
		$rows = apply_filters( 'wc_zapier_system_status_rows', $this->wc_zapier_system_status_rows() );

		include plugin_dir_path( WC_ZAPIER_PLUGIN_FILE ) . 'templates/system-status.php';
	}

	/**
	 * Retrieve our own information for display on the WooCommerce Status screen.
	 *
	 * @return array
	 */
	protected function wc_zapier_system_status_rows() {
		return array_merge(
			$this->get_status_rows_settings(),
			$this->get_status_rows_api(),
			$this->get_status_rows_task_history(),
			$this->get_status_rows_triggers(),
			$this->get_status_rows_actions()
		);
	}

	/**
	 * Get the Status Screen Rows that relate to the plugin's settings.
	 *
	 * @return array
	 */
	protected function get_status_rows_settings() {
		$rows[] = array(
			'name'    => __( 'Database Version', 'woocommerce-zapier' ),
			'success' => $this->installer->is_up_to_date(),
			'note'    => (string) $this->installer->get_db_version(),
		);
		$rows[] = array(
			'name' => __( 'Detailed Logging Enabled', 'woocommerce-zapier' ),
			'note' => $this->settings->is_detailed_logging_enabled() ? __( 'Yes', 'woocommerce-zapier' ) : __( 'No', 'woocommerce-zapier' ),
		);
		return $rows;
	}

	/**
	 * Get the Status Screen Rows that UI to the plugin's API functionality.
	 *
	 * @return array
	 */
	protected function get_status_rows_api() {
		// Check if is_ssl() returns true.
		// Required because the WooCommerce REST API will only perform basic/query authentication if is_ssl() is true.
		// Ref: https://github.com/woocommerce/woocommerce/blob/4.0.1/includes/class-wc-rest-authentication.php#L81.
		$rows[] = array(
			'name'    => __( 'SSL', 'woocommerce-zapier' ),
			'help'    => __( 'Whether or not your store passes the WordPress is_ssl() check. This is required in order for authentication to work correctly.', 'woocommerce-zapier' ),
			'success' => is_ssl(),
			// Translators: 1: Opening Link HTML Tag. 2: Closing Link HTML Tag.
			'note'    => is_ssl() ? __( 'Yes', 'woocommerce-zapier' ) : sprintf( __( 'No. Authentication will not work until SSL is configured correctly. Please %1$ssee here for details on how to fix this%2$s.', 'woocommerce-zapier' ), '<a href=" ' . Plugin::DOCUMENTATION_URL . 'troubleshooting/' . Plugin::UTM_TAGS . 'status#ssl-check-failing" target="_blank">', '</a>' ),
		);

		$count  = $this->key_data_store->count();
		$rows[] = array(
			'name'    => __( 'REST API Authentication Key(s)', 'woocommerce-zapier' ),
			'help'    => __( 'The number of Zapier-specific WooCommerce REST API Authentication Keys.', 'woocommerce-zapier' ),
			'success' => $count > 0,
			'note'    => (string) $count,
		);
		return $rows;
	}

	/**
	 * Get the Status Screen Row that relates to the plugin's Task History functionality.
	 *
	 * If there are no Task History records, then the row will display a count of 0.
	 * If there are Task History records, then the row will display a count of each status type.
	 *
	 * @return array
	 */
	protected function get_status_rows_task_history() {
		$total      = $this->task_data_store->get_tasks_count();
		$successful = $this->task_data_store->get_tasks_count( array( 'status' => Task::STATUS_SUCCESS ) );
		$errored    = $this->task_data_store->get_tasks_count( array( 'status_not' => Task::STATUS_SUCCESS ) );
		$rows[]     = array(
			'name' => __( 'Task History Record Counts', 'woocommerce-zapier' ),
			'help' => __( 'Indicates how many task history records exist for each status', 'woocommerce-zapier' ),
			'note' => 0 === $total
				? (string) $total
				: sprintf(
					// Both spaces and <br /> tags are used below so that the HTML output (and the text output) both
					// show in a readable way.
					// Translators: 1: Error record count. 2: OK (successful) record count. 3: Total record count.
					__(
						'Error: %1$d <br />OK: %2$d <br />Total: %3$d',
						'woocommerce-zapier'
					),
					$errored,
					$successful,
					$total
				),
		);
		return $rows;
	}

	/**
	 * Get the Status Screen Rows that relate to the plugin's Trigger functionality.
	 *
	 * These rows include a list of existing WC Zapier Webhooks
	 *
	 * @return array
	 */
	protected function get_status_rows_triggers() {
		$rows                    = array();
		$webhooks                = $this->webhook_data_store->get_zapier_webhooks();
		$webhook_delivery_counts = $this->task_data_store->get_trigger_task_count();
		foreach ( $webhooks as $webhook ) {
			$count = 0;
			foreach ( $webhook_delivery_counts as $i => $webhook_count ) {
				if ( $webhook->get_id() === $webhook_count['webhook_id'] ) {
					$count = $webhook_count['count'];
					unset( $webhook_delivery_counts[ $i ] );
					break;
				}
			}

			$rows[] = array(
				// Translators: 1: Webhook ID.
				'name' => sprintf( __( 'Webhook #%1$s', 'woocommerce-zapier' ), $webhook->get_id() ),
				'note' => sprintf(
					// Both line breaks and <br /> tags are used below so that the HTML output (and the text output) both show a new line.
					// Translators: 1: Webhook Name. 2: Webhook Status. 3: Webhook Topic. 4: Webhook Delivery Count. 5: Webhook User Info.
					__(
						'%1$s<br />
- Status: %2$s<br />
- Trigger: %3$s<br />
- Delivery Count: %4$s%5$s',
						'woocommerce-zapier'
					),
					$webhook->get_name(),
					$webhook->get_status(),
					$webhook->get_topic(),
					(string) $count,
					$this->get_webhook_user_info_output( $webhook )
				),
			);
		}
		if ( ! empty( $webhook_delivery_counts ) ) {
			$count    = 0;
			$iterator = new RecursiveIteratorIterator( new RecursiveArrayIterator( $webhook_delivery_counts ) );
			foreach ( $iterator as $key => $value ) {
				if ( 'count' === $key ) {
					$count += $value;
				}
			}
			$rows[] = array(
				'name' => __( 'Deleted Webhook Delivery Count', 'woocommerce-zapier' ),
				'note' => $count,
			);
		}
		return $rows;
	}

	/**
	 * Check if the webhook has a valid (administrator) user.
	 *
	 * If it does, return an empty string.
	 * If it does not, return a string with information about the user.
	 *
	 * @since 2.4.2
	 *
	 * @param  ZapierWebhook $webhook Webhook to check.
	 *
	 * @return string
	 */
	protected function get_webhook_user_info_output( ZapierWebhook $webhook ) {
		// Check if the webhook has a valid (administrator) user.

		$user = get_user_by( 'id', $webhook->get_user_id() );
		if ( ! $user ) {
			return sprintf(
				// Translators: 1: User ID.
				__( "<br />\n- User: Id %d does not exist", 'woocommerce-zapier' ),
				$webhook->get_user_id()
			);
		}
		/**
		 * Check user has the correct permissions.
		 *
		 * @see \OM4\WooCommerceZapier\Auth\SessionAuthenticate::authenticate()
		 */
		if ( $user->has_cap( 'manage_woocommerce' ) && $user->has_cap( 'promote_users' ) && $user->has_cap( 'edit_users' ) ) {
			return sprintf(
				// Translators: 1: User Login.
				__( "<br />\n- User: %s", 'woocommerce-zapier' ),
				$user->user_login
			);
		}

		return sprintf(
			// Translators: 1: User Login.
			__( "<br />\n- User: %s does not have the correct permissions", 'woocommerce-zapier' ),
			$user->user_login
		);
	}

	/**
	 * Get the Status Screen Rows that relate to the plugin's Actions functionality.
	 *
	 * @return array
	 */
	protected function get_status_rows_actions() {
		$rows          = array();
		$action_counts = $this->task_data_store->get_action_task_counts();
		// TODO: display all 5 resources, not just the ones that have we have task records for.
		// Can be implemented once we have a centralised way of getting the list of supported WooCommerce resources.
		foreach ( $action_counts as $action_count ) {
			$resource_type = ucfirst( $action_count['resource_type'] );
			$rows[]        = array(
				// Translators: 1: Resource Name.
				'name' => sprintf( __( '%1$s Action Count', 'woocommerce-zapier' ), $resource_type ),
				'note' => (string) $action_count['count'],
				// Translators: 1: Resource Name.
				'help' => sprintf( __( 'The number of times that a %1$s has been created or updated in WooCommerce from a Zapier Zap.', 'woocommerce-zapier' ), $resource_type ),
			);
		}
		return $rows;
	}
}
