<?php
/**
 * class-woocommerce-groupon.php
 *
 * Copyright (c) "kento" Karim Rahimpur www.itthinx.com
 *
 * This code is provided subject to the license granted.
 * Unauthorized use and distribution is prohibited.
 * See COPYRIGHT.txt and LICENSE.txt
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * This header and all notices must be kept intact.
 *
 * @author Karim Rahimpur
 * @package woocommerce-groupons
 * @since woocommerce-groupons 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Coupon handler.
 */
class WooCommerce_Groupon {

	/**
	 * Initialize hooks and filters.
	 */
	public static function init() {
		if ( WOO_GROUPONS_GROUPS_IS_ACTIVE ) {
			add_action( 'init', array( __CLASS__, 'wp_init' ) );
			add_filter( 'woocommerce_coupon_is_valid', array( __CLASS__, 'woocommerce_coupon_is_valid' ), 10, 2 );
			add_filter( 'woocommerce_coupon_data_tabs', array( __CLASS__, 'woocommerce_coupon_data_tabs' ) );
			add_action( 'woocommerce_process_shop_coupon_meta', array( __CLASS__, 'woocommerce_process_shop_coupon_meta' ), 10, 2 );
			add_filter( 'woocommerce_coupon_error', array( __CLASS__, 'woocommerce_coupon_error' ), 10, 3 );
			add_action( 'woocommerce_calculate_totals', array( __CLASS__, 'woocommerce_calculate_totals' ) );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'admin_enqueue_scripts' ) );
			// this one now has wp_kses applied so we can't use it anymore directly ...
			// only to accumulate the Javascript used to remove the [Remove] link from automatically applied coupons
			add_filter( 'woocommerce_cart_totals_coupon_html', array( __CLASS__, 'woocommerce_cart_totals_coupon_html' ), 10, 2 );
			// ... which is then rendered here
			add_filter( 'woocommerce_cart_totals_order_total_html', array( __CLASS__, 'woocommerce_cart_totals_order_total_html' ) );
		} else {
			add_action( 'woocommerce_coupon_options', array( __CLASS__, 'woocommerce_coupon_options_groups_missing' ) );
		}
	}

	/**
	 * Data panel actions.
	 */
	public static function wp_init() {
		add_action( 'woocommerce_coupon_data_panels', array( __CLASS__, 'woocommerce_coupon_data_panels' ) );
	}

	/**
	 * Enqueues the select script on the coupon editing screen.
	 */
	public static function admin_enqueue_scripts() {
		$screen = get_current_screen();
		if ( isset( $screen->id ) ) {
			switch( $screen->id ) {
				case 'shop_coupon' :
					require_once GROUPS_VIEWS_LIB . '/class-groups-uie.php';
					Groups_UIE::enqueue( 'select' );
					break;
			}
		}
	}

	/**
	 * Applies auto coupons for group members.
	 */
	public static function woocommerce_calculate_totals( $cart ) {
		global $wpdb, $woocommerce;

		// @since 1.22.0 don't trigger this action again until we're done, to avoid infinite loops
		remove_action( 'woocommerce_calculate_totals', array( __CLASS__, 'woocommerce_calculate_totals' ) );

		if (
			isset( $woocommerce ) &&
			isset( $woocommerce->cart ) &&
			$woocommerce->cart !== null &&
			wc_coupons_enabled()
		) {
			$options = get_option( 'woocommerce-groupons', null );
			$auto_coupons = isset( $options[WOO_GROUPONS_AUTO_COUPONS] ) ? $options[WOO_GROUPONS_AUTO_COUPONS] : WOO_GROUPONS_AUTO_COUPONS_DEFAULT;
			if ( $auto_coupons ) {
				$coupons = $wpdb->get_results(
					"SELECT DISTINCT p.ID, p.post_title, pr.meta_value priority FROM $wpdb->posts p " .
					"LEFT JOIN $wpdb->postmeta pm ON p.ID = pm.post_id ".
					"LEFT JOIN $wpdb->postmeta pr ON p.ID = pr.post_id AND pr.meta_key = '_groupon_priority' " .
					"WHERE " .
					"p.post_status = 'publish' AND pm.meta_key = '_groupon_auto_groups'"
				);
				if ( $coupons && ( count( $coupons ) > 0 ) ) {
					$discounts = new WC_Discounts( $cart );
					usort( $coupons, array( __CLASS__, 'by_priority' ) );
					// Note that this may include guests at some point ...
					$user_id = get_current_user_id();
					foreach ( $coupons as $coupon ) {
						$coupon_code = $coupon->post_title;
						$coupon = new WC_Coupon( $coupon_code );
						if ( $coupon->get_id() ) {
							// Check it can be used with cart
							$coupon_is_valid = $discounts->is_coupon_valid( $coupon ) === true;
							if ( $coupon_is_valid ) {
								if ( !$woocommerce->cart->has_discount( $coupon_code ) ) {
									$apply_coupon = false;
									$coupon_auto_groups = !empty( $coupon ) && $coupon->get_id() ? get_post_meta( $coupon->get_id(), '_groupon_auto_groups', false ) : array();
									if ( count( $coupon_auto_groups ) > 0 ) {
										foreach ( $coupon_auto_groups as $group_id ) {
											if ( Groups_User_Group::read( $user_id, $group_id ) ) {
												$apply_coupon = true;
												break;
											}
										}
									}
									// @since 1.22.0 Don't apply this coupon if it's individual use and there are other coupons already applied.
									// The other coupons would get removed if we apply it, so we have to avoid this now.
									// This change implies a deviation from the previous behavior, where an auto-apply individual-use coupon would have
									// been applied and causing other already applied coupons to be removed.
									if ( $coupon->get_individual_use() ) {
										$applied_coupons = $woocommerce->cart->get_applied_coupons();
										if ( count( $applied_coupons ) > 0 ) {
											$apply_coupon = false;
										}
									}
									if ( $apply_coupon ) {
										$woocommerce->cart->add_discount( $coupon_code );
									}
								}
							}
						}
					}
				}
			}
		}

		add_action( 'woocommerce_calculate_totals', array( __CLASS__, 'woocommerce_calculate_totals' ) );
	}

	/**
	 * Priority sort.
	 *
	 * @param object $o1
	 * @param object $o2
	 *
	 * @return int
	 */
	public static function by_priority( $o1, $o2 ) {
		$p1 = 0;
		$p2 = 0;
		if ( isset( $o1->priority ) && $o1->priority !== null ) {
			$p1 = intval( $o1->priority );
		}
		if ( isset( $o2->priority ) && $o2->priority !== null ) {
			$p2 = intval( $o2->priority );
		}
		$d = -( $p1 - $p2 );
		return $d;
	}

	/**
	 * Filter the validity of a coupon based on group membership.
	 * @param boolean $valid
	 * @param WC_Coupon $coupon
	 */
	public static function woocommerce_coupon_is_valid( $valid, $coupon ) {
		// Only act if the coupon is still considered valid at this point.
		if ( $valid ) {
			$user_id = get_current_user_id(); // Don't restrict, this applies to guests, too.
			$coupon_groups = !empty( $coupon ) && $coupon->get_id() ? get_post_meta( $coupon->get_id(), '_groupon_groups', false ) : array();
			if ( count( $coupon_groups ) > 0 ) {
				$is_member = false;
				foreach ( $coupon_groups as $group_id ) {
					if ( Groups_User_Group::read( $user_id, $group_id ) ) {
						$is_member = true;
						break;
					}
				}
				$valid = $is_member;
			}
			if ( $valid ) {
				$coupon_exclude_groups = !empty( $coupon ) && $coupon->get_id() ? get_post_meta( $coupon->get_id(), '_groupon_exclude_groups', false ) : array();
				if ( count( $coupon_exclude_groups ) > 0 ) {
					$is_member = false;
					foreach ( $coupon_exclude_groups as $group_id ) {
						if ( Groups_User_Group::read( $user_id, $group_id ) ) {
							$is_member = true;
							break;
						}
					}
					if ( $is_member ) {
						$valid = false;
					}
				}
			}
			// Allow others to plug in here:
			$valid = apply_filters( 'woocommerce_group_coupon_is_valid', $valid, $coupon );
		}
		return $valid;
	}

	/**
	 * Adds the Groups tab.
	 * @param array $tabs
	 * @return array
	 */
	public static function woocommerce_coupon_data_tabs( $tabs ) {
		$tabs['groups'] = array(
			'label'  => __( 'Groups', 'woocommerce-group-coupons' ),
			'target' => 'custom_coupon_groups',
			'class'  => 'coupon-groups'
		); 
		return $tabs;
	}

	/**
	 * Renders group options.
	 */
	public static function woocommerce_coupon_data_panels() {

		global $wpdb, $post;

		// guard against woocommerce_coupon_options action invoked during save
		if ( isset( $_POST['action'] ) ) {
			return;
		}

		wp_enqueue_style( 'woocommerce-group-coupons-admin-product' );

		$options = get_option( 'woocommerce-groupons', null );

		echo '<div id="custom_coupon_groups" class="panel woocommerce_options_panel" style="min-height: 75em;">'; // min-height is for selectize options not to be hidden

		echo '<div class="options_group">';

		//
		// restrict to groups
		//
		echo '<h3>';
		echo esc_html__( 'Groups', 'woocommerce-group-coupons' );
		echo '</h3>';
		echo '<p>';
		echo esc_html__( 'Limit the coupon to group members', 'woocommerce-group-coupons' );
		echo '</p>';

		$coupon_groups = !empty( $post ) ? get_post_meta( $post->ID, '_groupon_groups', false ) : array();

		$group_table = _groups_get_tablename( "group" );
		$groups = $wpdb->get_results( "SELECT * FROM $group_table ORDER BY name" );

		if ( count( $groups ) > 0 ) {
			// limit to groups
			$groups_panel_groups = '<div class="groups-selects">';
			$groups_panel_groups .= '<label>';
			$groups_panel_groups .= esc_html__( 'Limit to groups', 'woocommerce-group-coupons' );
			$groups_panel_groups .= ' ';
			$groups_panel_groups .= sprintf(
				'<select id="coupon-groups" class="" name="_groupon_groups[]" multiple="multiple" placeholder="%s" data-placeholder="%s">',
				esc_attr__( 'Choose groups &hellip;', 'woocommerce-group-coupons' ),
				esc_attr__( 'Choose groups &hellip;', 'woocommerce-group-coupons' )
			);
			foreach( $groups as $group ) {
				$selected = is_array( $coupon_groups ) && in_array( $group->group_id, $coupon_groups );
				$groups_panel_groups .= sprintf( '<option value="%d" %s>%s</option>', Groups_Utility::id( $group->group_id ), $selected ? ' selected="selected" ' : '', stripslashes( wp_filter_nohtml_kses( $group->name ) ) );
			}
			$groups_panel_groups .= '</select>';
			$groups_panel_groups .= '</label>';
			$groups_panel_groups .= Groups_UIE::render_select( '#coupon-groups' );
			$groups_panel_groups .= '</div>'; // .groups-selects
			echo $groups_panel_groups;
		} else {
			echo esc_html__( 'There are no groups available to select. At least one group must exist.', 'woocommerce-group-coupons' );
		}

		echo '<p>';
		echo esc_html__( 'Only members of the selected groups will be allowed to use the coupon.', 'woocommerce-group-coupons' );
		echo ' ';
		echo esc_html__( 'If no group is selected, the coupon is not restricted to any group members.', 'woocommerce-group-coupons' );
		echo '</p>';

		echo '</div>'; // .options_group

		//
		// automatically apply for groups
		//
		$auto_coupons = isset( $options[WOO_GROUPONS_AUTO_COUPONS] ) ? $options[WOO_GROUPONS_AUTO_COUPONS] : WOO_GROUPONS_AUTO_COUPONS_DEFAULT;
		if ( $auto_coupons ) {

			echo '<div class="options_group">';

			echo '<h3>';
			echo esc_html__( 'Automatic application', 'woocommerce-group-coupons' );
			echo '</h3>';
			echo '<p>';
			echo esc_html__( 'Apply the coupon to group members automatically', 'woocommerce-group-coupons' );
			echo '</p>';

			$coupon_auto_groups = !empty( $post ) ? get_post_meta( $post->ID, '_groupon_auto_groups', false ) : array();

			if ( count( $groups ) > 0 ) {
				// automatic application for members of groups
				$groups_panel_groups = '<div class="groups-selects">';
				$groups_panel_groups .= '<label>';
				$groups_panel_groups .= esc_html__( 'Apply automatically for groups', 'woocommerce-group-coupons' );
				$groups_panel_groups .= ' ';
				$groups_panel_groups .= sprintf(
					'<select id="coupon-auto-groups" class="" name="_groupon_auto_groups[]" multiple="multiple" placeholder="%s" data-placeholder="%s">',
					esc_attr__( 'Choose groups &hellip;', 'woocommerce-group-coupons' ),
					esc_attr__( 'Choose groups &hellip;', 'woocommerce-group-coupons' )
				);
				foreach( $groups as $group ) {
					$selected = is_array( $coupon_auto_groups ) && in_array( $group->group_id, $coupon_auto_groups );
					$groups_panel_groups .= sprintf( '<option value="%d" %s>%s</option>', Groups_Utility::id( $group->group_id ), $selected ? ' selected="selected" ' : '', stripslashes( wp_filter_nohtml_kses( $group->name ) ) );
				}
				$groups_panel_groups .= '</select>';
				$groups_panel_groups .= '</label>';
				$groups_panel_groups .= Groups_UIE::render_select( '#coupon-auto-groups' );
				$groups_panel_groups .= '</div>'; // .groups-selects
				echo $groups_panel_groups;
			} else {
				echo esc_html__( 'There are no groups available to select. At least one group must exist.', 'woocommerce-group-coupons' );
			}

			echo '<p>';
			echo esc_html__( 'The coupon will be applied automatically to members of any of the selected groups.', 'woocommerce-group-coupons' );
			echo '</p>';

			echo '<p>';
			echo '<strong>';
			echo esc_html__( 'Important: ', 'woocommerce-group-coupons' );
			echo '</strong>';
			echo esc_html__( 'If the coupon should only be available to members of the selected groups, you must also limit the coupon to these groups.', 'woocommerce-group-coupons' );
			echo '</p>';

			// @since 1.22.0 priority for automatically applied coupons
			$priority = !empty( $post ) ? get_post_meta( $post->ID, '_groupon_priority', true ) : null;
			if ( is_numeric( $priority ) ) {
				$priority = intval( $priority );
			} else {
				$priority = '';
			}
			woocommerce_wp_text_input(
				array(
					'id'                => '_groupon_priority',
					'label'             => __( 'Priority', 'woocommerce-group-coupons' ),
					'placeholder'       => '0',
					'description'       => __( 'Coupons that are applied automatically are applied in descending order of priority, those with higher priority first.', 'woocommerce-group-coupons' ),
					'desc_tip'          => true,
					'class'             => 'short',
					'type'              => 'number',
					'custom_attributes' => array( 'step' => 1 ),
					'value'             => $priority,
				)
			);

			echo '</div>'; // .options_group
		}

		//
		// exclude groups
		//
		echo '<div class="options_group">';

		echo '<h3>';
		echo esc_html__( 'Exclude Groups', 'woocommerce-group-coupons' );
		echo '</h3>';
		echo '<p>';
		echo esc_html__( 'Exclude group members from using this coupon', 'woocommerce-group-coupons' );
		echo '</p>';

		$coupon_exclude_groups = !empty( $post ) ? get_post_meta( $post->ID, '_groupon_exclude_groups', false ) : array();

		if ( count( $groups ) > 0 ) {
			// add to groups
			$groups_panel_groups = '<div class="groups-selects">';
			$groups_panel_groups .= '<label>';
			$groups_panel_groups .= esc_html__( 'Exclude groups', 'woocommerce-group-coupons' );
			$groups_panel_groups .= ' ';
			$groups_panel_groups .= sprintf(
				'<select id="coupon-exclude-groups" class="" name="_groupon_exclude_groups[]" multiple="multiple" placeholder="%s" data-placeholder="%s">',
				esc_attr__( 'Choose groups &hellip;', 'woocommerce-group-coupons' ),
				esc_attr__( 'Choose groups &hellip;', 'woocommerce-group-coupons' )
			);
			foreach( $groups as $group ) {
				$selected = is_array( $coupon_exclude_groups ) && in_array( $group->group_id, $coupon_exclude_groups );
				$groups_panel_groups .= sprintf( '<option value="%d" %s>%s</option>', Groups_Utility::id( $group->group_id ), $selected ? ' selected="selected" ' : '', stripslashes( wp_filter_nohtml_kses( $group->name ) ) );
			}
			$groups_panel_groups .= '</select>';
			$groups_panel_groups .= '</label>';
			$groups_panel_groups .= Groups_UIE::render_select( '#coupon-exclude-groups' );
			$groups_panel_groups .= '</div>'; // .groups-selects
			echo $groups_panel_groups;
		} else {
			echo esc_html__( 'There are no groups available to select. At least one group must exist.', 'woocommerce-group-coupons' );
		}
		echo '<p>';
		echo esc_html__( 'If any group is selected, the coupon cannot be used by the group members.', 'woocommerce-group-coupons' );
		echo '</p>';
		echo '</div>'; // .options_group

		echo WooCommerce_Groupons_Admin::extensions();

		echo '</div>'; // #custom_coupon_groups .panel .woocommerce_options_panel
	}

	/**
	 * Options reminder when Groups is missing.
	 */
	public static function woocommerce_coupon_options_groups_missing() {
		// guard against woocommerce_coupon_options action invoked during save
		if ( isset( $_POST['action'] ) ) {
			return;
		}
		echo '<div class="options_group">';
		echo '<p>';
		echo wp_kses_post( __( '<strong>Groups</strong> - limit the coupon to group members', 'woocommerce-group-coupons' ) );
		echo '</p>';
		echo '<p>';
		echo wp_kses_post( __( 'The <a href="https://www.itthinx.com/plugins/groups/">Groups</a> plugin must be installed and activated to limit coupons to group members.', 'woocommerce-group-coupons' ) );
		echo '</p>';
		echo '</div>';
	}

	/**
	 * Saves group data for the coupon.
	 * @param int $post_id coupon ID
	 * @param object $post coupon
	 */
	public static function woocommerce_process_shop_coupon_meta( $post_id, $post ) {
		global $wpdb;
		delete_post_meta( $post_id, '_groupon_groups' );
		delete_post_meta( $post_id, '_groupon_auto_groups' );
		delete_post_meta( $post_id, '_groupon_priority' );
		delete_post_meta( $post_id, '_groupon_exclude_groups' );
		$group_table = _groups_get_tablename( "group" );
		$groups = $wpdb->get_results( "SELECT group_id FROM $group_table" );
		if ( count( $groups ) > 0 ) {
			if ( !empty( $_POST['_groupon_groups'] ) && is_array( $_POST['_groupon_groups'] ) ) {
				foreach( $_POST['_groupon_groups'] as $group_id ) {
					if ( $group = Groups_Group::read( $group_id ) ) {
						add_post_meta( $post_id, '_groupon_groups', $group->group_id );
					}
				}
			}
			if ( !empty( $_POST['_groupon_auto_groups'] ) && is_array( $_POST['_groupon_auto_groups'] ) ) {
				foreach( $_POST['_groupon_auto_groups'] as $group_id ) {
					if ( $group = Groups_Group::read( $group_id ) ) {
						add_post_meta( $post_id, '_groupon_auto_groups', $group->group_id );
					}
				}
			}
			if ( !empty( $_POST['_groupon_priority'] ) && is_numeric( $_POST['_groupon_priority'] ) ) {
				add_post_meta( $post_id, '_groupon_priority', intval ( $_POST['_groupon_priority'] ), true );
			}
			if ( !empty( $_POST['_groupon_exclude_groups'] ) && is_array( $_POST['_groupon_exclude_groups'] ) ) {
				foreach( $_POST['_groupon_exclude_groups'] as $group_id ) {
					if ( $group = Groups_Group::read( $group_id ) ) {
						add_post_meta( $post_id, '_groupon_exclude_groups', $group->group_id );
					}
				}
			}
		}
	}

	/**
	 * Modifies the coupon error message if enabled.
	 * @param string $err error message
	 * @param int $err_code error code
	 * @param WC_Coupon $coupon the coupon
	 */
	public static function woocommerce_coupon_error( $err, $err_code, $coupon ) {
		global $woocommerce;
		if ( $err_code == WC_Coupon::E_WC_COUPON_INVALID_FILTERED ) {
			$options = get_option( 'woocommerce-groupons', null );
			$show_msg = isset( $options[WOO_GROUPONS_SHOW_YOU_MUST_BE_A_MEMBER] ) ? $options[WOO_GROUPONS_SHOW_YOU_MUST_BE_A_MEMBER] : WOO_GROUPONS_SHOW_YOU_MUST_BE_A_MEMBER_DEFAULT;
			if ( $show_msg ) {

				$user_id = get_current_user_id();
				$groups_user = new Groups_User( $user_id );

				$coupon_groups = !empty( $coupon ) ? get_post_meta( $coupon->get_id(), '_groupon_groups', false ) : array();
				if ( is_array( $coupon_groups ) && ( count( $coupon_groups ) > 0 ) ) {
					$group_names = array();
					foreach( $coupon_groups as $group_id ) {
						if ( $group = Groups_Group::read( $group_id ) ) {
							if ( !$groups_user->is_member( $group_id ) ) {
								$group_names[] = stripslashes( wp_filter_nohtml_kses( $group->name ) );
							}
						}
					}
					if ( !empty( $group_names ) ) {
						$group_names = implode( esc_html__( ' or ', 'woocommerce-group-coupons' ), $group_names );
						$msg = isset( $options[WOO_GROUPONS_YOU_MUST_BE_A_MEMBER] ) ? $options[WOO_GROUPONS_YOU_MUST_BE_A_MEMBER] : WOO_GROUPONS_YOU_MUST_BE_A_MEMBER_DEFAULT;
						// @since 1.21.0 apply WPML translation
						$wpml_msg = apply_filters( 'wpml_translate_single_string', $msg, 'woocommerce-group-coupons', WOO_GROUPONS_YOU_MUST_BE_A_MEMBER, null ); // null -> $locale -> current language
						if ( $wpml_msg === $msg ) {
							$msg = __( $msg, 'woocommerce-group-coupons' );
						}
						$err = sprintf( $msg, $group_names );
					}
				}
			}
		}
		return $err;
	}

	/**
	 * Removes the Remove link for automatically applied coupons.
	 * 
	 * @param string $value
	 * @param WC_Coupon $coupon
	 */
	public static function woocommerce_cart_totals_coupon_html( $value, $coupon ) {

		global $WGC_remove_coupons;

		$options = get_option( 'woocommerce-groupons', null );
		$auto_coupons = isset( $options[WOO_GROUPONS_AUTO_COUPONS] ) ? $options[WOO_GROUPONS_AUTO_COUPONS] : WOO_GROUPONS_AUTO_COUPONS_DEFAULT;
		if ( $auto_coupons ) {
			$user_id = get_current_user_id();
			$is_auto_apply = false;
			$coupon_auto_groups = !empty( $coupon ) && $coupon->get_id() ? get_post_meta( $coupon->get_id(), '_groupon_auto_groups', false ) : array();
			if ( count( $coupon_auto_groups ) > 0 ) {
				$is_member = false;
				foreach ( $coupon_auto_groups as $group_id ) {
					if ( Groups_User_Group::read( $user_id, $group_id ) ) {
						$is_auto_apply = true;
						break;
					}
				}
			}
			if ( $is_auto_apply ) {
				$WGC_remove_coupons[] =
					'<script type="text/javascript">' .
					'if (typeof jQuery !== "undefined") {' .
					sprintf( 'jQuery(\'a.woocommerce-remove-coupon[data-coupon="%s"]\').remove();', esc_attr( $coupon->get_code() ) ) .
					'}' .
					'</script>';
			}
		}
		return $value;
	}

	/**
	 * Renders the accumulated Javascript that removes the [Remove] link from automatically applied coupons.
	 * 
	 * @param string $value
	 * @return string
	 */
	public static function woocommerce_cart_totals_order_total_html( $value ) {
		global $WGC_remove_coupons;
		if ( !empty( $WGC_remove_coupons ) ) {
			$value .= implode( '', $WGC_remove_coupons );
		}
		return $value;
	}

	/**
	 * Check if the user has the role.
	 * @param string $role
	 * @param int $user_id current user by default
	 */
	public static function has_role( $role, $user_id = null ) {
		$result = false;
		if ( $user_id === null ) {
			$user = wp_get_current_user();
		} else {
			$user = get_userdata( intval( $user_id ) );
		}
		if ( $user ) {
			$result = in_array( $role, (array) $user->roles );
		}
		return $result;
	}
}
WooCommerce_Groupon::init();
