<?php
/**
 * class-woocommerce-group-coupon-roles.php
 *
 * Copyright (c) "kento" Karim Rahimpur www.itthinx.com
 *
 * This code is provided subject to the license granted.
 * Unauthorized use and distribution is prohibited.
 * See COPYRIGHT.txt and LICENSE.txt
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * This header and all notices must be kept intact.
 *
 * @author Karim Rahimpur
 * @package woocommerce-groupons
 * @since woocommerce-groupons 1.1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Coupon handler.
 */
class WooCommerce_Group_Coupon_Roles {

	/**
	 * Initialize hooks and filters.
	 */
	public static function init() {
		add_filter( 'woocommerce_coupon_is_valid', array( __CLASS__, 'woocommerce_coupon_is_valid' ), 10, 2 );
		add_filter( 'woocommerce_coupon_data_tabs', array( __CLASS__, 'woocommerce_coupon_data_tabs' ) );
		add_action( 'woocommerce_process_shop_coupon_meta', array( __CLASS__, 'woocommerce_process_shop_coupon_meta' ), 10, 2 );
		add_action( 'init', array( __CLASS__, 'wp_init' ) );
	}

	/**
	 * Registers panel actions.
	 */
	public static function wp_init() {
		add_action( 'woocommerce_coupon_data_panels', array( __CLASS__, 'woocommerce_coupon_data_panels' ) );
	}

	/**
	 * Filter the validity of a coupon based on roles.
	 * @param boolean $valid
	 * @param WC_Coupon $coupon
	 */
	public static function woocommerce_coupon_is_valid( $valid, $coupon ) {
		if ( $valid ) {
			$coupon_roles = !empty( $coupon ) && $coupon->get_id() ? get_post_meta( $coupon->get_id(), '_groupon_roles', false ) : array();
			if ( count( $coupon_roles ) > 0 ) {
				$has_role = false;
				foreach ( $coupon_roles as $role ) {
					global $wp_roles;
					if ( $wp_roles->is_role( $role ) && self::has_role( $role ) ) {
						$has_role = true;
						break;
					}
				}
				$valid = $has_role;
			}
			if ( $valid ) {
				$coupon_exclude_roles = !empty( $coupon ) && $coupon->get_id() ? get_post_meta( $coupon->get_id(), '_groupon_exclude_roles', false ) : array();
				if ( count( $coupon_exclude_roles ) > 0 ) {
					$has_role = false;
					foreach ( $coupon_exclude_roles as $role ) {
						global $wp_roles;
						if ( $wp_roles->is_role( $role ) && self::has_role( $role ) ) {
							$has_role = true;
							break;
						}
					}
					$valid = !$has_role;
				}
			}
			// Allow others to plug in here:
			$valid = apply_filters( 'woocommerce_group_coupon_is_valid', $valid, $coupon );
		}
		return $valid;
	}

	/**
	 * Adds the Groups tab.
	 * @param array $tabs
	 * @return array
	 */
	public static function woocommerce_coupon_data_tabs( $tabs ) {
		$tabs['roles'] = array(
			'label'  => __( 'Roles', 'woocommerce-group-coupons' ),
			'target' => 'custom_coupon_roles',
			'class'  => 'coupon-roles'
		); 
		return $tabs;
	}

	/**
	 * Renders role options.
	 */
	public static function woocommerce_coupon_data_panels() {

		global $post;

		// guard against woocommerce_coupon_options action invoked during save
		if ( isset( $_POST['action'] ) ) {
			return;
		}

		wp_enqueue_style( 'woocommerce-group-coupons-admin-product' );

		echo '<div id="custom_coupon_roles" class="panel woocommerce_options_panel">';

		echo '<div class="options_group">';

		echo '<h3>';
		echo esc_html__( 'Roles', 'woocommerce-group-coupons' );
		echo '</h3>';
		echo '<p>';
		echo esc_html__( 'Limit the coupon to roles', 'woocommerce-group-coupons' );
		echo '</p>';

		$coupon_roles = !empty( $post ) ? get_post_meta( $post->ID, '_groupon_roles', false ) : array();
		global $wp_roles;
		$roles = $wp_roles->get_names();
		if ( count( $roles ) > 0 ) {
			echo '<p class="form-field">';
			echo '<label for="_groupon_roles">';
			esc_html_e( 'Roles', 'woocommerce-group-coupons' );
			echo '</label>';
			echo '<select id="_groupon_roles" name="_groupon_roles[]" style="width: 50%;"  class="wc-enhanced-select" multiple="multiple" data-placeholder="' . esc_attr__( 'No roles', 'woocommerce-group-coupons' ) . '">';
			foreach( $roles as $role => $name ) {
				printf( '<option value="%s" %s>%s</option>', esc_attr( $role ), in_array( $role, $coupon_roles ) ? ' selected="selected" ' : '', stripslashes( wp_filter_nohtml_kses( $name ) ) );
			}
			echo '</select>';
			echo '</p>';
		} else {
			echo '<p>';
			esc_html__( 'There are no roles available.', 'woocommerce-group-coupons' );
			echo '</p>';
		}

		echo '<p>';
		echo esc_html__( 'Only users who have one of the selected roles will be allowed to use the coupon.', 'woocommerce-group-coupons' );
		echo ' ';
		echo esc_html__( 'If no role is selected, the coupon is not restricted to any roles.', 'woocommerce-group-coupons' );
		echo '</p>';
		echo '</div>'; // .options_group

		echo '<div class="options_group">';

		echo '<h3>';
		echo esc_html__( 'Exclude Roles', 'woocommerce-group-coupons' );
		echo '</h3>';
		echo '<p>';
		echo esc_html__( 'Exclude roles from using this coupon', 'woocommerce-group-coupons' );
		echo '</p>';

		$coupon_exclude_roles = !empty( $post ) ? get_post_meta( $post->ID, '_groupon_exclude_roles', false ) : array();
		global $wp_roles;
		$roles = $wp_roles->get_names();
		if ( count( $roles ) > 0 ) {
			echo '<p class="form-field">';
			echo '<label for="_groupon_exclude_roles">';
			esc_html_e( 'Exclude roles', 'woocommerce-group-coupons' );
			echo '</label>';
			echo '<select id="_groupon_exclude_roles" name="_groupon_exclude_roles[]" style="width: 50%;"  class="wc-enhanced-select" multiple="multiple" data-placeholder="' . esc_attr__( 'No roles', 'woocommerce-group-coupons' ) . '">';
			foreach( $roles as $role => $name ) {
				printf( '<option value="%s" %s>%s</option>', esc_attr( $role ), in_array( $role, $coupon_exclude_roles ) ? ' selected="selected" ' : '', stripslashes( wp_filter_nohtml_kses( $name ) ) );
			}
			echo '</select>';
			echo '</p>';
		} else {
			echo '<p>';
			esc_html__( 'There are no roles available.', 'woocommerce-group-coupons' );
			echo '</p>';
		}

		echo '<p>';
		echo esc_html__( 'If any role is selected, the coupon cannot be used by users with those roles.', 'woocommerce-group-coupons' );
		echo '</p>';
		echo '</div>'; // .options_group

		echo WooCommerce_Groupons_Admin::extensions();

		echo '</div>'; // #custom_coupon_roles
	}

	/**
	 * Saves role data for the coupon.
	 * @param int $post_id coupon ID
	 * @param object $post coupon
	 */
	public static function woocommerce_process_shop_coupon_meta( $post_id, $post ) {
		global $wp_roles;
		delete_post_meta( $post_id, '_groupon_roles' );
		delete_post_meta( $post_id, '_groupon_exclude_roles' );
		$roles = $wp_roles->get_names();
		if ( isset( $_POST['_groupon_roles'] ) ) {
			foreach( $roles as $role => $name ) {
				if ( in_array( $role, $_POST['_groupon_roles'] ) ) {
					add_post_meta( $post_id, '_groupon_roles', $role );
				}
			}
		}
		if ( isset( $_POST['_groupon_exclude_roles'] ) ) {
			foreach( $roles as $role => $name ) {
				if ( in_array( $role, $_POST['_groupon_exclude_roles'] ) ) {
					add_post_meta( $post_id, '_groupon_exclude_roles', $role );
				}
			}
		}
	}

	/**
	 * Check if the user has the role.
	 * @param string $role
	 * @param int $user_id current user by default
	 */
	public static function has_role( $role, $user_id = null ) {
		$result = false;
		if ( $user_id === null ) {
			$user = wp_get_current_user();
		} else {
			$user = get_userdata( intval( $user_id ) );
		}
		if ( $user ) {
			$result = in_array( $role, (array) $user->roles );
		}
		return $result;
	}
}
WooCommerce_Group_Coupon_Roles::init();
