<?php
/**
 * class-woocommerce-groupons-admin.php
 *
 * Copyright (c) "kento" Karim Rahimpur www.itthinx.com
 *
 * This code is provided subject to the license granted.
 * Unauthorized use and distribution is prohibited.
 * See COPYRIGHT.txt and LICENSE.txt
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * This header and all notices must be kept intact.
 *
 * @author Karim Rahimpur
 * @package woocommerce-groupons
 * @since woocommerce-groupons 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WooCommerce Groupons settings.
 */
class WooCommerce_Groupons_Admin {

	const NONCE              = 'woocommerce-groupons-admin-nonce';
	const DISMISS_EXTENSIONS = 'woocommerce-group-coupons-dismiss-extensions';
	const DISMISS_META_KEY   = 'woocommerce-group-coupons-hide-extensions';

	/**
	 * Admin setup.
	 */
	public static function init() {
		add_action( 'admin_menu', array( __CLASS__, 'admin_menu' ), 40 );
		add_action( 'admin_init', array( __CLASS__, 'admin_init' ) );
		add_filter( 'plugin_row_meta', array( __CLASS__, 'plugin_row_meta' ), 10, 4 );
		add_filter( 'plugin_action_links_' . plugin_basename( WOO_GROUPONS_FILE ), array( __CLASS__, 'plugin_action_links' ) );
	}

	/**
	 * Adds the admin section.
	 */
	public static function admin_menu() {
		$admin_page = add_submenu_page(
			'woocommerce',
			__( 'Group Coupons' ),
			__( 'Group Coupons' ),
			'manage_woocommerce',
			'woocommerce_group_coupons',
			array( __CLASS__, 'woocommerce_groupons' )
		);
		// add_action( 'admin_print_scripts-' . $admin_page, array( __CLASS__, 'admin_print_scripts' ) );
		// add_action( 'admin_print_styles-' . $admin_page, array( __CLASS__, 'admin_print_styles' ) );
	}

	/**
	 * Admin request handling.
	 */
	public static function admin_init() {
		if ( !empty( $_GET[self::DISMISS_EXTENSIONS] ) && wp_verify_nonce( $_GET[self::NONCE], 'hide' ) ) {
			$user_id = get_current_user_id();
			add_user_meta( $user_id, self::DISMISS_META_KEY, 'yes', true );
		}
	}

	/**
	 * Adds links to documentation and support to the plugin's row meta.
	 *
	 * @param array $plugin_meta plugin row meta entries
	 * @param string $plugin_file path to the plugin file - relative to the plugins directory
	 * @param array $plugin_data plugin data entries
	 * @param string $status current status of the plugin
	 *
	 * @return array[string]
	 */
	public static function plugin_row_meta( $plugin_meta, $plugin_file, $plugin_data, $status ) {
		if ( $plugin_file == plugin_basename( WOO_GROUPONS_FILE ) ) {
			$plugin_meta[] = '<a href="https://woocommerce.com/products/group-coupons/">' . esc_html__( 'Extension', 'woocommerce-group-coupons' ) . '</a>';
			$plugin_meta[] = '<a style="font-weight:bold" href="https://woocommerce.com/document/group-coupons/">' . esc_html__( 'Documentation', 'woocommerce-group-coupons' ) . '</a>';
			$plugin_meta[] = '<a style="font-weight:bold" href="https://woocommerce.com/my-account/create-a-ticket/">' . esc_html__( 'Support', 'woocommerce-group-coupons' ) . '</a>';
		}
		return $plugin_meta;
	}

	/**
	 * Add plugin links.
	 *
	 * @param string[] $links
	 *
	 * @return string[]
	 */
	public static function plugin_action_links( $links ) {
		if ( current_user_can( 'manage_woocommerce' ) ) {
			$links[] = sprintf(
				'<a href="%s">%s</a>',
				esc_url( admin_url( 'admin.php?page=woocommerce_group_coupons' ) ),
				esc_html__( 'Settings', 'woocommerce-group-coupons' )
			);
		}
		$links[] = sprintf(
			'<a href="https://woocommerce.com/document/group-coupons/">%s</a>',
			esc_html__( 'Documentation', 'woocommerce-group-coupons' )
		);
		return $links;
	}

	/**
	 * Renders the admin section.
	 */
	public static function woocommerce_groupons() {

		if ( !current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'Access denied.', 'woocommerce-group-coupons' ) );
		}

		wp_enqueue_style( 'woocommerce-group-coupons-admin' );

		$options = get_option( 'woocommerce-groupons', null );
		if ( $options === null ) {
			if ( add_option( 'woocommerce-groupons', array(), null, 'no' ) ) {
				$options = get_option( 'woocommerce-groupons' );
			}
		}

		if ( isset( $_POST['submit'] ) ) {
			if ( wp_verify_nonce( $_POST[self::NONCE], 'set' ) ) {
				if ( WOO_GROUPONS_GROUPS_IS_ACTIVE ) {
					$options[WOO_GROUPONS_SHOW_YOU_MUST_BE_A_MEMBER] = isset( $_POST[WOO_GROUPONS_SHOW_YOU_MUST_BE_A_MEMBER] );
					$options[WOO_GROUPONS_YOU_MUST_BE_A_MEMBER] = !empty( $_POST[WOO_GROUPONS_YOU_MUST_BE_A_MEMBER] ) ? wp_filter_kses( $_POST[WOO_GROUPONS_YOU_MUST_BE_A_MEMBER] ) : '';
					$options[WOO_GROUPONS_AUTO_COUPONS] = isset( $_POST[WOO_GROUPONS_AUTO_COUPONS] );
				}
				$options[WOO_GROUPONS_ENABLE_ROLES] = isset( $_POST[WOO_GROUPONS_ENABLE_ROLES] );
				$options[WOO_GROUPONS_LOGOUT_REMOVE_COUPONS] = isset( $_POST[WOO_GROUPONS_LOGOUT_REMOVE_COUPONS] );
				update_option( 'woocommerce-groupons', $options );
			}
		}

		$show_you_must_be_a_member = isset( $options[WOO_GROUPONS_SHOW_YOU_MUST_BE_A_MEMBER] ) ? $options[WOO_GROUPONS_SHOW_YOU_MUST_BE_A_MEMBER] : WOO_GROUPONS_SHOW_YOU_MUST_BE_A_MEMBER_DEFAULT;
		$you_must_be_a_member = isset( $options[WOO_GROUPONS_YOU_MUST_BE_A_MEMBER] ) ? $options[WOO_GROUPONS_YOU_MUST_BE_A_MEMBER] : WOO_GROUPONS_YOU_MUST_BE_A_MEMBER_DEFAULT;
		$auto_coupons = isset( $options[WOO_GROUPONS_AUTO_COUPONS] ) ? $options[WOO_GROUPONS_AUTO_COUPONS] : WOO_GROUPONS_AUTO_COUPONS_DEFAULT;
		$enable_roles = isset( $options[WOO_GROUPONS_ENABLE_ROLES] ) ? $options[WOO_GROUPONS_ENABLE_ROLES] : WOO_GROUPONS_ENABLE_ROLES_DEFAULT;
		$logout_remove_coupons = isset( $options[WOO_GROUPONS_LOGOUT_REMOVE_COUPONS] ) ? $options[WOO_GROUPONS_LOGOUT_REMOVE_COUPONS] : WOO_GROUPONS_LOGOUT_REMOVE_COUPONS_DEFAULT;

		echo '<div class="woocommerce-groupons">';

		echo '<h2>' . esc_html__( 'Group Coupons', 'woocommerce-group-coupons' ) . '</h2>';

		echo '<form action="" name="options" method="post">';
		echo '<div>';

		if ( WOO_GROUPONS_GROUPS_IS_ACTIVE ) {

			echo '<h3>';
			echo esc_html__( 'Groups', 'woocommerce-group-coupons' );
			echo '</h3>';

			echo '<h4>' . esc_html__( 'Automatic group coupons', 'woocommerce-group-coupons' ) . '</h4>';

			echo '<p>';
			echo '<label>';
			printf( '<input name="%s" type="checkbox" %s />', WOO_GROUPONS_AUTO_COUPONS, $auto_coupons ? ' checked="checked" ' : '' );
			echo ' ';
			esc_html_e( 'Enable automatic coupons', 'woocommerce-group-coupons' );
			echo '</label>';
			echo '</p>';
			echo '<p class="description">';
			esc_html_e( 'Automatic group coupons are coupons that are applied to group members automatically.', 'woocommerce-group-coupons' );
			echo '</p>';

			echo '<h4>' . esc_html__( 'Coupon error message', 'woocommerce-group-coupons' ) . '</h4>';

			echo '<p>';
			echo '<label>';
			printf( '<input name="%s" type="checkbox" %s />', WOO_GROUPONS_SHOW_YOU_MUST_BE_A_MEMBER, $show_you_must_be_a_member ? ' checked="checked" ' : '' );
			echo ' ';
			esc_html_e( 'Show the customized message', 'woocommerce-group-coupons' );
			echo '</label>';
			echo '</p>';
			echo '<p class="description">';
			esc_html_e( 'Show a customized error message when applying a coupon fails due to the user not being a member of a required group.', 'woocommerce-group-coupons' );
			echo '</p>';

			echo '<p>';
			echo '<label>';
			printf( '<input style="%s" name="%s" type="text" value="%s" />', 'width:62%;', WOO_GROUPONS_YOU_MUST_BE_A_MEMBER, esc_attr( $you_must_be_a_member ) );
			echo '</p>';
			echo '<p class="description">';
			echo wp_kses_post( __( 'Use the <code>%s</code> placeholder where the names of the groups should be shown.', 'woocommerce-group-coupons' ) );
			echo ' ';
			echo wp_kses_post( sprintf( __( 'The default error message is <blockquote>%s</blockquote>', 'woocommerce-group-coupons' ), htmlentities( WOO_GROUPONS_YOU_MUST_BE_A_MEMBER_DEFAULT ) ) );
			echo '</label>';
			echo '</p>';
		} else {
			echo '<div class="options_group">';
			echo '<h3>';
			echo esc_html__( 'Groups', 'woocommerce-group-coupons' );
			echo '</h3>';
			echo '<p class="description">';
			echo wp_kses_post( __( 'The <a href="https://www.itthinx.com/plugins/groups/">Groups</a> plugin must be installed and activated to limit coupons to group members.', 'woocommerce-group-coupons' ) );
			echo '</p>';
			echo '<p class="description">';
			echo wp_kses_post( __( 'Related settings will be shown if <em>Groups</em> is activated.', 'woocommerce-group-coupons' ) );
			echo '</p>';
			echo '</div>';
		}

		echo '<h3>' . esc_html__( 'Roles', 'woocommerce-group-coupons' ) . '</h3>';

		echo '<p>';
		echo '<label>';
		printf( '<input name="%s" type="checkbox" %s />', WOO_GROUPONS_ENABLE_ROLES, $enable_roles ? ' checked="checked" ' : '' );
		echo ' ';
		esc_html_e( 'Enable coupon restrictions based on roles.', 'woocommerce-group-coupons' );
		echo '</label>';
		echo '</p>';
		echo '<p class="description">';
		esc_html_e( 'Coupons can be limited by user role when this option is enabled.', 'woocommerce-group-coupons' );
		echo '</p>';

		echo '<h3>' . esc_html__( 'Logout', 'woocommerce-group-coupons' ) . '</h3>';

		echo '<p>';
		echo '<label>';
		printf( '<input name="%s" type="checkbox" %s />', WOO_GROUPONS_LOGOUT_REMOVE_COUPONS, $logout_remove_coupons ? ' checked="checked" ' : '' );
		echo ' ';
		esc_html_e( 'Remove coupons after logout.', 'woocommerce-group-coupons' );
		echo '</label>';
		echo '</p>';
		echo '<p class="description">';
		esc_html_e( 'When this option is enabled, all coupons are removed from the cart after a user logs out.', 'woocommerce-group-coupons' );
		echo '</p>';

		echo '<p>';
		echo wp_nonce_field( 'set', self::NONCE, true, false );
		echo '<input class="button button-primary" type="submit" name="submit" value="' . esc_attr__( 'Save', 'woocommerce-group-coupons' ) . '"/>';
		echo '</p>';
		echo '</div>';

		echo '</form>';

		echo '</div>'; // .woocommerce-groupons

		echo self::extensions();
	}

	/**
	 * Returns extensions output.
	 *
	 * @return string
	 */
	public static function extensions() {

		global $woocommerce_group_coupons_extensions;

		$output = '';

		$user_id = get_current_user_id();
		$dismiss = get_user_meta( $user_id, self::DISMISS_META_KEY, true );

		if ( $dismiss !== 'yes' ) {

			$woocommerce_group_coupons_extensions = true;

			wp_enqueue_style( 'woocommerce-group-coupons-admin' );

			$extensions = array(
				array(
					'title'       => 'Groups WooCommerce',
					'description' => esc_html__( 'Sell Memberships with Groups and WooCommerce – the best Group Membership and Access Control solution for WordPress and WooCommerce.', 'woocommerce-group-coupons' ),
					'url'         => 'https://woocommerce.com/products/groups-woocommerce/',
					'featured'    => true
				),
				array(
					'title'       => 'WooCommerce Product Search',
					'description' => esc_html__( 'The essential extension for every WooCommerce store! The perfect Search Engine for your store helps your customers to find and buy the right products quickly.', 'woocommerce-group-coupons' ),
					'url'         => 'https://woocommerce.com/products/woocommerce-product-search/',
					'featured'    => true
				),
				array(
					'title'       => 'Sales Analysis',
					'description' => esc_html__( 'Sales Analysis oriented at Marketing & Management. Get in-depth views on fundamental Business Intelligence, focused on Sales and net Revenue Trends, International Sales Reports, Product Market and Customer Trends.', 'woocommerce-group-coupons' ),
					'url'         => 'https://woocommerce.com/products/sales-analysis-for-woocommerce/',
					'featured'    => true
				),
				array(
					'title'       => 'Volume Discount Coupons',
					'description' => esc_html__( 'Increase your sales by giving customers coupons and automatic discounts based on the quantities purchased.', 'woocommerce-group-coupons' ),
					'url'         => 'https://woocommerce.com/products/volume-discount-coupons/',
					'featured'    => true
				)
			);

			$output .= '<div class="woocommerce-group-coupons-extensions-container">';

			// dismiss
			$current_url = ( is_ssl() ? 'https://' : 'http://' ) . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
			$hide_url = wp_nonce_url(
				add_query_arg( self::DISMISS_EXTENSIONS, true, $current_url ),
				'hide',
				self::NONCE
			);
			$output .= sprintf(
				'<a class="woocommerce-message-close notice-dismiss" href="%s">%s</a>',
				esc_url( $hide_url ),
				esc_html__( 'Dismiss', 'woocommerce-coupon-shortcodes' )
			);

			// content
			$output .= '<h2>' . esc_html__( 'Related Extensions', 'woocommerce-group-coupons' ) . '</h2>';

			$output .= '<div class="options_group">';

			$output .= '<p>';
			$output .= esc_html__( 'These related extensions help to improve your store &hellip;', 'woocommerce-group-coupons' );
			$output .= '</p>';

			$output .= '<div class="woocommerce-group-coupons-extensions">';

			foreach ( $extensions as $ext ) {
				$output .= sprintf(
					'<div class="woocommerce-group-coupons-extension-container %s">',
					isset( $ext['featured'] ) && $ext['featured'] ? 'featured' : ''
				);
				$output .= sprintf(
					'<a target="_blank" href="%s"><div class="extension-title">%s</div><div class="extension-description">%s</div></a>',
					$ext['url'],
					$ext['title'],
					$ext['description']
				);
				$output .= '</div>';
			}

			$output .= '</div>'; // .woocommerce-group-coupons-extensions
			$output .= '</div>'; // .options_group
			$output .= '</div>'; // .woocommerce-group-coupons-extensions-container
		}
		return $output;
	}
}
WooCommerce_Groupons_Admin::init();
